﻿using CsvHelper;
using OfficeOpenXml;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.IO.Compression;
using System.Linq;

namespace QueryPlatform.Api.Infrastructure.Domain
{
    public class SearchPathInfo
    {
        private string relativePath; // 相对路径
        public string resultPath; // 结果文件目录
        public string filecode; // 标识
        public SearchPathInfo(string rootPath, string directory)
        {
            filecode = Guid.NewGuid().ToString("N");
            relativePath = $"/export/{directory}";
            // 压缩路径 
            resultPath = Path.Combine(rootPath, "wwwroot", "export", directory);
            TempPath = Path.Combine(rootPath, "wwwroot", "export", directory, filecode);

            Directory.CreateDirectory(resultPath);
            Directory.CreateDirectory(TempPath);
        }
        /// <summary>
        /// 临时文件目录
        /// </summary>
        public string TempPath { get; private set; }
        /// <summary>
        /// 网络路径
        /// </summary>
        public string NetworkPath { get; private set; }

        public void ClearTemp()
        {
            Directory.Delete(TempPath, true);
        }

        public void CreateFromDirectory(string fileName)
        {
            NetworkPath = $"{relativePath}/{fileName}_{filecode}.zip";
            var filePath = Path.Combine(resultPath, $"{fileName}_{filecode}.zip");
            ZipFile.CreateFromDirectory(TempPath, filePath);
        }

    }
    public abstract class StorageFormat
    {
        public void Export(string tempPath, string name, IEnumerable<dynamic> dynamics)
        {
            Directory.CreateDirectory(tempPath);
            ExportFile(tempPath, name, dynamics);
        }

        protected abstract void ExportFile(string tempPath, string name, IEnumerable<dynamic> dynamics);
    }

    public class StorageFormatCsv : StorageFormat
    {
        protected override void ExportFile(string tempPath, string name, IEnumerable<dynamic> dynamics)
        {
            string path = Path.Combine(tempPath, $"{name}.csv");
            using (FileStream fs = new FileStream(path, FileMode.OpenOrCreate))
            using (var writer = new StreamWriter(fs))
            using (var csv = new CsvWriter(writer, CultureInfo.InvariantCulture))
            {
                csv.WriteRecords(dynamics);
                csv.Flush();
            }
        }
    }
    public class StorageFormatExcel : StorageFormat
    {
        protected override void ExportFile(string tempPath, string name, IEnumerable<dynamic> dynamics)
        {
            string path = Path.Combine(tempPath, $"{name}.xlsx");
            using (FileStream fs = new FileStream(path, FileMode.OpenOrCreate))
            using (ExcelPackage package = new ExcelPackage(fs))
            {
                if (dynamics == null || dynamics.Count() == 0)
                    return;

                var worksheet = package.Workbook.Worksheets.Add(name);

                var headers = ((IDictionary<string, object>)dynamics.ElementAt(0)).Keys;
                for (int col = 0; col < headers.Count; col++)
                {
                    worksheet.SetValue(1, col + 1, headers.ElementAt(col));
                }
                for (int col = 0; col < headers.Count; col++)
                {
                    for (int row = 0; row < dynamics.Count(); row++)
                    {
                        var key = headers.ElementAt(col);
                        var data = dynamics.ElementAt(row);

                        var temp = (IDictionary<string, object>)data;

                        var value = temp[key]?.ToString();
                        worksheet.SetValue(row + 2, col + 1, value);
                    }
                }

                package.Save();
            }
        }
    }
}
