﻿using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Filters;
using Microsoft.Extensions.Hosting;
using Microsoft.Extensions.Logging;
using System.Linq;
using System.Net;

namespace QueryPlatform.Api.Infrastructure.Filters
{
    public class HttpGlobalExceptionFilter : IExceptionFilter
    {
        private readonly IWebHostEnvironment env;
        private readonly ILogger<HttpGlobalExceptionFilter> logger;

        public HttpGlobalExceptionFilter(IWebHostEnvironment env, ILogger<HttpGlobalExceptionFilter> logger)
        {
            this.env = env;
            this.logger = logger;
        }

        public void OnException(ExceptionContext context)
        {
            logger.LogError(new EventId(context.Exception.HResult), context.Exception, context.Exception.Message);

            if (context.Exception.GetType() == typeof(TokenErrorException))
            {
                context.Result = new ObjectResult(new UnauthorizedResponse());
                context.HttpContext.Response.StatusCode = (int)HttpStatusCode.Unauthorized;
            }
            else if (context.Exception.GetType() == typeof(DomainException))
            {
                if (context.Exception.InnerException?.GetType() == typeof(FluentValidation.ValidationException))
                {
                    var validationException = context.Exception.InnerException as FluentValidation.ValidationException;

                    var result = new ParameterErrorResponse
                    {
                        Message = validationException.Message,
                        Instance = context.HttpContext.Request.Path,
                    };
                    var errors = validationException.Errors?.Select(err => err.ErrorMessage);
                    foreach (var item in validationException.Errors)
                    {
                        if (!result.Errors.ContainsKey(item.PropertyName))
                            result.Errors.Add(item.PropertyName, item.ErrorMessage);
                    }

                    context.Result = new BadRequestObjectResult(result);
                    context.HttpContext.Response.StatusCode = (int)HttpStatusCode.BadRequest;
                }
                else
                {
                    var result = new DomainErrorResponse
                    {
                        Status = Status.Failed,
                        Message = context.Exception.Message,
                        Instance = context.HttpContext.Request.Path,
                    };
                    context.Result = new BadRequestObjectResult(result);
                    context.HttpContext.Response.StatusCode = (int)HttpStatusCode.BadRequest;
                }
            }
            else
            {
                var result = new longernalServerErrorResponse
                {
                    Status = Status.Failed,
                    Message = "请求发生一个未知错误，请参考错误信息。",
                    Instance = context.HttpContext.Request.Path,
                };
                if (env.IsDevelopment())
                {
                    result.Developer = context.Exception;
                }
                context.Result = new BadRequestObjectResult(result);
                context.HttpContext.Response.StatusCode = (int)HttpStatusCode.InternalServerError;
            }
            context.ExceptionHandled = true;
        }
    }
}
