﻿using Microsoft.AspNetCore.Http;
using Microsoft.IdentityModel.Tokens;
using QueryPlatform.Infra;
using System;
using System.Collections.Generic;
using System.IdentityModel.Tokens.Jwt;
using System.Linq;
using System.Security.Claims;
using System.Text;

namespace QueryPlatform.Api.Infrastructure
{
    public class JwtOption
    {
        public string Secret { get; set; }
        public string Issuer { get; set; }
        public string Audience { get; set; }
        public int Expires { get; set; }
    }
    public class JwtToken
    {
        private readonly byte[] secret;
        private readonly string audience;
        private readonly string issuer;
        private readonly int expiresMinute;

        public JwtToken(JwtOption options)
        {
            secret = Encoding.UTF8.GetBytes(options.Secret);
            audience = options.Audience;
            issuer = options.Issuer;
            expiresMinute = options.Expires;
        }

        public JwtResult GenerateToken(IEnumerable<Claim> claims)
        {
            var authTime = DateTime.UtcNow;
            var expiresAt = authTime.AddMinutes(expiresMinute);

            var creds = new SigningCredentials(new SymmetricSecurityKey(secret), SecurityAlgorithms.HmacSha256);
            var token = new JwtSecurityToken(issuer, audience, claims, DateTime.Now, DateTime.Now.AddHours(2), creds);

            var tokenString = new JwtSecurityTokenHandler().WriteToken(token);
            return new JwtResult
            {
                AccessToken = tokenString,
                TokenType = "Bearer",
                AuthTime = new DateTimeOffset(authTime).ToUnixTimeSeconds(),
                ExpiresAt = new DateTimeOffset(expiresAt).ToUnixTimeSeconds(),
            };
        }
    }
    public class JwtResult
    {
        /// <summary>
        /// Access Token
        /// </summary>
        public string AccessToken { get; set; }
        /// <summary>
        /// token 类型
        /// </summary>
        public string TokenType { get; set; }
        /// <summary>
        /// 授权时间
        /// </summary>
        public long AuthTime { get; set; }
        /// <summary>
        /// 过期时间
        /// </summary>
        public long ExpiresAt { get; set; }
    }
    public class UserIdentity : IUserIdentity
    {
        private readonly HttpContext _context;

        public UserIdentity(IHttpContextAccessor contextAccessor)
        {
            _context = contextAccessor.HttpContext;
        }

        public int UserId
        {
            get
            {
                var claims = _context.User.Claims.ToArray();
                var value = claims.FirstOrDefault(w => w.Type == ClaimTypes.Sid)?.Value;
                if (!int.TryParse(value, out int userId))
                    throw new TokenErrorException();

                return userId;
            }
        }

        public IEnumerable<Claim> Claims
        {
            get
            {
                if (_context.User == null)
                    throw new TokenErrorException();

                return _context.User.Claims;
            }
        }

        public bool IsAuthenticated => _context.User.Identity.IsAuthenticated;

        public bool TryClaim(string type, out string value)
        {
            if (_context.User == null)
                throw new TokenErrorException();
            value = _context.User.Claims?.FirstOrDefault(t => t.Type == type)?.Value;
            return !string.IsNullOrEmpty(value);
        }
    }

}
