﻿using Microsoft.AspNetCore.Mvc;
using QueryPlatform.Api.Infrastructure.Modules;
using QueryPlatform.Api.Infrastructure.Queries;
using QueryPlatform.Api.Infrastructure.Request;
using QueryPlatform.Infra;
using System.Collections.Generic;
using System.Net;

namespace QueryPlatform.Api.Controllers
{
    [Route("api/user")]
    [ApiController]
    public class UserController : ControllerBase
    {
        private readonly IUserIdentity _identity;
        private readonly UserStorage _userStorage;
        private readonly UserQueries _userQueries;

        public UserController(IUserIdentity identity, UserStorage userStorage, UserQueries userQueries)
        {
            _identity = identity;
            _userStorage = userStorage;
            _userQueries = userQueries;
        }
        /// <summary>
        /// 用户信息
        /// </summary> 
        /// <returns></returns>
        [ProducesResponseType(typeof(SYS_USER), (int)HttpStatusCode.OK)]
        [HttpGet("info")]
        public ActionResult Info()
        {
            var user = _userQueries.GetUserById(_identity.UserId);
            if (user == null)
                throw new DomainException("用户信息错误");

            return Ok(user);
        }
        /// <summary>
        /// 用户修改
        /// </summary> 
        /// <returns></returns>
        [ProducesResponseType(typeof(ApiResponse), (int)HttpStatusCode.OK)]
        [HttpPost("password")]
        public IActionResult Update(UserRequest request)
        {
            //PasswordRequest
            //if (string.IsNullOrEmpty(request.OldPassword)
            //    || string.IsNullOrEmpty(request.NewPassword)
            //    || request.NewPassword.Length < 4)
            //    throw new DomainException("参数错误");

            //_userStorage.Update(_identity.UserId, request.OldPassword, request.NewPassword);
            if (string.IsNullOrEmpty(request.ID.ToString()))
                throw new DomainException("参数错误");
            var user = _userQueries.GetUserByAccount(request.Login, request.ID);
            if (user != null)
                throw new DomainException("登录名称重复");

            _userStorage.Update(request);
            return Ok();
        }
        /// <summary>
        /// 查询用户列表
        /// </summary>
        /// <returns></returns>
        [ProducesResponseType(typeof(IEnumerable<SYS_USER>), (int)HttpStatusCode.OK)]
        [HttpGet]
        public IActionResult Get(string realname)
        {
            var users = _userQueries.GetUsers(realname);
            return Ok(users);
        }
        /// <summary>
        /// 新增用户
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [ProducesResponseType(typeof(SYS_USER), (int)HttpStatusCode.OK)]
        [HttpPost]
        public IActionResult Post(UserRequest request)
        {
            var user = _userQueries.GetUserByAccount(request.Login, 0);
            if (user != null)
                throw new DomainException("登录名称重复");
            user = _userStorage.Add(request);

            return Ok(user);
        }
        /// <summary>
        /// 删除
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        [ProducesResponseType(typeof(ApiResponse), (int)HttpStatusCode.OK)]
        [HttpPost("delete/{userId}")]
        public IActionResult Delete(int userId)
        {
            //delete: true
            var user = _userStorage.Update(userId, delete: true);
            return Ok(user);
        }
        /// <summary>
        /// 禁用
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        [ProducesResponseType(typeof(ApiResponse), (int)HttpStatusCode.OK)]
        [HttpPost("disable/{userId}")]
        public IActionResult Disable(int userId, bool zt)
        {
            //delete: true
            var user = _userStorage.Update(userId, disable: zt);
            return Ok(user);
        }
        /// <summary>
        /// 重置密码
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        [ProducesResponseType(typeof(ApiResponse), (int)HttpStatusCode.OK)]
        [HttpPost("reset/{userId}")]
        public IActionResult Reset(int userId)
        {
            var user = _userStorage.Update(userId, reset: true);
            return Ok(user.Password);
        }
    }
}
