using Autofac;
using FluentValidation.AspNetCore;
using Microsoft.AspNetCore.Authentication.JwtBearer;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Diagnostics.HealthChecks;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.HttpOverrides;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.DependencyModel;
using Microsoft.Extensions.FileProviders;
using Microsoft.Extensions.Hosting;
using Microsoft.IdentityModel.Tokens;
using Microsoft.OpenApi.Models;
using Newtonsoft.Json;
using QueryPlatform.Api.Infrastructure;
using QueryPlatform.Api.Infrastructure.AutofacModules;
using QueryPlatform.Api.Infrastructure.Filters;
using QueryPlatform.Api.Infrastructure.Hubs;
using QueryPlatform.Infra;
using System;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.Loader;
using System.Text;
using System.Threading.Tasks;

namespace QueryPlatform.Api
{
    public class Startup
    {
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        public void ConfigureContainer(ContainerBuilder builder)
        {
            builder.RegisterType<UserIdentity>().As<IUserIdentity>().InstancePerLifetimeScope();
            builder.RegisterType<JsonHelper>().As<IJsonConverter>().InstancePerLifetimeScope();

            var assemblies = DependencyContext.Default.CompileLibraries
                .Where(lib => !lib.Serviceable && lib.Type == "project")
                .Select(lib => AssemblyLoadContext.Default.LoadFromAssemblyName(new AssemblyName(lib.Name)));

            builder.RegisterModule(new QueriesModule(Configuration));
        }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            services.AddHttpContextAccessor();
            services.AddHealthChecks();
            services.AddMiniProfiler(options =>
            {
                options.RouteBasePath = "/profiler";
            });

            services.AddSignalR();
            services.AddEasyCaching(options => options.UseInMemory("default"));

            services
                .AddCustomMvc(Configuration)
                .AddCustomSwagger(Configuration)
                .AddCustomCaching(Configuration)
                .AddCustomDbContext(Configuration)
                .AddCustomJwtAuthentication(Configuration);

            services.Configure<JwtOption>(Configuration.GetSection("JwtAuthorization"));
            //services.AddTransient<Microsoft.Extensions.Hosting.IHostedService, Job>();
        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IWebHostEnvironment env)
        {
            if (env.IsDevelopment())
            {
                app.UseDeveloperExceptionPage();
            }
            app.UseForwardedHeaders(new ForwardedHeadersOptions
            {
                ForwardedHeaders = ForwardedHeaders.XForwardedFor | ForwardedHeaders.XForwardedProto
            });

            app.UseSwagger();
            app.UseSwaggerUI(c =>
            {
                var swaggerEndpolong = Configuration.GetValue("SwaggerEndpolong", "");
                c.SwaggerEndpoint($"{swaggerEndpolong}/swagger/v1/swagger.json", "v1");
                c.IndexStream = () => GetType().GetTypeInfo().Assembly.GetManifestResourceStream("QueryPlatform.Api.index.html");
            });
            app.UseMiniProfiler();
            app.UseStaticFiles(new StaticFileOptions
            {
                FileProvider = new PhysicalFileProvider(Path.Combine(env.ContentRootPath, "wwwroot")),
                RequestPath = "/export"
            });
            //app.UseDecodeReqResponse();
            //app.UseDoubleRequest();

            app.UseRouting();
            //δ½
            app.UseAuthentication();
            app.UseAuthorization();

            app.UseEndpoints(endpolongs =>
            {
                endpolongs.MapHealthChecks("/QueryPlatform", new HealthCheckOptions
                {
                    ResponseWriter = QueryPlatformCheck.WriteResponse
                });
                endpolongs.MapControllers();
                endpolongs.MapHub<NotificationHub>("/api/hubs/notification");
            });
        }
    }

    static class CustomExtensionsMethods
    {
        public static IServiceCollection AddCustomCaching(this IServiceCollection services, IConfiguration configuration)
        {
            services.AddEasyCaching(options =>
            {
                //use memory cache that named default
                options.UseInMemory();
            });
            return services;
        }

        public static IServiceCollection AddCustomJwtAuthentication(this IServiceCollection services, IConfiguration configuration)
        {
            var jwtOption = new JwtOption();
            configuration.Bind("JwtAuthorization", jwtOption);

            services
                .AddAuthentication(JwtBearerDefaults.AuthenticationScheme)
                .AddJwtBearer(options =>
                {
                    options.TokenValidationParameters = new TokenValidationParameters
                    {
                        ValidateLifetime = true,
                        ClockSkew = TimeSpan.FromMinutes(5),
                        ValidateAudience = true,
                        ValidAudience = jwtOption.Audience,
                        ValidateIssuer = true,
                        ValidIssuer = jwtOption.Issuer,
                        ValidateIssuerSigningKey = true,
                        IssuerSigningKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(jwtOption.Secret)),
                    };

                    options.Events = new JwtBearerEvents()
                    {
                        OnMessageReceived = context =>
                        {
                            var accessToken = context.Request.Query["access_token"];
                            var path = context.HttpContext.Request.Path;
                            if (!string.IsNullOrEmpty(accessToken) && (path.StartsWithSegments("/api/hubs/notification")))
                                context.Token = accessToken;
                            return Task.CompletedTask;
                        },
                        OnAuthenticationFailed = context =>
                        {
                            if (context.Exception.GetType() == typeof(SecurityTokenExpiredException))
                                context.Response.Headers.Add("Token-Expired", "true");
                            return Task.CompletedTask;
                        },
                        OnChallenge = context =>
                        {
                            context.HandleResponse();
                            context.Response.StatusCode = (int)System.Net.HttpStatusCode.Unauthorized;
                            context.Response.ContentType = "application/json";
                            var payload = JsonConvert.SerializeObject(new UnauthorizedResponse());
                            context.Response.WriteAsync(payload);
                            return Task.CompletedTask;
                        }
                    };
                });
            return services;
        }
        public static IServiceCollection AddCustomMvc(this IServiceCollection services, IConfiguration configuration)
        {
            services
                .AddControllers(options =>
                {
                    options.Filters.Add(typeof(ValidateModelFilter));
                    options.Filters.Add(typeof(ArgumentsActionFilter));
                    options.Filters.Add(typeof(HttpGlobalExceptionFilter));
                    // AllowAnonymous
                    options.EnableEndpointRouting = false;
                })
                .AddNewtonsoftJson(options =>
                {
                    options.SerializerSettings.Converters.Add(new IsoDateTimeConverterContent() { DateTimeFormat = "yyyy-MM-dd HH:mm:ss" });
                    //options.SerializerSettings.ContractResolver = new UppercaseContractResolver();
                    options.SerializerSettings.Formatting = Formatting.Indented;
                    options.SerializerSettings.NullValueHandling = NullValueHandling.Include;
                    options.SerializerSettings.DateFormatHandling = DateFormatHandling.MicrosoftDateFormat;
                    options.SerializerSettings.DateTimeZoneHandling = DateTimeZoneHandling.Utc;
                    options.SerializerSettings.Culture = new System.Globalization.CultureInfo("zh-CN");
                    options.SerializerSettings.ReferenceLoopHandling = ReferenceLoopHandling.Ignore;
                })
                .AddFluentValidation(fv =>
                {
                    // ֤
                    //fv.RunDefaultMvcValidationAfterFluentValidationExecutes = false;
                    //fv.RegisterValidatorsFromAssemblyContaining<Domain.Request.LoginRequestValidator>();

                    //fv.AutomaticValidationEnabled = false;
                })
                .SetCompatibilityVersion(CompatibilityVersion.Version_3_0);

            return services;
        }
        public static IServiceCollection AddCustomSwagger(this IServiceCollection services, IConfiguration configuration)
        {
            services.AddSwaggerGen(options =>
            {
                options.ResolveConflictingActions(apiDescriptions => apiDescriptions.First());
                options.SwaggerDoc("v1", new OpenApiInfo { Version = "v1.1.0", Title = "API DOCUMENT" });
                string[] xmls = new string[] { "QueryPlatform.Api.xml" };
                foreach (var xml in xmls)
                {
                    var xmlPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "wwwroot", xml);
                    options.IncludeXmlComments(xmlPath, true);
                }
                var security = new OpenApiSecurityScheme { Reference = new OpenApiReference() { Id = "Bearer", Type = ReferenceType.SecurityScheme } };
                options.AddSecurityRequirement(new OpenApiSecurityRequirement { { security, Array.Empty<string>() } });
                options.AddSecurityDefinition("Bearer", new OpenApiSecurityScheme
                {
                    Description = "JWTȨ(ݽͷнд) ֱ¿Bearer {token} (ע֮Ŀո)",
                    Name = "Authorization",
                    In = ParameterLocation.Header,
                });
            });
            return services;
        }
        public static IServiceCollection AddCustomDbContext(this IServiceCollection services, IConfiguration configuration)
        {
            //services.AddDbContext<SearchContext>(options =>
            //{
            //    var connectionString = configuration["ConnectionStrings:DefaultConnection"];
            //    options.UseSqlServer(connectionString);
            //});

            //services.AddDbContext<SearchContext>(options =>
            //{
            //    options.UseNpgsql( )
            //        .UseLowerCaseNamingConvention();
            //});
            return services;
        }
    }
}
