﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Logging;
using Performance.DtoModels;
using Performance.DtoModels.Response;
using Performance.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace Performance.Api.Controllers
{
    [Route("api/collect")]
    [ApiController]
    public class CollectController : ControllerBase
    {
        private readonly ILogger<CollectController> logger;
        private readonly ClaimService claim;
        private readonly CollectService collectService;

        public CollectController(
            ILogger<CollectController> logger,
            ClaimService claim,
            CollectService collectService)
        {
            this.logger = logger;
            this.claim = claim;
            this.collectService = collectService;
        }

        /// <summary>
        /// 查询采集内容
        /// </summary>
        [HttpPost]
        [Route("getcollectcontent")]
        public ApiResponse GetCollectContent([FromQuery] int hospitalId, [FromQuery] int userId)
        {
            if (hospitalId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            var result = collectService.GetCollectContent(hospitalId, userId);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 设置采集录入权限
        /// </summary>
        [HttpPost]
        [Route("setpermission/{userid}")]
        public ApiResponse SetPermission(int userid, [FromBody] IEnumerable<CollectPermission> collects)
        {
            if (collects == null || !collects.Any())
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            collectService.SetPermission(userid, collects);
            return new ApiResponse(ResponseType.OK);
        }

        /// <summary>
        /// 获取采集SHEET名称
        /// </summary>
        [HttpPost]
        [Route("getcollectsheet/{hospitalId}")]
        public ApiResponse GetCollectSheet(int hospitalId)
        {
            if (hospitalId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            var regex = new Regex("^[0-9]");
            var userId = claim.GetUserId();
            var result = collectService.GetCollectSheet(hospitalId, userId)
                ?.Where(w => !collectService.retain.Contains(w.HeadName))
                ?.Select(w => new { w.SheetType, w.SheetName })
                .Distinct()
                .ToList();
            //result?.Add(new { SheetType = -1, SheetName = "预留比例" });

            if (result == null || !result.Any()) return new ApiResponse(ResponseType.OK, "", new List<string>());

            result = result.OrderBy(w => regex.IsMatch(w.SheetName) ? w.SheetName : $"0{w.SheetName}")
                 .ThenBy(w => w.SheetType)
                 .ToList();

            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 加载采集内容
        /// </summary>
        [HttpPost]
        [Route("getcollectdata/{allotId}")]
        public ApiResponse GetCollectData(int allotId, [FromQuery] string sheetName)
        {
            if (allotId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");
            var userId = claim.GetUserId();
            var result = collectService.GetCollectData(userId, allotId, sheetName);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 保存采集内容
        /// </summary>
        [HttpPost]
        [Route("savecollectdata/{allotId}")]
        public ApiResponse SaveCollectData(int allotId, [FromBody] SaveCollectData request)
        {
            if (allotId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");
            collectService.SaveCollectData(allotId, request, claim.GetUserId());
            return new ApiResponse(ResponseType.OK);
        }

        /// <summary>
        /// 提交采集信息
        /// </summary>
        [HttpPost]
        [Route("submitcollectdata/{allotId}")]
        public ApiResponse SubmitCollectData(int allotId, [FromBody] SaveCollectData request)
        {
            if (allotId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");
            collectService.SaveCollectData(allotId, request, claim.GetUserId(), 2);
            return new ApiResponse(ResponseType.OK);
        }

        /// <summary>
        /// 数据录入显示列表
        /// </summary>
        [HttpPost]
        [Route("collectdatalist/{hospitalId}")]
        public ApiResponse CollectDataList(int hospitalId)
        {
            if (hospitalId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            var data = collectService.CollectDataList(hospitalId);
            return new ApiResponse(ResponseType.OK, data);
        }

        /// <summary>
        /// 核算办显示列表
        /// </summary>
        [HttpPost]
        [Route("performanceforlist/{allotId}")]
        public ApiResponse PerformanceforList(int allotId)
        {
            if (allotId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            var data = collectService.PerformanceforList(allotId);
            return new ApiResponse(ResponseType.OK, data);
        }

        /// <summary>
        /// 科室显示列表
        /// </summary>
        [HttpPost]
        [Route("departmentlist/{allotId}")]
        public ApiResponse DepartmentList(int allotId, [FromQuery] int userId)
        {
            if (allotId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            var data = collectService.DepartmentList(allotId, userId);
            return new ApiResponse(ResponseType.OK, data);
        }

        /// <summary>
        /// 提交数据审核
        /// </summary>
        [HttpPost]
        [Route("collectdataaudit")]
        public ApiResponse CollectDataAudit([FromBody] CollectAuditRequest request)
        {
            if (request == null || request.AllotId <= 0 || request.UserId <= 0 || !new int[] { 3, 4 }.Contains(request.Status))
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            var result = collectService.CollectDataAudit(request.AllotId, request.UserId, request.Status);
            string remark = request.Status == 3 ? "审核通过" : "驳回成功";
            return result ? new ApiResponse(ResponseType.OK, remark) : new ApiResponse(ResponseType.Fail, "操作失败");
        }
    }
}
