﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using AutoMapper;
using FluentValidation.AspNetCore;
using Microsoft.AspNetCore.Mvc;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Services;

namespace Performance.Api.Controllers
{
    [Route("api/[controller]")]
    public class EmployeeController : Controller
    {
        private EmployeeService employeeService;
        private ClaimService claim;
        public EmployeeController(EmployeeService employeeService, ClaimService claim)
        {
            this.employeeService = employeeService;
            this.claim = claim;
        }

        /// <summary>
        /// 获取人员列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getlist")]
        [HttpPost]
        public ApiResponse GetEmployeeList([CustomizeValidator(RuleSet = "Select"), FromBody]EmployeeRequest request)
        {
            var employee = employeeService.GetEmployeeList(request.AllotID, claim.GetUserId());
            return new ApiResponse(ResponseType.OK, "ok", employee);
        }

        /// <summary>
        /// 新增人员
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns> 
        [Route("insert")]
        [HttpPost]
        public ApiResponse Insert([CustomizeValidator(RuleSet = "Insert"), FromBody]EmployeeRequest request)
        {
            var employee = employeeService.Insert(request);
            return new ApiResponse(ResponseType.OK, "ok", employee);
        }

        /// <summary>
        /// 修改人员
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("update")]
        [HttpPost]
        public ApiResponse Update([CustomizeValidator(RuleSet = "Update"), FromBody]EmployeeRequest request)
        {
            var employee = employeeService.Update(request);
            return new ApiResponse(ResponseType.OK, "ok", employee);
        }

        /// <summary>
        /// 删除人员
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("delete")]
        [HttpPost]
        public ApiResponse Delete([CustomizeValidator(RuleSet = "Delete"), FromBody]EmployeeRequest request)
        {
            if (!employeeService.Delete(request))
                return new ApiResponse(ResponseType.Fail);
            return new ApiResponse(ResponseType.OK);
        }

        /// <summary>
        /// 获取临床人员列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("clinic/getlist")]
        [HttpPost]
        public ApiResponse GetEmployeeClinicList([CustomizeValidator(RuleSet = "Select"), FromBody]im_employee_clinic request)
        {
            //if ((request.AllotID ?? 0) == 0)
            //    return new ApiResponse(ResponseType.ParameterError, "参数AllotId无效！");

            var employee = employeeService.GetEmployeeClinicList(request.AllotID, claim.GetUserId());
            return new ApiResponse(ResponseType.OK, "ok", employee);
        }

        /// <summary>
        /// 新增临床人员
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns> 
        [Route("clinic/insert")]
        [HttpPost]
        public ApiResponse InsertClinic([CustomizeValidator(RuleSet = "Insert"), FromBody]im_employee_clinic request)
        {
            if ((request.AllotID ?? 0) == 0)
                return new ApiResponse(ResponseType.ParameterError, "参数AllotId无效！");

            if (string.IsNullOrEmpty(request.AccountingUnit))
                return new ApiResponse(ResponseType.ParameterError, "参数AccountingUnit无效！");

            if (string.IsNullOrEmpty(request.DoctorName))
                return new ApiResponse(ResponseType.ParameterError, "参数DoctorName无效！");

            var employee = employeeService.InsertClinic(request);
            return new ApiResponse(ResponseType.OK, "ok", employee);
        }

        /// <summary>
        /// 修改临床人员
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("clinic/update")]
        [HttpPost]
        public ApiResponse UpdateClinic([CustomizeValidator(RuleSet = "Update"), FromBody]im_employee_clinic request)
        {
            if ((request.AllotID ?? 0) == 0)
                return new ApiResponse(ResponseType.ParameterError, "参数AllotId无效！");

            if (string.IsNullOrEmpty(request.AccountingUnit))
                return new ApiResponse(ResponseType.ParameterError, "参数AccountingUnit无效！");

            if (string.IsNullOrEmpty(request.DoctorName))
                return new ApiResponse(ResponseType.ParameterError, "参数DoctorName无效！");
            var employee = employeeService.UpdateClinic(request);
            return new ApiResponse(ResponseType.OK, "ok", employee);
        }

        /// <summary>
        /// 删除临床人员
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("clinic/delete")]
        [HttpPost]
        public ApiResponse DeleteClinic([CustomizeValidator(RuleSet = "Delete"), FromBody]im_employee_clinic request)
        {
            if (request.ID == 0)
                return new ApiResponse(ResponseType.ParameterError, "参数ID无效！");

            if (!employeeService.DeleteClinic(request))
                return new ApiResponse(ResponseType.Fail);
            return new ApiResponse(ResponseType.OK);
        }
    }
}