﻿using AutoMapper;
using NPOI.HSSF.UserModel;
using NPOI.SS.UserModel;
using NPOI.SS.Util;
using NPOI.XSSF.UserModel;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Repository;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace Performance.Services
{
    public class PerExcelService : IAutoInjection
    {
        private PerSheetService _perSheetService;
        private PerHeaderService _perHeaderService;
        private PerforImSheetRepository _perforImSheetRepository;
        private PerforImDataRepository _perforImDataRepository;
        private PerforImHeaderRepository _perforImHeaderRepository;
        private PerforImEmployeeRepository _perforImEmployeeRepository;
        private PerforImaccountbasicRepository _perforImaccountbasicRepository;
        private PerforImaccountdoctorRepository _perforImaccountdoctorRepository;
        private PerforImaccountnurseRepository _perforImaccountnurseRepository;
        private PerforCofdrugpropRepository _perforCofdrugpropRepository;
        public PerExcelService(PerSheetService perSheetService,
            PerHeaderService perHeaderService,
            PerforImSheetRepository perforImSheetRepository,
            PerforImDataRepository perforImDataRepository,
            PerforImHeaderRepository perforImHeaderRepository,
            PerforImEmployeeRepository perforImEmployeeRepository,
            PerforImaccountbasicRepository perforImaccountbasicRepository,
            PerforImaccountdoctorRepository perforImaccountdoctorRepository,
            PerforImaccountnurseRepository perforImaccountnurseRepository,
            PerforCofdrugpropRepository perforCofdrugpropRepository)
        {
            _perSheetService = perSheetService;
            _perHeaderService = perHeaderService;
            _perforImSheetRepository = perforImSheetRepository;
            _perforImDataRepository = perforImDataRepository;
            _perforImHeaderRepository = perforImHeaderRepository;
            _perforImEmployeeRepository = perforImEmployeeRepository;
            _perforImaccountbasicRepository = perforImaccountbasicRepository;
            _perforImaccountdoctorRepository = perforImaccountdoctorRepository;
            _perforImaccountnurseRepository = perforImaccountnurseRepository;
            _perforCofdrugpropRepository = perforCofdrugpropRepository;
        }


        public void Execute(sys_allot allot)
        {

            // 拷贝配置信息
            Copy(allot.ID);

            // 导出数据
            var excel = Import(allot.Path);

            // 保存数据
            Save(excel.PerSheet, allot.ID, 1);

            // 计算合并数据
            List<PerSheet> list = ProcessCompute(excel);

            // 保存过程数据
            Save(list, allot.ID, 2);

            // 计算最总数据
            Compute(excel);
        }

        /// <summary>
        /// 导入excel数据
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        public PerExcel Import(string path)
        {
            PerExcel excel = new PerExcel
            {
                Path = path,
                FileName = FileHelper.GetFileNameNoExtension(path),
                Version = FileHelper.GetExtension(path) == ".xlsx" ? ExcelVersion.xlsx : ExcelVersion.xls,
                PerSheet = new List<PerSheet>()
            };
            using (FileStream fs = new FileStream(path, FileMode.Open))
            {
                IWorkbook workbook = (excel.Version == ExcelVersion.xlsx)
                    ? (IWorkbook)(new XSSFWorkbook(fs))
                    : (IWorkbook)(new HSSFWorkbook(fs));
                for (int i = 0; i < workbook.NumberOfSheets; i++)
                {
                    var sheet = workbook.GetSheetAt(i);
                    if (SheetType.Unidentifiable != _perSheetService.GetSheetType(sheet.SheetName))
                    {
                        var st = _perSheetService.Sheet(sheet);
                        excel.PerSheet.Add(st);
                    }
                }
                return excel;
            }
        }


        /// <summary>
        /// 复制报表基础配置
        /// </summary>
        /// <param name="iD"></param>
        public void Copy(int iD)
        {
            //throw new NotImplementedException();
        }

        /// <summary>
        /// 计算绩效
        /// </summary>
        /// <param name="excel"></param>
        public List<PerSheet> ProcessCompute(PerExcel excel)
        {
            var confs = GetDrugConfig(excel);
            return _perSheetService.ProcessCompute(excel, confs);
        }

        private List<CofDrugProp> GetDrugConfig(PerExcel excel)
        {
            //计算药占比
            List<CofDrugProp> cofs = new List<CofDrugProp>();

            var incomeSheet = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Income && t.SheetName.Contains("门诊") && t.SheetName.Contains("就诊"));
            var datalist = incomeSheet.PerData.Select(t => (PerData)t);
            var drugData = datalist.Where(t => t.TypeName == "西药费" && t.TypeName == "中成药费").GroupBy(t => t.AccountingUnit).Select(t => new { AccountingUnit = t.Key, SumValue = t.Sum(s => s.CellValue) });
            var allData = datalist.GroupBy(t => t.AccountingUnit).Select(t => new { AccountingUnit = t.Key, SumValue = t.Sum(s => s.CellValue) });

            var cofList = _perforCofdrugpropRepository.GetEntities();

            var unitList = drugData.Select(t => t.AccountingUnit).Union(allData.Select(t => t.AccountingUnit));
            foreach (var unit in unitList)
            {
                var dsv = drugData.FirstOrDefault(t => t.AccountingUnit == unit)?.SumValue;
                var asv = allData.FirstOrDefault(t => t.AccountingUnit == unit)?.SumValue;

                var prop = asv.HasValue || asv.Value == 0 ? 0 : Math.Round(dsv.Value / asv.Value, 2);
                var fvalue = prop == 0
                    ? 0
                    : cofList.FirstOrDefault(t => prop > t.MinRange && prop <= t.MaxRange)?.Value ?? 0;
                cofs.Add(new CofDrugProp { AccoutingUnit = unit, Factor = fvalue, Prop = prop });
            }
            return cofs;
        }

        /// <summary>
        /// 保存绩效结果
        /// </summary>
        /// <param name="perSheets"></param>
        /// <param name="allotId"></param>
        /// <param name="source">数据来源 1 excel 导入 2 计算</param>
        public void Save(List<PerSheet> perSheets, int allotId, int source)
        {
            foreach (var sheet in perSheets)
            {
                var imsheet = new im_sheet { AllotID = allotId, SheetName = sheet.SheetName, Source = source, SheetType = (int)sheet.SheetType };
                _perforImSheetRepository.Add(imsheet);
                if (sheet.SheetType == SheetType.Employee)
                {
                    var dataList = sheet.PerData.Select(t => (PerDataEmployee)t);
                    foreach (var data in dataList)
                    {
                        var imdata = Mapper.Map<im_employee>(data);
                        imdata.SheetID = imsheet.ID;
                        _perforImEmployeeRepository.Add(imdata);
                    }
                }
                else if (sheet.SheetType == SheetType.DeptAccounting)
                {
                    var dataList = sheet.PerData.Select(t => (PerDataAccountBaisc)t);
                    foreach (var data in dataList)
                    {
                        var imdata = Mapper.Map<im_accountbasic>(data);
                        imdata.SheetID = imsheet.ID;
                        _perforImaccountbasicRepository.Add(imdata);
                    }
                }
                else if (sheet.SheetType == SheetType.ComputeDoctorAccount)
                {
                    var dataList = sheet.PerData.Select(t => (PerDataAccountDoctor)t);
                    foreach (var data in dataList)
                    {
                        var imdata = Mapper.Map<im_accountdoctor>(data);
                        imdata.SheetID = imsheet.ID;
                        _perforImaccountdoctorRepository.Add(imdata);
                    }
                }
                else if (sheet.SheetType == SheetType.ComputeNurseAccount)
                {
                    var dataList = sheet.PerData.Select(t => (PerDataAccountNurse)t);
                    foreach (var data in dataList)
                    {
                        var imdata = Mapper.Map<im_accountnurse>(data);
                        imdata.SheetID = imsheet.ID;
                        _perforImaccountnurseRepository.Add(imdata);
                    }
                }
                else
                {
                    foreach (var header in sheet.PerHeader)
                    {
                        var imheader = Mapper.Map<im_header>(header);
                        imheader.SheetID = imsheet.ID;
                        _perforImHeaderRepository.Add(imheader);
                        if (header.IsHasChildren)
                        {
                            foreach (var child in header.Children)
                            {
                                var imheaderChild = Mapper.Map<im_header>(child);
                                imheaderChild.SheetID = imsheet.ID;
                                imheaderChild.ParentID = imheader.ID;
                                _perforImHeaderRepository.Add(imheaderChild);
                            }
                        }
                    }
                    var dataList = sheet.PerData.Select(t => (PerData)t);
                    foreach (var data in dataList)
                    {
                        var imdata = Mapper.Map<im_data>(data);
                        imdata.SheetID = imsheet.ID;
                        _perforImDataRepository.Add(imdata);
                    }
                }
            }

            //throw new NotImplementedException();
        }

        public IWorkbook ExportCompute(List<PerSheet> sheetList)
        {
            IWorkbook workbook = new XSSFWorkbook();
            var cellstyle = workbook.CreateCellStyle();
            cellstyle.VerticalAlignment = VerticalAlignment.Center;
            cellstyle.Alignment = HorizontalAlignment.Center;

            foreach (var sheet in sheetList)
            {
                ISheet exportSheet = workbook.CreateSheet(sheet.SheetName);
                //创建列头行
                IRow row = null, childRow = null;
                foreach (var header in sheet.PerHeader.OrderBy(t => t.PointCell))
                {
                    row = CreateRow(cellstyle, exportSheet, row, header);
                    //创建二级列头
                    if (header.IsHasChildren)
                    {
                        foreach (var child in header.Children.OrderBy(t => t.PointCell))
                        {
                            childRow = CreateRow(cellstyle, exportSheet, childRow, child);
                        }
                    }
                }
                //反转列头
                var headList = _perHeaderService.GetPerHeaderReverse(sheet.PerHeader);
                var dataList = sheet.PerData.Select(t => (PerData)t);
                var maxrow = headList.Max(t => t.PointRow);
                //循环核算单元 创建数据行
                var accountingUnitList = dataList.OrderBy(t => t.RowNumber).Select(t => t.AccountingUnit).Distinct();
                for (int i = 0; i < accountingUnitList.Count(); i++)
                {
                    var accountingUnit = accountingUnitList.ElementAt(i);
                    IRow dataRow = exportSheet.CreateRow(i + maxrow + 2);
                    dataRow.CreateCell(0).SetCellValue(accountingUnit);
                    dataRow.GetCell(0).CellStyle = cellstyle;
                    foreach (var header in headList.OrderBy(t => t.PointCell))
                    {
                        var cellValue = dataList.FirstOrDefault(t => t.AccountingUnit == accountingUnit && t.TypeName == header.CellValue)?.CellValue;
                        if (cellValue.HasValue && cellValue.Value > 0)
                        {
                            dataRow.CreateCell(header.PointCell).SetCellValue(Convert.ToDouble(cellValue.Value));
                            dataRow.GetCell(header.PointCell).CellStyle = cellstyle;
                        }
                    }
                }
            }
            return workbook;
        }

        /// <summary>
        /// 创建行
        /// </summary>
        /// <param name="cellstyle"></param>
        /// <param name="exportSheet"></param>
        /// <param name="row"></param>
        /// <param name="header"></param>
        /// <returns></returns>
        private IRow CreateRow(ICellStyle cellstyle, ISheet exportSheet, IRow row, PerHeader header)
        {
            if (header.IsMerge)
            {
                var cellRange = new CellRangeAddress(header.PointRow, header.PointRow + header.MergeRow - 1, header.PointCell, header.PointCell + header.MergeCell - 1);
                exportSheet.AddMergedRegion(cellRange);
            }
            row = row ?? exportSheet.CreateRow(header.PointRow);
            row.CreateCell(header.PointCell).SetCellValue(header.CellValue);
            row.GetCell(header.PointCell).CellStyle = cellstyle;
            return row;
        }

        /// <summary>
        /// 计算最终数据
        /// </summary>
        /// <param name="excel"></param>
        public void Compute(PerExcel excel)
        {
            //throw new NotImplementedException();
        }
    }
}
