﻿using AutoMapper;
using NPOI.SS.UserModel;
using Performance.DtoModels;
using Performance.Infrastructure;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services
{
    public class PerSheetService : IAutoInjection
    {
        PerHeaderService _perHeader;
        public PerSheetService(PerHeaderService perHeader)
        {
            _perHeader = perHeader;
        }

        /// <summary>
        /// 解析excel数据
        /// </summary>
        /// <param name="sheet"></param>
        /// <returns></returns>
        public PerSheet Sheet(ISheet sheet)
        {
            PerSheet perSheet = new PerSheet();
            perSheet.SheetName = sheet.SheetName;
            perSheet.SheetType = GetSheetType(sheet.SheetName);
            perSheet.ModuleName = EnumHelper.GetDescription(perSheet.SheetType);

            if (perSheet.SheetType == SheetType.Unidentifiable)
                return null;

            var sheetRead = PerSheetDataFactory.GetDataRead(perSheet.SheetType);
            var perHeader = _perHeader.GetPerHeader(sheet, sheetRead.Point);
            var headerReverse = _perHeader.GetPerHeaderReverse(perHeader);

            perSheet.PerHeader = perHeader;
            perSheet.PerData = sheetRead.ReadData(sheet, headerReverse);
            return perSheet;
        }

        /// <summary>
        /// 判断sheet适用模板类型
        /// </summary>
        /// <param name="sheetName"></param>
        /// <returns></returns>
        public SheetType GetSheetType(string sheetName)
        {
            if (sheetName.StartsWith("医院人员名单"))
                return SheetType.Employee;
            else if (sheetName.StartsWith("1.0"))
                return SheetType.OtherIncome;
            else if (sheetName.StartsWith("1.1"))
                return SheetType.Income;
            else if (sheetName.StartsWith("2."))
                return SheetType.Expend;
            else if (sheetName.StartsWith("3."))
                return SheetType.Workload;
            else if (sheetName.StartsWith("4.1"))
                return SheetType.DeptAccounting;
            return SheetType.Unidentifiable;
        }

        internal List<PerSheet> ProcessCompute(PerExcel excel, List<CofDrugProp> confs)
        {
            List<PerSheet> perSheet = new List<PerSheet>();


            //合并科室收入、支出
            var economicCompute = new PerSheetDataComputeEconomic();
            var mergeResult = economicCompute.MergeCompute(excel);
            //一次计算
            var onceEconomic = economicCompute.OnceCompute(mergeResult);

            //二次计算 
            var twiceEconomicResult = economicCompute.TwiceCompute(onceEconomic);

            twiceEconomicResult.Sheet.SheetType = SheetType.ComputeEconomic;
            perSheet.Add(twiceEconomicResult.Sheet);

            //工作量
            var workloadCompute = new PerSheetDataComputeWorkload();
            var workload1 = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Workload && t.SheetName.Contains("医生组"));
            workload1.SheetName = "医生组工作量绩效测算表";

            //医生组 一次计算
            var onceWorkload1 = workloadCompute.OnceCompute(workload1, confs);
            //医生组 二次计算 
            var twiceWorkloadResult1 = workloadCompute.TwiceCompute(onceWorkload1);

            twiceWorkloadResult1.Sheet.SheetType = SheetType.ComputeDoctorWorkload;
            perSheet.Add(twiceWorkloadResult1.Sheet);

            var workload2 = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Workload && t.SheetName.Contains("护理组"));
            workload2.SheetName = "护理组工作量绩效测算表";
            //护理组 一次计算
            var onceWorkload2 = workloadCompute.OnceCompute(workload2);
            //护理组 二次计算 
            var twiceWorkloadResult2 = workloadCompute.TwiceCompute(onceWorkload2);

            twiceWorkloadResult2.Sheet.SheetType = SheetType.ComputeNurseWorkload;
            perSheet.Add(twiceWorkloadResult2.Sheet);

            var deptAccounting = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.DeptAccounting);
            var dataList = deptAccounting.PerData.Select(t => (PerDataAccountBaisc)t);

            PerSheet doctorSheet = new PerSheet("医生组临床科室单元核算表", "医生组临床科室单元核算表", SheetType.ComputeDoctorAccount, new List<PerHeader>(), new List<IPerData>());
            PerSheet nurseSheet = new PerSheet("护理组临床科室单元核算表", "护理组临床科室单元核算表", SheetType.ComputeNurseAccount, new List<PerHeader>(), new List<IPerData>());
            foreach (var dept in dataList)
            {
                var doctor = Mapper.Map<PerDataAccountDoctor>(dept);
                var econDoctor = twiceEconomicResult.PerData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                doctor.Income = econDoctor?.CellValue ?? 0;
                var workDoctor = twiceWorkloadResult1.PerData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                doctor.WorkloadFee = workDoctor?.CellValue ?? 0;
                doctorSheet.PerData.Add(doctor);

                var nurse = Mapper.Map<PerDataAccountNurse>(dept);
                var econNurse = twiceEconomicResult.PerData.FirstOrDefault(t => t.UnitType == "护理组" && t.AccountingUnit == dept.AccountingUnit);
                nurse.Income = econNurse?.CellValue ?? 0;
                var workNurse = twiceWorkloadResult2.PerData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                nurse.WorkloadFee = workNurse?.CellValue ?? 0;
                nurseSheet.PerData.Add(nurse);
            }
            perSheet.Add(doctorSheet);
            perSheet.Add(nurseSheet);

            return perSheet;
        }
    }
}
