﻿using System;
using System.Collections.Generic;
using System.Linq;
using AutoMapper;
using Performance.DtoModels;
using Performance.DtoModels.Request;
using Performance.EntityModels;
using Performance.Repository;

namespace Performance.Services
{
    public class MenuService : IAutoInjection
    {
        private readonly IMapper _mapper;
        private PerforMenuRepository _menuRepository;
        private PerforRolemenuRepository _rolemenuRepository;

        public MenuService(
            IMapper mapper,
            PerforMenuRepository menuRepository,
            PerforRolemenuRepository perforRolemenuRepository
            )
        {
            _mapper = mapper;
            _menuRepository = menuRepository;
            _rolemenuRepository = perforRolemenuRepository;
        }

        public List<MenuResponse> GetMenuList(int userid)
        {
            var menuList = _menuRepository.GetMenuList(userid);

            return RecursionFill(menuList, 0);
        }

        private List<MenuResponse> RecursionFill(List<sys_menu> menuList, int parentId)
        {
            if (menuList.Any(t => t.ParentID == parentId))
            {
                var list = new List<MenuResponse>();
                foreach (var item in menuList.Where(t => t.ParentID == parentId).OrderBy(t => t.Sort))
                {
                    var menu = _mapper.Map<MenuResponse>(item);
                    menu.Children = RecursionFill(menuList, item.ID);
                    list.Add(menu);
                }
                return list;
            }
            return null;
        }

        public List<MenuResponse> GetSystemMenu()
        {
            var sysMenus = _menuRepository.GetEntities();
            return RecursionFill(sysMenus, 0);
        }

        public ApiResponse AddSystemMenu(MenuRequest menuRequest)
        {
            try
            {
                if (menuRequest == null)
                    throw new PerformanceException("请填写数据");
                var data = _mapper.Map<sys_menu>(menuRequest);
                _menuRepository.Add(data);
                return new ApiResponse(ResponseType.OK);
            }
            catch (Exception)
            {
                return new ApiResponse(ResponseType.Fail, "添加失败");
            }

        }

        public ApiResponse UpdateSystemMenu(MenuRequest menuRequest)
        {
            try
            {
                List<sys_menu> sys_Menus = new List<sys_menu>();
                var datas = _menuRepository.GetEntities(w => w.ID == menuRequest.ParentID || w.ParentID == menuRequest.ParentID);
                if (datas == null)
                    throw new PerformanceException("未找到需要修改的数据，请检查后重试");
                if (datas.Where(w => w.ParentID == menuRequest.ParentID && w.States == 1).Count() <= 1)
                {
                    var temp = datas.FirstOrDefault(w => w.ID == menuRequest.ParentID);
                    temp.States = menuRequest.States;
                    sys_Menus.Add(temp);
                }
                var data = datas.FirstOrDefault(w => w.ID == menuRequest.ID);
                if (data == null)
                    throw new PerformanceException("未找到需要修改的数据");
                data.ParentID = menuRequest.ParentID;
                data.MenuName = menuRequest.MenuName;
                data.MenuUrl = menuRequest.MenuUrl;
                data.States = menuRequest.States;
                data.Sort = menuRequest.Sort;
                sys_Menus.Add(data);
                if (sys_Menus == null)
                    throw new PerformanceException("未找到需要修改的数据，请检查后重试");
                _menuRepository.UpdateRange(sys_Menus.ToArray());
                return new ApiResponse(ResponseType.OK);
            }
            catch (Exception)
            {
                return new ApiResponse(ResponseType.Fail, "修改失败");
            }

        }

        public ApiResponse DeleteSystemMenu(int menuID)
        {
            try
            {
                var allMenus = _menuRepository.GetEntities();
                var currMenu = allMenus.FirstOrDefault(w => w.ID == menuID);
                if (currMenu == null) return new ApiResponse(ResponseType.NotFound, "未找到相关信息，请刷新后重试");
                var sysMenuIds = GetRelevanceMenuId(currMenu, allMenus, orientation: false);
                sysMenuIds.Add(menuID);
                var sysRoleMenus = _rolemenuRepository.GetEntities(w => sysMenuIds.Contains(w.MenuID));
                if (sysRoleMenus != null && sysRoleMenus.Any())
                    _rolemenuRepository.RemoveRange(sysRoleMenus.ToArray());
                var sysMenus = _menuRepository.GetEntities(w => sysMenuIds.Contains(w.ID));
                if (sysMenus != null && sysMenus.Any())
                    _menuRepository.RemoveRange(sysMenus.ToArray());
                return new ApiResponse(ResponseType.OK);
            }
            catch (Exception)
            {
                return new ApiResponse(ResponseType.Fail, "删除失败");
            }
        }
        public ApiResponse<List<sys_role_menu>> GetRoleSystemMenu(int menuID)
        {
            var roles = _rolemenuRepository.GetEntities(t => t.MenuID == menuID);
            return new ApiResponse<List<sys_role_menu>>(ResponseType.OK, roles);
        }

        public ApiResponse SetRoleSystemMenu(RoleRequest roleRequest)
        {
            try
            {
                var allMenus = _menuRepository.GetEntities();
                var currMenu = allMenus.FirstOrDefault(w => w.ID == roleRequest.MenuID);
                if (currMenu == null) return new ApiResponse(ResponseType.NotFound, "未找到相关信息，请刷新后重试");

                var menuIds = GetRelevanceMenuId(currMenu, allMenus, orientation: true);
                menuIds.Add(roleRequest.MenuID);

                var sysRoleMenus = _rolemenuRepository.GetEntities(w => w.MenuID == roleRequest.MenuID);
                if (sysRoleMenus != null && sysRoleMenus.Any())
                    _rolemenuRepository.RemoveRange(sysRoleMenus.ToArray());

                var temps = new List<sys_role_menu>();
                foreach (var menuId in menuIds)
                {
                    foreach (var roleID in roleRequest.RoleID)
                    {
                        if (_rolemenuRepository.GetEntity(w => w.MenuID == menuId && w.RoleID == roleID) == null)
                            temps.Add(new sys_role_menu { MenuID = menuId, RoleID = roleID });
                    }
                }
                _rolemenuRepository.AddRange(temps.ToArray());
                return new ApiResponse(ResponseType.OK);
            }
            catch (Exception)
            {
                return new ApiResponse(ResponseType.Fail, "添加失败");
            }
        }

        /// <summary>
        /// 递归查找指定菜单的所有上层菜单或下层菜单
        /// </summary>
        /// <param name="menuIds"></param>
        /// <param name="currMenu"></param>
        /// <param name="allMenus"></param>
        /// <param name="orientation"></param>
        private List<int> GetRelevanceMenuId(sys_menu currMenu, List<sys_menu> allMenus, bool orientation)
        {

            // orientation = true 向上找 false 乡下找
            Func<int, sys_menu> getRelevanceMenu = orientation
                ? (menuIdOrParentId) => allMenus.FirstOrDefault(w => w.ID == menuIdOrParentId)
                : (menuIdOrParentId) => allMenus.FirstOrDefault(w => w.ParentID == menuIdOrParentId);
            if (currMenu.ParentID <= 0) return new List<int>();

            var menuIdOrParentId = orientation ? currMenu.ParentID : currMenu.ID;
            var tempSysMenu = getRelevanceMenu(menuIdOrParentId);
            if (tempSysMenu == null) return new List<int>();

            List<int> menuIds = new List<int>() { tempSysMenu.ID };
            var relevanceMenuIds = GetRelevanceMenuId(tempSysMenu, allMenus, orientation);
            if (relevanceMenuIds?.Any() == true) { menuIds.AddRange(relevanceMenuIds); }

            return menuIds;
        }

    }
}
