using FluentValidation.AspNetCore;
using Hangfire;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Http.Internal;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.StaticFiles;
using Microsoft.Extensions.Options;
using Performance.DtoModels;
using Performance.DtoModels.AppSettings;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Services;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace Performance.Api.Controllers
{
    [Route("api/[controller]")]
    public class TemplateController : Controller
    {
        private readonly TemplateService templateService;
        private readonly ExtractService extractService;
        private HospitalService hospitalService;
        private IHostingEnvironment evn;
        private ClaimService claim;
        private Application application;
        private WebapiUrl url;
        private readonly AllotService allotService;
        public TemplateController(TemplateService templateService,
            HospitalService hospitalService,
            ExtractService extractService,
            IHostingEnvironment evn,
            ClaimService claim,
            IOptions<Application> options,
            IOptions<WebapiUrl> url,
            AllotService allotService)
        {
            this.templateService = templateService;
            this.extractService = extractService;
            this.hospitalService = hospitalService;
            this.evn = evn;
            this.claim = claim;
            this.application = options.Value;
            this.url = url.Value;
            this.allotService = allotService;
        }

        /// <summary>
        /// 上传文件
        /// </summary>
        /// <param name="form"></param>
        /// <returns></returns>
        [Route("import")]
        [HttpPost]
        public ApiResponse Import([FromForm] IFormCollection form)
        {
            var user = claim.At(form.ToDictionary().GetValue("token", ""));
            var hospitalid = form.ToDictionary().GetValue("hospitalid", 0);
            if (hospitalid <= 0)
                return new ApiResponse(ResponseType.Fail, "参数错误", "hospitalid无效");

            var file = ((FormFileCollection)form.Files).FirstOrDefault();
            if (file == null)
                return new ApiResponse(ResponseType.Fail, "参数错误", "文件无效");

            var hospital = hospitalService.GetHopital(hospitalid);
            if (hospital == null)
                return new ApiResponse(ResponseType.Fail, "hospitalid不存在");

            var name = FileHelper.GetFileNameNoExtension(file.FileName) + DateTime.Now.ToString("yyyyMMddHHmmssfff");
            var ext = FileHelper.GetExtension(file.FileName);
            var dpath = Path.Combine(evn.ContentRootPath, "Files", $"{hospitalid}", "first");
            FileHelper.CreateDirectory(dpath);

            var path = Path.Combine(dpath, $"{name}{ext}");

            using (var stream = file.OpenReadStream())
            {
                byte[] bytes = new byte[stream.Length];
                stream.Read(bytes, 0, bytes.Length);
                if (!FileHelper.CreateFile(path, bytes))
                    return new ApiResponse(ResponseType.Fail, $"{file.FileName}上传失败");
                var template = new per_first
                {
                    HospitalId = hospitalid,
                    CreateUser = user.UserID,
                    CreateDate = DateTime.Now,
                    Path = path,
                    UploadDate = DateTime.Now,
                    Remark = "上传成功"
                };
                if (templateService.InsertFirst(template))
                {
                    templateService.SendEmail(application.Receiver.ToList(), path, $"{hospital.HosName}首次上传模板", "上传成功");
                }
            }
            return new ApiResponse(ResponseType.OK);
        }

        /// <summary>
        /// 提取绩效数据
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("extractdata")]
        [HttpPost]
        public ApiResponse ExtractData([CustomizeValidator(RuleSet = "Template"), FromBody]AllotRequest request)
        {

            var allot = allotService.GetAllot(request.ID);
            if (allot == null)
                return new ApiResponse(ResponseType.Fail, "该绩效无效");
            allot.IsExtracting = allot.IsExtracting ?? 0;
            if (allot.IsExtracting == 1)
                return new ApiResponse(ResponseType.Fail, "正在提取数据，请稍等。");

            var hospital = hospitalService.GetHopital(request.HospitalId.Value);
            if (hospital == null)
                return new ApiResponse(ResponseType.Fail, "医院无效");

            var user = claim.At(request.Token);
            string param = JsonHelper.Serialize(new
            {
                id = request.ID,
                mail = user.Mail,
                hospitalId = hospital.ID
            });

            allot.IsExtracting = 1;
            allotService.Update(allot);
            HttpHelper.HttpPostNoRequest(url.ExtractData, param, true);
            //extractService.ExtractData(request.ID, user.Mail, hospital);
            //BackgroundJob.Enqueue(() => extractService.ExtractData(request.ID, user.Mail, hospital));
            return new ApiResponse(ResponseType.OK, "HIS绩效数据提取任务正在执行，稍后我们将以邮件的通知您！");
        }

        /// <summary>
        /// 从WebAPI下载文件
        /// </summary>
        /// <returns></returns>
        [Route("down")]
        public IActionResult DownFile([FromQuery]AllotRequest request)
        {
            var allot = allotService.GetAllot(request.ID);
            if (allot == null || string.IsNullOrWhiteSpace(allot.ExtractPath) || !FileHelper.IsExistFile(allot.ExtractPath))
            {
                return new ObjectResult(new ApiResponse(ResponseType.Fail, "文件不存在"));
            }

            var memoryStream = new MemoryStream();
            using (var stream = new FileStream(allot.ExtractPath, FileMode.Open))
            {
                stream.CopyToAsync(memoryStream).Wait();
            }
            memoryStream.Seek(0, SeekOrigin.Begin);
            string fileExt = Path.GetExtension(allot.ExtractPath);
            var provider = new FileExtensionContentTypeProvider();
            var memi = provider.Mappings[fileExt];
            return File(memoryStream, memi, Path.GetFileName(allot.ExtractPath));
        }

        /// <summary>
        /// 保存提取文件
        /// </summary>
        /// <param name="form"></param>
        /// <returns></returns>
        [Route("savefile")]
        [HttpPost]
        public ApiResponse SaveFile([FromForm] IFormCollection form, int allotId, int hospitalId)
        {
            var file = ((FormFileCollection)form.Files).FirstOrDefault();
            if (file == null)
                return new ApiResponse(ResponseType.Error, "上传文件无效");

            var dpath = Path.Combine(evn.ContentRootPath, "Files", $"{hospitalId}", "autoextract");
            FileHelper.CreateDirectory(dpath);

            var path = Path.Combine(dpath, FileHelper.GetFileName(file.FileName));

            using (var stream = file.OpenReadStream())
            {
                byte[] bytes = new byte[stream.Length];
                stream.Read(bytes, 0, bytes.Length);
                if (!FileHelper.CreateFile(path, bytes))
                    return new ApiResponse(ResponseType.Error, "保存失败");
            }
            var allot = allotService.GetAllot(allotId);
            allot.ExtractPath = path;
            allot.IsExtracting = 2;
            if (!string.IsNullOrEmpty(path) && allotService.Update(allot))
                return new ApiResponse(ResponseType.OK, "上传成功！");
            else
                return new ApiResponse(ResponseType.Error);
        }
    }
}