﻿using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Http.Internal;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Controllers;
using Microsoft.AspNetCore.Mvc.Filters;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Logging;
using Performance.DtoModels;
using Performance.Infrastructure;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Performance.Api
{
    public class ActionsFilter : IAsyncActionFilter
    {
        private readonly ILogger<ActionsFilter> _logger;
        private readonly IMemoryCache _cache;
        private readonly IHostingEnvironment _env;

        public ActionsFilter(ILoggerFactory factory, IMemoryCache cache, IHostingEnvironment env)
        {
            this._logger = factory.CreateLogger<ActionsFilter>();
            this._cache = cache;
            this._env = env;
        }

        public async Task OnActionExecutionAsync(ActionExecutingContext context, ActionExecutionDelegate next)
        {
            var request = context.HttpContext.Request;
            //启用body倒带功能
            request.EnableRewind();
            //记录Request请求
            var kv = GetRequestContent(request);
            _logger.LogInformation($"请求内容 {request.Method}:{JsonHelper.Serialize(kv)}");

            LogHelper.Information($"请求地址：{context.HttpContext.Request.Path};请求参数：{JsonHelper.Serialize(kv)}", "请求内容");
            //接口禁用
            var disable = ((ControllerActionDescriptor)context.ActionDescriptor).MethodInfo.GetCustomAttributes(typeof(DisableAttribute), true);
            if (disable.Length > 0)
            {
                var response = new ApiResponse(ResponseType.Disable, "接口已禁用");
                context.Result = new ObjectResult(response);
                return;
            }
            //token验证
            if (!_env.IsDevelopment())
            {
                var arry = ((ControllerActionDescriptor)context.ActionDescriptor).MethodInfo.GetCustomAttributes(typeof(NoVerifyAttribute), true);
                if (arry.Length == 0)
                {
                    var token = kv.GetValue("token", "");
                    var user = _cache.Get<UserIdentity>(token);
                    if (string.IsNullOrEmpty(token) || user == null || !user.Token.Equals(token))
                    {
                        var response = new ApiResponse(ResponseType.TokenError, "Token无效");
                        context.Result = new ObjectResult(response);
                        return;
                    }
                }
            }
            //验证请求参数
            if (!context.ModelState.IsValid)
            {
                var messageList = context.ModelState.Values
                    .Where(t => !string.IsNullOrEmpty(t?.Errors?.FirstOrDefault()?.ErrorMessage))
                    .Select(t => t?.Errors?.FirstOrDefault()?.ErrorMessage);
                var response = new ApiResponse(ResponseType.ParameterError, "参数错误", messageList);
                context.Result = new ObjectResult(response);
                var jsonData = JsonHelper.Serialize(context.Result);
                _logger.LogInformation($"响应结果:{jsonData}");
                LogHelper.Information($"请求地址：{context.HttpContext.Request.Path};响应结果：{jsonData}", "响应结果");
            }
            //记录response结果
            else
            {
                var executedContext = await next();
                if (executedContext.Exception != null)
                    throw executedContext.Exception;

                if (executedContext.Result is ObjectResult)
                {
                    LogHelper.Information(JsonHelper.Serialize(executedContext.Result), "响应结果");
                    var objectResult = (ObjectResult)executedContext.Result;
                    var jsonData = JsonHelper.Serialize(objectResult.Value);
                    _logger.LogInformation($"响应结果:{jsonData}");
                    LogHelper.Information($"请求地址：{context.HttpContext.Request.Path};响应结果：{jsonData}", "响应结果");
                }
            }
        }

        /// <summary>
        /// 读取请求内容
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        private SortedDictionary<string, object> GetRequestContent(HttpRequest request)
        {
            if (request.Method.Equals("POST"))
            {
                if (request.Body.CanSeek)
                {
                    var types = request.ContentType.Split(';');
                    if (types.Contains("application/json"))
                    {
                        using (var stream = request.Body)
                        {
                            stream.Position = 0;
                            var reader = new StreamReader(stream, Encoding.UTF8);
                            var requestContext = reader.ReadToEnd();
                            return JsonHelper.DeserializeLower(requestContext);
                        }
                    }
                    else if (types.Contains("application/x-www-form-urlencoded") || types.Contains("multipart/form-data"))
                    {
                        return request.Form.ToDictionary();
                    }
                    else if (types.Contains("text/xml"))
                    {
                        //暂不处理
                    }
                }
            }
            else
            {
                if (request.Query.Count > 0)
                {
                    var kv = new SortedDictionary<string, object>();
                    foreach (var item in request.Query)
                    {
                        kv.Add(item.Key, item.Value);
                    }
                    return kv;
                }
            }
            return new SortedDictionary<string, object>();
        }
    }

    [AttributeUsage(AttributeTargets.Method)]
    public class NoVerifyAttribute : Attribute { }

    [AttributeUsage(AttributeTargets.Method | AttributeTargets.Class)]
    public class DisableAttribute : Attribute { }
}
