﻿using AutoMapper;
using Newtonsoft.Json.Linq;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services.AllotCompute
{
    /// <summary>
    /// 对excel导入数据进行合并、计算 并保存
    /// </summary>
    public class ProcessComputService : IAutoInjection
    {
        private PerforCofincomeRepository perforCofincomeRepository;
        private PerforCofdrugpropRepository perforCofdrugpropRepository;
        private PerforPersheetRepository perforPerSheetRepository;
        private PerforImdataRepository perforImDataRepository;
        private PerforImheaderRepository perforImHeaderRepository;
        private PerforImemployeeRepository perforImEmployeeRepository;
        private PerforResaccountdoctorRepository perforResAccountdoctorRepository;
        private PerforResaccountnurseRepository perforResAccountnurseRepository;
        private PerforCofdrugtypeRepository perforCofdrugtypeRepository;
        public ProcessComputService(PerforCofincomeRepository perforCofincomeRepository,
            PerforCofdrugpropRepository perforCofdrugpropRepository,
            PerforPersheetRepository perforPerSheetRepository,
            PerforImdataRepository perforImDataRepository,
            PerforImheaderRepository perforImHeaderRepository,
            PerforImemployeeRepository perforImEmployeeRepository,
            PerforResaccountdoctorRepository perforResAccountdoctorRepository,
            PerforResaccountnurseRepository perforResAccountnurseRepository,
            PerforCofdrugtypeRepository perforCofdrugtypeRepository)
        {
            this.perforCofincomeRepository = perforCofincomeRepository;
            this.perforCofdrugpropRepository = perforCofdrugpropRepository;
            this.perforPerSheetRepository = perforPerSheetRepository;
            this.perforImDataRepository = perforImDataRepository;
            this.perforImHeaderRepository = perforImHeaderRepository;
            this.perforImEmployeeRepository = perforImEmployeeRepository;
            this.perforResAccountdoctorRepository = perforResAccountdoctorRepository;
            this.perforResAccountnurseRepository = perforResAccountnurseRepository;
            this.perforCofdrugtypeRepository = perforCofdrugtypeRepository;
        }

        /// <summary>
        /// 合并计算并保存
        /// </summary>
        /// <param name="excel"></param>
        /// <param name="allot"></param>
        /// <returns></returns>
        public List<PerSheet> MergeAndSave(PerExcel excel, per_allot allot)
        {
            List<PerSheet> list = MergeCompute(excel, allot.ID);
            Save(list, allot.ID);
            return list;
        }

        #region save
        /// <summary>
        /// 保存计算后的医生组绩效
        /// </summary>
        /// <param name="sheet"></param>
        /// <param name="allotId"></param>
        /// <returns></returns>
        private void SaveComputeDoctorAccount(PerSheet sheet, int allotId)
        {
            var imsheet = new per_sheet { AllotID = allotId, SheetName = sheet.SheetName, Source = 2, SheetType = (int)sheet.SheetType };
            perforPerSheetRepository.Add(imsheet);

            var dataList = sheet.PerData.Select(t => (PerDataAccountDoctor)t);
            List<res_accountdoctor> addList = new List<res_accountdoctor>();
            foreach (var data in dataList)
            {
                var imdata = Mapper.Map<res_accountdoctor>(data);
                imdata.SheetID = imsheet.ID;
                imdata.AllotID = allotId;
                addList.Add(imdata);
            }
            perforResAccountdoctorRepository.AddRange(addList.ToArray());
        }

        /// <summary>
        /// 保存计算后的护理组绩效
        /// </summary>
        /// <param name="sheet"></param>
        /// <param name="allotId"></param>
        /// <returns></returns>
        private void SaveComputeNurseAccount(PerSheet sheet, int allotId)
        {
            var imsheet = new per_sheet { AllotID = allotId, SheetName = sheet.SheetName, Source = 2, SheetType = (int)sheet.SheetType };
            perforPerSheetRepository.Add(imsheet);

            var dataList = sheet.PerData.Select(t => (PerDataAccountNurse)t);
            List<res_accountnurse> addList = new List<res_accountnurse>();
            foreach (var data in dataList)
            {
                var imdata = Mapper.Map<res_accountnurse>(data);
                imdata.SheetID = imsheet.ID;
                imdata.AllotID = allotId;
                addList.Add(imdata);
            }
            perforResAccountnurseRepository.AddRange(addList.ToArray());
        }


        /// <summary>
        /// 保存通用格式
        /// </summary>
        /// <param name="sheet"></param>
        /// <param name="allotId"></param>
        /// <returns></returns>
        private void SaveCommon(PerSheet sheet, int allotId)
        {
            var imsheet = new per_sheet { AllotID = allotId, SheetName = sheet.SheetName, Source = 2, SheetType = (int)sheet.SheetType };
            perforPerSheetRepository.Add(imsheet);

            List<im_header> addHeadList = new List<im_header>();
            foreach (var header in sheet.PerHeader)
            {
                var imheader = Mapper.Map<im_header>(header);
                imheader.SheetID = imsheet.ID;
                imheader.AllotID = allotId;
                perforImHeaderRepository.Add(imheader);
                if (header.IsHasChildren)
                {
                    foreach (var child in header.Children)
                    {
                        var imheaderChild = Mapper.Map<im_header>(child);
                        imheaderChild.SheetID = imsheet.ID;
                        imheaderChild.ParentID = imheader.ID;
                        imheaderChild.AllotID = allotId;
                        addHeadList.Add(imheaderChild);
                    }
                }
            }
            perforImHeaderRepository.AddRange(addHeadList.ToArray());

            List<im_data> addDataList = new List<im_data>();
            var dataList = sheet.PerData.Select(t => (PerData)t);
            foreach (var data in dataList)
            {
                var imdata = Mapper.Map<im_data>(data);
                imdata.SheetID = imsheet.ID;
                imdata.AllotID = allotId;
                addDataList.Add(imdata);
            }
            perforImDataRepository.AddRange(addDataList.ToArray());
        }

        /// <summary>
        /// 保存计算过程数据
        /// </summary>
        /// <param name="perSheets"></param>
        /// <param name="allotId"></param>
        private void Save(List<PerSheet> perSheets, int allotId)
        {
            foreach (var sheet in perSheets)
            {
                if (sheet.SheetType == SheetType.ComputeDoctorAccount)
                {
                    SaveComputeDoctorAccount(sheet, allotId);
                }
                else if (sheet.SheetType == SheetType.ComputeNurseAccount)
                {
                    SaveComputeNurseAccount(sheet, allotId);
                }
                else
                {
                    SaveCommon(sheet, allotId);
                }
            }
        }
        #endregion

        #region compute
        /// <summary>
        /// 合并计算
        /// </summary>
        private List<PerSheet> MergeCompute(PerExcel excel, int allotid)
        {
            List<PerSheet> perSheet = new List<PerSheet>();

            //合并科室收入、支出
            var incomeconfs = perforCofincomeRepository.GetEntities();
            var economicCompute = new PerSheetDataComputeEconomic();
            var mergeResult = economicCompute.MergeCompute(excel, incomeconfs);
            //一次计算
            var onceEconomic = economicCompute.OnceCompute(mergeResult);

            //二次计算 
            var twiceEconomicResult = economicCompute.TwiceCompute(onceEconomic);

            twiceEconomicResult.Sheet.SheetType = SheetType.ComputeEconomic;
            perSheet.Add(twiceEconomicResult.Sheet);

            //工作量
            var workloadCompute = new PerSheetDataComputeWorkload();
            var workload1 = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Workload && t.SheetName.Contains("医生组"));
            workload1.SheetName = "医生组工作量绩效测算表";

            var confs = GetDrugConfig(excel, allotid);
            //医生组 一次计算
            var onceWorkload1 = workloadCompute.OnceCompute(workload1, confs);
            //医生组 二次计算 
            var twiceWorkloadResult1 = workloadCompute.TwiceCompute(onceWorkload1);

            twiceWorkloadResult1.Sheet.SheetType = SheetType.ComputeDoctorWorkload;
            perSheet.Add(twiceWorkloadResult1.Sheet);

            var workload2 = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Workload && t.SheetName.Contains("护理组"));
            workload2.SheetName = "护理组工作量绩效测算表";
            //护理组 一次计算
            var onceWorkload2 = workloadCompute.OnceCompute(workload2);
            //护理组 二次计算 
            var twiceWorkloadResult2 = workloadCompute.TwiceCompute(onceWorkload2);

            twiceWorkloadResult2.Sheet.SheetType = SheetType.ComputeNurseWorkload;
            perSheet.Add(twiceWorkloadResult2.Sheet);

            var deptAccounting = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.AccountBasic);
            var dataList = deptAccounting.PerData.Select(t => (PerDataAccountBaisc)t);

            var doctorSheet = ComputeDoctor(dataList, twiceEconomicResult.PerData, twiceWorkloadResult1.PerData);
            perSheet.Add(doctorSheet);
            var nurseSheet = ComputeNurse(dataList, twiceEconomicResult.PerData, twiceWorkloadResult2.PerData);
            perSheet.Add(nurseSheet);

            return perSheet;
        }
        /// <summary>
        /// 计算医生组科室绩效
        /// </summary>
        /// <param name="dataList"></param>
        /// <param name="economicData"></param>
        /// <param name="workloadData"></param>
        /// <returns></returns>
        private PerSheet ComputeDoctor(IEnumerable<PerDataAccountBaisc> dataList, List<PerData> economicData, List<PerData> workloadData)
        {
            PerSheet doctorSheet = new PerSheet("医生组临床科室单元核算表", "医生组临床科室单元核算表", SheetType.ComputeDoctorAccount, new List<PerHeader>(), new List<IPerData>());
            foreach (var dept in dataList)
            {
                if (string.IsNullOrEmpty(dept.AccountingUnit)) continue;

                var econDoctor = economicData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                var workDoctor = workloadData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                var doctor = new PerDataAccountDoctor
                {
                    AccountingUnit = dept.AccountingUnit,
                    Department = dept.AccountingUnit,
                    Number = dept.DoctorNumber,
                    BasicFactor = dept.DoctorBasicFactor,
                    SlopeFactor = dept.DoctorSlopeFactor,
                    OtherPerfor1 = dept.DoctorOtherPerfor1,
                    OtherPerfor2 = dept.DoctorOtherPerfor2,
                    Extra = dept.DoctorExtra,
                    ScoringAverage = dept.DoctorScoringAverage == 0m ? 1 : dept.DoctorScoringAverage,
                    AdjustFactor = dept.DoctorAdjustFactor == 0m ? 1 : dept.DoctorAdjustFactor,
                    Income = econDoctor?.CellValue ?? 0,
                    WorkloadFee = workDoctor?.CellValue ?? 0,
                };
                doctor.PerforFee = doctor.Income * (doctor.BasicFactor + (doctor.BasicFactor * doctor.SlopeFactor));
                doctor.PerforTotal = doctor.PerforFee + doctor.WorkloadFee + doctor.OtherPerfor1;
                doctor.RealGiveFee = (doctor.PerforTotal * doctor.ScoringAverage + doctor.Extra + doctor.OtherPerfor2) * doctor.AdjustFactor;
                doctor.Avg = doctor.Number == 0 ? 0 : doctor.PerforTotal / doctor.Number;

                doctorSheet.PerData.Add(doctor);
            }
            return doctorSheet;
        }

        /// <summary>
        /// 计算护理组科室绩效
        /// </summary>
        /// <param name="dataList"></param>
        /// <param name="economicData"></param>
        /// <param name="workloadData"></param>
        private PerSheet ComputeNurse(IEnumerable<PerDataAccountBaisc> dataList, List<PerData> economicData, List<PerData> workloadData)
        {
            PerSheet nurseSheet = new PerSheet("护理组临床科室单元核算表", "护理组临床科室单元核算表", SheetType.ComputeNurseAccount, new List<PerHeader>(), new List<IPerData>());
            foreach (var dept in dataList)
            {
                if (string.IsNullOrEmpty(dept.Department)) continue;

                var econNurse = economicData.FirstOrDefault(t => t.UnitType == "护理组" && t.AccountingUnit == dept.Department);
                var workNurse = workloadData.FirstOrDefault(t => t.UnitType == "护理组" && t.AccountingUnit == dept.Department);
                var nurse = new PerDataAccountNurse
                {
                    AccountingUnit = dept.Department,
                    Department = dept.Department,
                    Number = dept.NurseNumber,
                    BasicFactor = dept.NurseBasicFactor,
                    SlopeFactor = dept.NurseSlopeFactor,
                    OtherPerfor1 = dept.NurseOtherPerfor1,
                    OtherPerfor2 = dept.NurseOtherPerfor2,
                    Extra = dept.NurseExtra,
                    ScoringAverage = dept.NurseScoringAverage == 0m ? 1 : dept.NurseScoringAverage,
                    AdjustFactor = dept.NurseAdjustFactor == 0m ? 1 : dept.NurseAdjustFactor,
                    Income = econNurse?.CellValue ?? 0,
                    WorkloadFee = workNurse?.CellValue ?? 0,
                };
                nurse.PerforFee = nurse.Income * (nurse.BasicFactor + (nurse.BasicFactor * nurse.SlopeFactor));
                nurse.PerforTotal = nurse.PerforFee + nurse.WorkloadFee + nurse.OtherPerfor1;
                nurse.RealGiveFee = (nurse.PerforTotal * nurse.ScoringAverage + nurse.Extra + nurse.OtherPerfor2) * nurse.AdjustFactor;
                nurse.Avg = nurse.Number == 0 ? 0 : nurse.PerforTotal / nurse.Number;

                nurseSheet.PerData.Add(nurse);
            }
            return nurseSheet;
        }

        /// <summary>
        /// 获取药占比分割比例
        /// </summary>
        /// <param name="excel"></param>
        /// <returns></returns>
        private List<CofDrugProp> GetDrugConfig(PerExcel excel, int allotid)
        {
            //计算药占比
            List<CofDrugProp> cofs = new List<CofDrugProp>();

            var incomeSheet = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Income && t.SheetName.Contains("门诊") && t.SheetName.Contains("就诊"));
            var datalist = incomeSheet.PerData.Select(t => (PerData)t);
            var drugtype = perforCofdrugtypeRepository.GetEntities(t => t.AllotID == allotid)?.Select(t => t.Charge).ToList();

            var drugData = datalist.Where(t => drugtype.Contains(t.TypeName)).GroupBy(t => t.AccountingUnit).Select(t => new { AccountingUnit = t.Key, SumValue = t.Sum(s => s.CellValue) });
            if (drugtype == null)
            {
                drugData = null;
                //throw new PerformanceException("未配置药占比类型");
            }

            var allData = datalist.GroupBy(t => t.AccountingUnit).Select(t => new { AccountingUnit = t.Key, SumValue = t.Sum(s => s.CellValue) });

            var cofList = perforCofdrugpropRepository.GetEntities();

            var unitList = (drugData?.Select(t => t.AccountingUnit) ?? new List<string>()).Union(allData.Select(t => t.AccountingUnit));

            foreach (var unit in unitList)
            {
                var dsv = drugData?.FirstOrDefault(t => t.AccountingUnit == unit)?.SumValue;
                var asv = allData.FirstOrDefault(t => t.AccountingUnit == unit)?.SumValue;

                var prop = asv.HasValue && asv.Value > 0 ? Math.Round((dsv ?? 0) / asv.Value, 2) : 0;
                var fvalue = prop == 0
                    ? 0
                    : cofList.FirstOrDefault(t => prop > t.MinRange && prop <= t.MaxRange)?.Value ?? 0;
                cofs.Add(new CofDrugProp { AccoutingUnit = unit, Factor = fvalue, Prop = prop });
            }
            return cofs;
        }
        #endregion
    }
}
