﻿using System;
using System.Collections.Generic;
using System.Linq;
using Performance.DtoModels;
using Performance.Infrastructure;

namespace Performance.Services
{
    public class UnitTypeUtil
    {
        public static string[] Office = new string[] { "行政后勤", "行政工勤" };
        /// <summary>
        /// 数据收集角色（可查看所有）
        /// </summary>
        public static int[] CollectRoles { get; } = new int[] { (int)Role.绩效管理员, (int)Role.医院管理员, (int)Role.绩效核算办, (int)Role.院领导, (int)Role.财务科, (int)Role.人事科 };
        /// <summary>
        /// 二次分配角色映射表
        /// </summary>
        public static Dictionary<int, string[]> Maps { get; } = new Dictionary<int, string[]>
        {
            { (int)Role.科主任,   new string[]{ UnitType.医生组.ToString(), UnitType.其他医生组.ToString(), UnitType.医技组.ToString(), UnitType.其他医技组.ToString() } },
            { (int)Role.护士长,   new string[]{ UnitType.护理组.ToString(), UnitType.其他护理组.ToString() } },
            { (int)Role.特殊科室, new string[]{ UnitType.特殊核算组.ToString(), } },
            { (int)Role.行政科室, new string[]{ UnitType.行政工勤.ToString(), "行政后勤" } },
        };
        public static Role[] SecondAdmin { get; } = new Role[] { Role.科主任, Role.护士长, Role.特殊科室, Role.行政科室 };

        public static string[] GetMaps(int? key) => (key.HasValue && Maps.ContainsKey(key.Value)) ? Maps[key.Value] : Array.Empty<string>();

        /// <summary>
        /// 是否是行政后勤/工勤
        /// </summary>
        /// <param name="unit"></param>
        /// <returns></returns>
        public static bool IsOffice(int? unit)
        {
            return unit == (int)UnitType.行政工勤;
        }

        /// <summary>
        /// 是否是行政后勤/工勤
        /// </summary>
        /// <param name="unit"></param>
        /// <returns></returns>
        public static bool IsOffice(string unit)
        {
            return !string.IsNullOrEmpty(unit) && Office.Contains(unit);
        }

        /// <summary>
        /// 获得标准名 行政后勤/工勤
        /// </summary>
        /// <param name="unit"></param>
        /// <returns></returns>
        public static string GetOffice(string unit)
        {
            return string.IsNullOrEmpty(unit) ? "" : unit.Replace("行政后勤", "行政工勤");
        }

        /// <summary>
        /// 核算单元类型是否相同
        /// </summary>
        /// <param name="left"></param>
        /// <param name="right"></param>
        /// <returns></returns>
        public static bool IsEqualsUnitType(string left, string right)
        {
            return left?.Replace("行政后勤", "行政工勤") == right?.Replace("行政后勤", "行政工勤");
        }
        /// <summary>
        /// 核算单元类型是否相同
        /// </summary>
        /// <param name="left"></param>
        /// <param name="right"></param>
        /// <returns></returns>
        public static bool IsEqualsUnitType(string left, int? right)
        {
            if (!Enum.IsDefined(typeof(UnitType), right))
                return false;

            return left?.Replace("行政后勤", "行政工勤") == ((UnitType)right).ToString()?.Replace("行政后勤", "行政工勤");
        }

        public static int GetUnitTypeValue(string unitType)
        {
            if (Enum.TryParse(typeof(UnitType), unitType, true, out object value))
                return (int)value;
            return 0;
        }

        public static bool Is(string unit, params string[] types)
        {
            return !string.IsNullOrEmpty(unit) && types.Any(w => IsEqualsUnitType(w, unit));
        }

        public static EnumItem GetUnitTypeFromEnum(string unitType)
        {
            unitType = unitType.Replace("行政后勤", "行政工勤");
            return EnumHelper.GetItems<UnitType>().FirstOrDefault(t => t.Name == unitType);
        }
        /// <summary>
        /// 行政工勤名称替换标准
        /// </summary>
        /// <returns></returns>
        public static List<EnumItem> GetUnitTypeFromEnum()
        {
            var enumItems = EnumHelper.GetItems<UnitType>();
            enumItems.ForEach(t =>
            {
                t.Name = t.Name == "行政后勤" ? UnitType.行政工勤.ToString() : t.Name;
            });
            return enumItems;
        }
    }

    public class AccountTypeUnit
    {
        private static string[] dic = new string[] { "科主任", "护士长" };

        /// <summary>
        /// 自动识别职称文本
        /// </summary>
        /// <param name="name"></param>
        /// <param name="defaultValue"></param>
        /// <returns></returns>
        public static string Recognition(string name, AccountUnitType defaultValue)
        {
            if (dic.Any(w => !string.IsNullOrEmpty(name) && name.Contains(w)))
                return dic.First(w => !string.IsNullOrEmpty(name) && name.Contains(w));

            return defaultValue.ToString();
        }

    }
}
