﻿using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Caching.Memory;
using Performance.DtoModels;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Claims;
using System.Threading.Tasks;

namespace Performance.Api
{
    public class ClaimService
    {
        private readonly IHttpContextAccessor contextAccessor;
        private readonly IMemoryCache memoryCache;

        public ClaimService(IHttpContextAccessor contextAccessor, IMemoryCache memoryCache)
        {
            this.contextAccessor = contextAccessor;
            this.memoryCache = memoryCache;
        }

        /// <summary>
        /// 获取当前请求登录ID
        /// </summary>
        /// <returns></returns>
        public int GetUserId()
        {
            var claim = GetUserClaim().FirstOrDefault(t => t.Type == JwtClaimTypes.Id);
            if (claim == null)
            {
                throw new PerformanceTokenErrorException("获取当前登录用户ID失败");
            }
            return Convert.ToInt32(claim.Value);
        }

        /// <summary>
        /// 获取当前请求登录ID
        /// </summary>
        /// <returns></returns>
        public string GetUserClaim(string jwtClaimTypes)
        {
            var claim = GetUserClaim().FirstOrDefault(t => t.Type == jwtClaimTypes);
            if (claim == null)
            {
                throw new PerformanceTokenErrorException("获取当前登录用户ID失败");
            }
            return claim.Value;
        }

        /// <summary>
        /// 获取当前请求所有身份信息
        /// </summary>
        /// <returns></returns>
        public List<Claim> GetUserClaim()
        {
            if (contextAccessor.HttpContext?.User == null)
            {
                throw new PerformanceException("获取当前请求登录信息失败");
            }
            return contextAccessor.HttpContext.User.Claims.ToList();
        }

        /// <summary>
        /// 获取当前请求Jwt Token
        /// </summary>
        /// <returns></returns>
        public string GetJwtToken()
        {
            var authorization = contextAccessor.HttpContext.Request.Headers["Authorization"];
            if (authorization.Count == 0 || string.IsNullOrEmpty(authorization.First()))
            {
                throw new PerformanceException("获取当前请求Authorization失败");
            }
            return authorization.First().Replace("Bearer ", "");
        }

        /// <summary>
        /// 设置jwt进入黑名单
        /// </summary>
        /// <param name="token"></param>
        /// <returns></returns>
        public bool SetJwtBlacklist(string token)
        {
            memoryCache.Set(token, DateTime.Now);
            return true;
        }

        /// <summary>
        /// 判断当前请求JWT是否可用 可用true
        /// </summary>
        /// <returns></returns>
        public bool JwtUsable()
        {
            string token = GetJwtToken();
            return JwtUsable(token);
        }

        /// <summary>
        /// 判断当前请求JWT是否可用 可用true
        /// </summary>
        /// <param name="token"></param>
        /// <returns></returns>
        public bool JwtUsable(string token)
        {
            var @object = memoryCache.Get(token);
            return @object == null;
        }
    }
}
