﻿using FluentValidation.AspNetCore;
using Microsoft.AspNetCore.Mvc;
using Performance.DtoModels;
using Performance.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Performance.Api.Controllers
{
    /// <summary>
    /// 报表
    /// </summary>
    [Route("api/[controller]")]
    public class ReportController : Controller
    {
        private ReportService reportService;
        private ClaimService claimService;
        public ReportController(ReportService reportService, ClaimService claimService)
        {
            this.reportService = reportService;
            this.claimService = claimService;
        }

        /// <summary>
        /// 首页数据概况
        /// </summary>
        /// <returns></returns>
        [Route("survey")]
        [HttpPost]
        public ApiResponse Survey([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var result = reportService.Survey(request.HospitalId);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 科室医生人均绩效（含科主任）
        /// </summary>
        /// <returns></returns>
        [Route("doctoravg")]
        [HttpPost]
        public ApiResponse DoctorAvg([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var result = reportService.DoctorAvg(request.HospitalId, request.IsIndex);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 科室护士人均绩效（含护士长）
        /// </summary>
        /// <returns></returns>
        [Route("nurseavg")]
        [HttpPost]
        public ApiResponse NurseAvg([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var result = reportService.NurseAvg(request.HospitalId, request.IsIndex);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 门诊患者均次费用
        /// </summary>
        /// <returns></returns>
        [Route("outfeeavg")]
        [HttpPost]
        public ApiResponse OutFeeAvg([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var list = reportService.OutFeeAvg(request.HospitalId);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 住院患者均次费用
        /// </summary>
        /// <returns></returns>
        [Route("inpatfeeavg")]
        [HttpPost]
        public ApiResponse InpatFeeAvg([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var list = reportService.InpatFeeAvg(request.HospitalId);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 科室药占比
        /// </summary>
        /// <returns></returns>
        [Route("medicine")]
        [HttpPost]
        public ApiResponse Medicine([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var list = reportService.Medicine(request.HospitalId, request.IsIndex);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 科室有效收入占比
        /// </summary>
        /// <returns></returns>
        [Route("income")]
        [HttpPost]
        public ApiResponse Income([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var list = reportService.Income(request.HospitalId, request.IsIndex);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 月群体人均绩效
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getperforavg")]
        [HttpPost]
        public ApiResponse AvgPerfor([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var list = reportService.GetAvgPerfor(request.HospitalId);
            return new ApiResponse(ResponseType.OK, "", list);
        }


        /// <summary>
        /// 人群绩效比
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("avgratio")]
        [HttpPost]
        public ApiResponse AvgRatio([CustomizeValidator(RuleSet = "Query"), FromBody]ReportRequest request)
        {
            var list = reportService.AvgRatio(request.HospitalId);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 首页报表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("index")]
        [HttpPost]
        public ApiResponse IndexReport([CustomizeValidator(RuleSet = "Index"), FromBody]ReportRequest request)
        {
            var list = reportService.IndexReport(request.HospitalId, request.Source);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 菜单报表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("menu")]
        [HttpPost]
        public ApiResponse MenuReport([CustomizeValidator(RuleSet = "Menu"), FromBody]ReportRequest request)
        {
            var list = reportService.MenuReport(request);
            return new ApiResponse(ResponseType.OK, "", list);
        }
    }
}