﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using FluentValidation.AspNetCore;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Options;
using Performance.DtoModels;
using Performance.DtoModels.AppSettings;
using Performance.Infrastructure;
using Performance.Services;

namespace Performance.Api.Controllers
{
    [Route("api/[controller]")]
    public class ModExtractController : Controller
    {
        private ModExtractService modExtractService;
        private WebapiUrl url;
        public ModExtractController(ModExtractService modExtractService,
            IOptions<WebapiUrl> url)
        {
            this.modExtractService = modExtractService;
            this.url = url.Value;
        }

        /// <summary>
        /// 绩效数据抽取模板
        /// </summary>
        /// <returns></returns>
        [Route("scheme")]
        [HttpPost]
        public ApiResponse Extract([CustomizeValidator(RuleSet = "Query"), FromBody]ModModuleRequest request)
        {
            if (request.HospitalId == null || request.HospitalId.Value == 0)
                return new ApiResponse(ResponseType.ParameterError, "HospitalId 不存在，请重新选择！");
            if (request.ExecuteType == null || !request.ExecuteType.Any())
                return new ApiResponse(ResponseType.ParameterError, "ExecuteType 不存在，请重新选择！");

            var list = modExtractService.ExtractScheme(request.HospitalId.Value, request.ExecuteType);
            return new ApiResponse(ResponseType.OK, list);
        }

        /// <summary>
        /// 费用类型
        /// </summary>
        /// <returns></returns>
        [Route("type")]
        [HttpPost]
        public ApiResponse FeeType()
        {
            var list = modExtractService.FeeType();
            return new ApiResponse(ResponseType.OK, list);
        }

        /// <summary>
        /// 绩效考核项费用来源
        /// </summary>
        /// <returns></returns>
        [Route("source")]
        [HttpPost]
        public ApiResponse FeeSource([FromBody]ModModuleRequest request)
        {
            if (request.HospitalId == null || request.HospitalId.Value == 0)
                return new ApiResponse(ResponseType.ParameterError, "HospitalId 参数错误！");

            string retJson = HttpHelper.HttpPost(url.HttpPost + "/modextract/source", JsonHelper.Serialize(request), true);
            var ret = JsonHelper.Deserialize<ApiResponse>(retJson);
            return new ApiResponse(ResponseType.OK, ret.Data);
        }

        /// <summary>
        /// 费用字典新增
        /// </summary>
        /// <returns></returns>
        [Route("addmodule")]
        [HttpPost]
        public ApiResponse AddModule([CustomizeValidator(RuleSet = "Add"), FromBody]ModModuleRequest request)
        {
            if (request.HospitalId == null || request.HospitalId.Value == 0)
                return new ApiResponse(ResponseType.ParameterError, "HospitalId 参数错误！");

            var entity = modExtractService.AddModule(request);
            return new ApiResponse(ResponseType.OK, "添加成功！", entity);
        }

        /// <summary>
        /// 费用字典（绩效模板）
        /// </summary>
        /// <returns></returns>
        [Route("modules")]
        [HttpPost]
        public ApiResponse Module([CustomizeValidator(RuleSet = "Query"), FromBody]ModModuleRequest request)
        {
            if (request.HospitalId == null || request.HospitalId.Value == 0)
                return new ApiResponse(ResponseType.ParameterError, "HospitalId 不存在，请重新选择！");

            var list = modExtractService.Module(request.HospitalId.Value);
            return new ApiResponse(ResponseType.OK, list);
        }

        /// <summary>
        /// 绩效模板修改
        /// </summary>
        /// <returns></returns>
        [Route("editmodule")]
        [HttpPost]
        public ApiResponse EditModule([FromBody]ModModuleRequest request)
        {
            if (request.ModuleId == null || request.ModuleId == 0)
                return new ApiResponse(ResponseType.ParameterError, "ModuleId 参数错误！");

            var entity = modExtractService.EditModule(request);
            return new ApiResponse(ResponseType.OK, "修改成功！", entity);
        }

        /// <summary>
        /// 绩效模板删除
        /// </summary>
        /// <returns></returns>
        [Route("deletemodule")]
        [HttpPost]
        public ApiResponse DelModule([FromBody]ModModuleRequest request)
        {
            if (request.ModuleId == null || request.ModuleId == 0)
                return new ApiResponse(ResponseType.ParameterError, "ModuleId 参数错误！");

            modExtractService.DelModule(request.ModuleId.Value);
            return new ApiResponse(ResponseType.OK, "删除成功！");
        }

        /// <summary>
        /// 绩效收入模板配置项新增
        /// </summary>
        /// <returns></returns>
        [Route("additem")]
        [HttpPost]
        public ApiResponse AddItem([FromBody]ItemListRequest request)
        {
            if (request.ModuleId == null && request.ModuleId == 0)
                return new ApiResponse(ResponseType.ParameterError, "ModuleId 参数错误！");

            if (request.Items == null && !request.Items.Any())
                return new ApiResponse(ResponseType.ParameterError, "Items 未发现任添加何项！");

            var list = modExtractService.AddItem(request);
            return new ApiResponse(ResponseType.OK, "添加成功！", list);
        }

        /// <summary>
        /// 绩效收入模板配置项列表
        /// </summary>
        /// <returns></returns>
        [Route("items")]
        [HttpPost]
        public ApiResponse Items([FromBody]ModItemRequest request)
        {
            LogHelper.Information(url.HttpPost + "/modextract/items", "请求地址");
            HttpHelper.HttpPost(url.HttpPost + "/modextract/items", JsonHelper.Serialize(request), true);
            var list = modExtractService.Items(request.ModuleId.Value);
            return new ApiResponse(ResponseType.OK, list);
        }

        /// <summary>
        /// 绩效收入模板配置项修改
        /// </summary>
        /// <returns></returns>
        [Route("edititem")]
        [HttpPost]
        public ApiResponse EditItem([FromBody]ItemListRequest request)
        {
            if (request.Items == null || !request.Items.Any())
                return new ApiResponse(ResponseType.ParameterError, "请选择需要修改的数据！");

            var entity = modExtractService.EditItem(request.Items[0]);
            return new ApiResponse(ResponseType.OK, "修改成功！", entity);
        }

        /// <summary>
        /// 绩效收入模板配置项删除
        /// </summary>
        /// <returns></returns>
        [Route("deleteitem")]
        [HttpPost]
        public ApiResponse DelItem([FromBody]ModItemRequest request)
        {
            if (request.ItemId == null && request.ItemId == 0)
                return new ApiResponse(ResponseType.ParameterError, "ItemId 参数错误！");

            modExtractService.DelItem(request.ItemId.Value);
            return new ApiResponse(ResponseType.OK, "删除成功！");
        }


        #region 特殊科室模板
        /// <summary>
        /// 特殊科室模板配置项新增
        /// </summary>
        /// <returns></returns>
        [Route("addspecial")]
        [HttpPost]
        public ApiResponse AddSpecial([FromBody]SpecialListRequest request)
        {
            if (request.HospitalId == 0)
                return new ApiResponse(ResponseType.ParameterError, "HospitalId 参数错误！");

            if (request.Items == null && !request.Items.Any())
                return new ApiResponse(ResponseType.ParameterError, "Items 未发现任添加何项！");

            var list = modExtractService.AddSpecial(request);
            return new ApiResponse(ResponseType.OK, "添加成功！", list);
        }

        /// <summary>
        /// 特殊科室模板配置项列表
        /// </summary>
        /// <returns></returns>
        [Route("specials")]
        [HttpPost]
        public ApiResponse Specials([FromBody]ModSpecialRequest request)
        {
            var list = modExtractService.Special(request.HospitalId.Value);
            return new ApiResponse(ResponseType.OK, list);
        }

        /// <summary>
        /// 特殊科室模板配置项修改
        /// </summary>
        /// <returns></returns>
        [Route("editspecial")]
        [HttpPost]
        public ApiResponse EditSpecial([FromBody]SpecialListRequest request)
        {
            if (request.Items == null || !request.Items.Any())
                return new ApiResponse(ResponseType.ParameterError, "请选择需要修改的数据！");

            var entity = modExtractService.EditSpecial(request.Items[0]);
            return new ApiResponse(ResponseType.OK, "修改成功！", entity);
        }

        /// <summary>
        /// 特殊科室模板配置项删除
        /// </summary>
        /// <returns></returns>
        [Route("deletespecial")]
        [HttpPost]
        public ApiResponse DelSpecial([FromBody]ModSpecialRequest request)
        {
            if (request.SpecialId == null && request.SpecialId == 0)
                return new ApiResponse(ResponseType.ParameterError, "SpecialId 参数错误！");

            modExtractService.DelSpecial(request.SpecialId.Value);
            return new ApiResponse(ResponseType.OK, "删除成功！");
        }


        /// <summary>
        /// 特殊科室人均
        /// </summary>
        /// <returns></returns>
        [Route("perfortype")]
        [HttpPost]
        public ApiResponse PerforType()
        {
            var list = modExtractService.PerforType();
            return new ApiResponse(ResponseType.OK, list);
        }
        #endregion
    }
}