﻿using AutoMapper;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services
{
    public class GuaranteeService : IAutoInjection
    {
        private readonly PerforPerallotRepository perforPerAllotRepository;
        private readonly PerforImdataRepository perforImdataRepository;
        private readonly PerforCofguaranteeRepository perforCofguaranteeRepository;
        public GuaranteeService(PerforPerallotRepository perforPerAllotRepository,
            PerforImdataRepository perforImdataRepository,
            PerforCofguaranteeRepository perforCofguaranteeRepository)
        {
            this.perforPerAllotRepository = perforPerAllotRepository;
            this.perforImdataRepository = perforImdataRepository;
            this.perforCofguaranteeRepository = perforCofguaranteeRepository;
        }


        #region cof_guarantee
        /// <summary>
        /// 保底绩效配置列表
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public List<cof_guarantee> Guarantee(int allotId)
        {
            var list = perforCofguaranteeRepository.GetEntities(t => t.AllotId == allotId);
            return list?.OrderBy(t => t.Priority).ThenBy(t => t.Target).ToList();
        }

        /// <summary>
        /// 保底绩效配置列表
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public List<GuaranteeResponse> GuaranTree(int allotId)
        {
            var list = perforCofguaranteeRepository.GetEntities(t => t.AllotId == allotId);
            if (list != null && list.Any())
            {
                var group = list
                    .Where(t => !string.IsNullOrEmpty(t.Target) && !string.IsNullOrEmpty(t.Source))
                    .GroupBy(t => new { t.UnitType, t.Target })
                    .Select(g => new GuaranteeResponse
                    {
                        AllotId = allotId,
                        Target = g.Key.Target,
                        Source = g.Select(t => new GuaranItems { GId = t.Id, GValue = t.Source }).ToList(),
                        Priority = g.Min(m => m.Priority.Value),
                        UnitType = ((UnitType)g.Key.UnitType.Value).ToString()
                    }).OrderBy(t => t.UnitType).ThenBy(t => t.Priority).ToList();
                return group;
            }
            return new List<GuaranteeResponse>();
        }

        /// <summary>
        /// 新增保底绩效配置
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public cof_guarantee GuarantInsert(GuaranteeRequest request)
        {
            var list = perforCofguaranteeRepository.GetEntities(t => t.AllotId == request.AllotId);
            if (list != null)
            {
                var config = list.FirstOrDefault(t => t.Priority == request.Priority && t.UnitType == request.UnitType && t.Target != request.Target);
                if (config != null)
                    throw new PerformanceException("优先级无效，已存在");
            }

            var guarantee = Mapper.Map<cof_guarantee>(request);
            perforCofguaranteeRepository.Add(guarantee);
            return guarantee;
        }

        /// <summary>
        /// 修改保底绩效配置
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public cof_guarantee GuarantUpdate(GuaranteeRequest request)
        {
            var guarantee = perforCofguaranteeRepository.GetEntity(t => t.Id == request.Id);
            if (guarantee == null)
                throw new PerformanceException("request.Id 参数错误");

            if (guarantee.Priority > request.Priority)
            {
                var list = perforCofguaranteeRepository.GetEntities(t => t.AllotId == guarantee.AllotId && t.UnitType == request.UnitType && t.Priority >= request.Priority && t.Priority < guarantee.Priority);
                if (list != null && list.Any())
                {
                    list?.ForEach(t => { t.Priority = t.Priority + 1; });
                    perforCofguaranteeRepository.UpdateRange(list.ToArray());
                }
            }
            else if (guarantee.Priority < request.Priority)
            {
                var list = perforCofguaranteeRepository.GetEntities(t => t.AllotId == guarantee.AllotId && t.UnitType == request.UnitType && t.Priority <= request.Priority && t.Priority > guarantee.Priority);
                if (list != null && list.Any())
                {
                    list.ForEach(t => { t.Priority = t.Priority - 1; });
                    perforCofguaranteeRepository.UpdateRange(list.ToArray());
                }
            }

            guarantee.Priority = request.Priority;
            guarantee.UnitType = request.UnitType;
            guarantee.Target = request.Target;
            guarantee.Source = request.Source;
            if (!perforCofguaranteeRepository.Update(guarantee))
                throw new PerformanceException("保存失败");
            return Mapper.Map<cof_guarantee>(request);
        }

        /// <summary>
        /// 删除保底绩效配置
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public bool GuarantDelete(int id)
        {
            var guarantee = perforCofguaranteeRepository.GetEntity(t => t.Id == id);
            if (guarantee == null)
                return false;
            return perforCofguaranteeRepository.Remove(guarantee);
        }

        /// <summary>
        /// 医院核算单元
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public List<TitleValue> Accounting(int allotId)
        {
            List<TitleValue> result = new List<TitleValue>();
            var allot = perforPerAllotRepository.GetEntity(t => t.ID == allotId);
            if (allot == null)
                return result;
            var allotIds = perforPerAllotRepository.GetEntities(t => t.HospitalId == allot.HospitalId).Select(t => t.ID);
            var data = perforImdataRepository.GetEntities(t => allotIds.Contains(t.AllotID.Value))?.Select(t => t.AccountingUnit);
            if (data != null && data.Any())
            {
                data = data.Distinct().Where(t => !string.IsNullOrEmpty(t)).OrderBy(t => t).ToList();
                result.AddRange(data.Select(t => new TitleValue { Title = t, Value = t }));
            }
            return result;
        }

        /// <summary>
        /// 获取优先级
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public int Priority(int allotId, int unitType, string target)
        {
            return perforCofguaranteeRepository.GetEntity(t => t.AllotId == allotId && t.UnitType == unitType && t.Target == target)?.Priority ?? 0;
        }
        #endregion
    }
}
