﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Performance.DtoModels;
using Performance.EntityModels.Entity;
using Performance.EntityModels.Other;
using Performance.Services;
using System.Collections.Generic;

namespace Performance.Api.Controllers
{
    [Route("api/[controller]")]
    [ApiController]
    public class AttendanceController : ControllerBase
    {
        private readonly AttendanceService _attendanceService;
        public AttendanceController(
            AttendanceService attendanceService
            )
        {
            _attendanceService = attendanceService;
        }
        /*
         
        per_attendance   考勤-调动记录表 
        per_attendance_type   考勤-考勤类型 
        per_attendance_vacation   考勤-考勤记录表 

        view_attendance  考勤视图

         */
        /// <summary>
        /// 查询绩效考勤记录
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        [HttpGet("GetAttendance/{allotId}")]
        public ApiResponse<List<AttendanceStatistics>> GetAttendance(int allotId)
        {
            // 查询考勤视图，并按照设计图做格式转换 仅查询开始结束
            var result = _attendanceService.GetAttendance(allotId);

            return result;
        }

        #region 调动记录

        /// <summary>
        /// 查询绩效调动记录
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        [HttpGet("CallIn/{allotId}")]
        public ApiResponse<List<view_attendance>> GetCallIn(int allotId)
        {
            // 查询考勤视图，并按照设计图做格式转换 仅查询调入
            var result = _attendanceService.GetCallIn(allotId);
            return result;
        }

        /// <summary>
        /// 返回HandsonTable格式调动记录
        /// </summary>
        /// <returns></returns>
        [HttpGet("CallIn/GetBatch")]
        [ProducesResponseType(typeof(HandsonTable), StatusCodes.Status200OK)]
        public ApiResponse GetBatchCallInHandsonTable()
        {
            // 返回HandsonTable格式调动记录
            return new ApiResponse(ResponseType.OK, _attendanceService.GetBatchCallInHandsonTable());
        }

        /// <summary>
        /// 批量插入调动记录
        /// </summary>
        /// <param name="allotId"></param>
        /// <param name="hospitalId"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost("CallIn/Batch/{allotId}")]
        public ApiResponse BatchCallIn(int allotId, int hospitalId, SaveCollectData request)
        {
            // obj自己定义结构
            // 批量插入调动记录，插入前需要删除所有后重新插入

            // 需要验证工号和姓名是否与“人员字典”（per_employee）完全匹配，不匹配则返回表格错误提示
            // 需要验证核算组别和核算单元是否与“核算单元及组别”（cof_accounting）完全匹配，不匹配则返回表格错误提示

            // 表格错误提醒参考PersonService.CreatePerson方法
            return _attendanceService.BatchCallIn(allotId, hospitalId, request);
        }
        #endregion

        #region 考勤类型
        /// <summary>
        /// 查询绩效考勤类型
        /// </summary>
        /// <param name="allotId"></param>
        /// <param name="hospitalId"></param>
        /// <returns></returns>
        [HttpGet("Type/{allotId},{hospitalId}")]
        public ApiResponse<List<per_attendance_type>> GetAttendanceType(int allotId, int hospitalId)
        {
            return _attendanceService.GetAttendanceType(allotId, hospitalId);
        }
        /// <summary>
        /// 新增或修改考勤类型
        /// </summary>
        /// <param name="allotId"></param>
        /// <param name="hospitalId"></param>
        /// <param name="attendanceType"></param>
        /// <returns></returns>
        [HttpPost("Type/Edit/{allotId},{hospitalId}")]
        public ApiResponse<AttendanceType> InsertAttendanceType(int allotId, int hospitalId, AttendanceType attendanceType)
        {
            // obj自己定义结构
            return _attendanceService.InsertAttendanceType(allotId, hospitalId, attendanceType);
        }
        /// <summary>
        /// 删除考勤类型
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpPost("Type/Delete/{id}")]
        public ApiResponse DeleteAttendanceType(int id)
        {
            // 删除前需要验证当前类型是否被使用，如果被使用则禁止删除
            return _attendanceService.DeleteAttendanceType(id);
        }
        #endregion

        #region 考勤记录
        /// <summary>
        /// 返回HandsonTable格式考勤记录
        /// </summary>
        /// <returns></returns>
        [HttpGet("Vacation")]
        [ProducesResponseType(typeof(HandsonTable), StatusCodes.Status200OK)]
        public ApiResponse GetAttendanceVacationHandsonTable()
        {
            // 返回HandsonTable格式考勤记录
            return new ApiResponse(ResponseType.OK, _attendanceService.GetAttendanceVacationHandsonTable());
        }

        /// <summary>
        /// 查询考勤记录
        /// </summary>
        /// <param name="allotId"></param>
        /// <param name="hospitalId"></param>
        /// <returns></returns>
        [HttpGet("Vacation/{allotId},{hospitalId}")]
        public ApiResponse<List<RecordAttendcance>> GetAttendanceVacation(int allotId, int hospitalId)
        {
            return _attendanceService.GetAttendanceVacation(allotId, hospitalId);
        }

        /// <summary>
        /// 批量插入考勤记录，插入前需要删除所有后重新插入
        /// </summary>
        /// <param name="allotId"></param>
        /// <param name="hospitalId"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost("Vacation/Batch/{allotId}")]
        public ApiResponse AttendanceBatch(int allotId, int hospitalId, SaveCollectData request)
        {
            // obj自己定义结构
            // 批量插入考勤记录，插入前需要删除所有后重新插入
            // 需要验证考勤类型是否正确
            // 需要验证工号和姓名是否与“人员字典”（per_employee）完全匹配，不匹配则返回表格错误提示

            // 表格错误提醒参考PersonService.CreatePerson方法
            return new ApiResponse(ResponseType.OK, _attendanceService.AttendanceBatch(allotId, hospitalId, request));
        }
        #endregion

        /// <summary>
        /// 考勤结果统计
        /// </summary>
        /// <param name="allotId"></param> 
        /// <returns></returns>
        [HttpGet("statistics/{allotId}")]
        public ApiResponse<List<AttendanceStatistics>> GetAttendanceStatistics(int allotId)
        {
            // 返回结果参考接口 employee/apr/getdeptdetail
            return _attendanceService.GetAttendanceStatistics(allotId);
        }
    }
}
