﻿using System;
using System.Collections.Generic;
using System.Text;

namespace Performance.Infrastructure
{
    /// <summary>
    /// 类型转换
    /// </summary>
    public static class ConvertHelper
    {
        public static int TryInt(string inValue, int defaultValue = default(int))
        {
            int ret = defaultValue;
            int.TryParse(inValue, out ret);
            return ret;
        }

        public static long TryLong(string inValue, long defaultValue = default(int))
        {
            long ret = defaultValue;
            long.TryParse(inValue, out ret);
            return ret;
        }

        public static decimal TryDecimal(string inValue, decimal defaultValue = default(decimal))
        {
            decimal ret = defaultValue;
            decimal.TryParse(inValue, System.Globalization.NumberStyles.Float, default, out ret);
            return ret;
        }

        /// <summary>
        /// 转换为128位可空浮点型,并按指定小数位舍入
        /// </summary>
        /// <param name="input">输入值</param>
        /// <param name="digits">小数位数</param>
        public static decimal? ToDecimalOrNull(object input)
        {
            try
            {
                return decimal.TryParse(input?.ToString(), out decimal tmpvalue) ? tmpvalue : null;
            }
            catch (Exception)
            {
                throw;
            }
        }

        public static DateTime TryDateTime(string inValue, DateTime defaultValue = default(DateTime))
        {
            DateTime ret = defaultValue;
            DateTime.TryParse(inValue, out ret);
            return ret;
        }

        /// <summary>
        /// 返回默认值
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="param"></param>
        /// <param name="defaultValue"></param>
        /// <returns></returns>
        public static T To<T>(object value, T defaultValue = default(T))
        {
            if (value == null)
                return defaultValue;
            if (value is string && string.IsNullOrWhiteSpace(value.ToString()))
                return defaultValue;

            var t = typeof(T);
            var type = Nullable.GetUnderlyingType(t) ?? t;

            var typeName = type.Name.ToLower();
            try
            {
                if (typeName == "string")
                    return (T)(object)value.ToString();
                if (typeName == "guid")
                    return (T)(object)new Guid(value.ToString());
                if (type.IsEnum)
                    return (T)Enum.Parse(type, value.ToString());
                if (value is IConvertible)
                    return (T)Convert.ChangeType(value, type);
                return (T)value;
            }
            catch (Exception)
            {
                return defaultValue;
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <typeparam name = "T"></typeparam>
        /// <param name = "value">The value.</param>
        /// <param name = "defaultValue">The default value.</param>
        /// <param name = "ignoreException">if set to <c>true</c> ignore any exception.</param>
        /// <returns>The target type</returns>
        public static T To<T>(object value, T defaultValue, bool ignoreException)
        {
            if (ignoreException)
            {
                try
                {
                    return To<T>(value);
                }
                catch
                {
                    return defaultValue;
                }
            }
            return To<T>(value);
        }
    }
}
