﻿using AutoMapper;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services
{
    public class HospitalService : IAutoInjection
    {
        private PerforHospitalRepository _hospitalRepository;
        private PerforUserhospitalRepository _joinRepository;
        private PerforHospitalconfigRepository _hospitalconfigRepository;
        private PerforPerfirstRepository _perfirstRepository;
        private PerforPerallotRepository _perallotRepository;
        public HospitalService(PerforHospitalRepository hospitalRepository,
            PerforUserhospitalRepository joinRepository,
            PerforHospitalconfigRepository hospitalconfigRepository,
            PerforPerfirstRepository perfirstRepository,
            PerforPerallotRepository perallotRepository)
        {
            this._hospitalRepository = hospitalRepository;
            this._joinRepository = joinRepository;
            this._hospitalconfigRepository = hospitalconfigRepository;
            this._perfirstRepository = perfirstRepository;
            this._perallotRepository = perallotRepository;
        }

        /// <summary>
        /// 查询用户下属医院
        /// </summary>
        /// <param name="userid"></param>
        /// <returns></returns>
        public List<HospitalResponse> GetUserHopital(int userid)
        {
            if (userid <= 0)
                throw new PerformanceException($"userid：{userid} 错误");

            var joinList = _joinRepository.GetEntities(t => t.UserID == userid);
            if (joinList == null || joinList.Count == 0)
                return null;

            var hosList = _hospitalRepository.GetEntities(t => joinList.Select(j => j.HospitalID).Contains(t.ID));
            var hosId = hosList?.Select(item => item.ID).ToList();
            //获取存在数据记录的hospital
            hosId = _hospitalconfigRepository.GetEntities(t => hosId.Contains(t.HospitalId.Value))?.Select(t => t.HospitalId.Value).ToList();
            //获取已经上传过模板的hospital
            var firstId = _perfirstRepository.GetEntities(t => hosId.Contains(t.HospitalId.Value))?.Select(t => t.HospitalId.Value).ToList();

            var list = Mapper.Map<List<sys_hospital>, List<HospitalResponse>>(hosList);
            list.ForEach(t =>
            {
                if (hosId != null && hosId.Contains(t.HosID))
                {
                    if (firstId != null && firstId.Count > 0)
                        t.HasConfig = firstId.Contains(t.HosID) ? 2 : 1;
                    else
                        t.HasConfig = 1;
                    //绩效列表
                    var allotList = _perallotRepository.GetEntities(item => item.HospitalId == t.HosID);
                    if (allotList != null && allotList.Count > 0)
                        t.HasConfig = 3;
                }
                else
                    t.HasConfig = 0;
            });
            return list;
        }

        /// <summary>
        /// 查询医院
        /// </summary>
        /// <param name="userid"></param>
        /// <returns></returns>
        public sys_hospital GetHopital(int hosid)
        {
            if (hosid <= 0)
                throw new PerformanceException($"hosid：{hosid} 错误");

            return _hospitalRepository.GetEntity(t => t.ID == hosid);
        }

        /// <summary>
        /// 新增
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public bool InsertUserHospital(int userid, int hospitalid)
        {
            var uh = new sys_user_hospital { UserID = userid, HospitalID = hospitalid };
            return _joinRepository.Add(uh);
        }

        /// <summary>
        /// 新增医院
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public HospitalResponse Insert(HospitalRequest request, int userid)
        {
            if (null != _hospitalRepository.GetEntity(t => t.HosName == request.HosName))
                throw new PerformanceException("医院名称重复");
            var hospital = Mapper.Map<sys_hospital>(request);
            hospital.CreateDate = DateTime.Now;
            hospital.CreateUser = userid;
            hospital.States = (int)States.Enabled;

            if (!_hospitalRepository.Add(hospital))
                throw new PerformanceException("保存失败");
            return Mapper.Map<HospitalResponse>(hospital);
        }

        /// <summary>
        /// 修改医院
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public HospitalResponse Update(HospitalRequest request)
        {
            var hospital = _hospitalRepository.GetEntity(t => t.ID == request.ID);
            if (null == hospital)
                throw new PerformanceException($"医院不存在 ID：{request.ID}");
            hospital.HosName = request.HosName;
            hospital.ShortName = request.ShortName;
            hospital.HosLevel = request.HosLevel;
            hospital.HosType = request.HosType;
            hospital.AreaCode = request.AreaCode;
            hospital.States = request.States;
            //hospital.IsOpenWorkYear = request.IsOpenWorkYear;
            //hospital.IsOpenDirector = request.IsOpenDirector;
            hospital.IsOpenDrugprop = request.IsOpenDrugprop;
            hospital.IsShowManage = request.IsShowManage;
            //hospital.IsOpenIncome = request.IsOpenIncome;

            if (!_hospitalRepository.Update(hospital))
                throw new PerformanceException("保存失败");
            return Mapper.Map<HospitalResponse>(hospital);
        }

        /// <summary>
        /// 删除医院
        /// </summary>
        /// <param name="request"></param>
        /// <param name="hosid"></param>
        /// <returns></returns>
        public bool Delete(HospitalRequest request)
        {
            var hospital = _hospitalRepository.GetEntity(t => t.ID == request.ID);
            if (null == hospital)
                throw new PerformanceException($"医院不存在 ID：{request.ID}");

            return _hospitalRepository.Remove(hospital);
        }

        /// <summary>
        /// 启用/禁用医院
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public bool Disabled(HospitalRequest request)
        {
            var hospital = _hospitalRepository.GetEntity(t => t.ID == request.ID);
            if (null == hospital)
                throw new PerformanceException($"医院不存在 ID：{request.ID}");
            hospital.States = request.States;
            return _hospitalRepository.Update(hospital);
        }
    }
}
