﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using Performance.DtoModels;
using Performance.DtoModels.Request;
using Performance.EntityModels;
using Performance.Services;

namespace Performance.Api.Controllers
{

    [Route("api/[controller]")]
    public class BudgetController : Controller
    {
        private readonly ClaimService claim;
        private readonly BudgetService budgetService;

        public BudgetController(ClaimService claim, BudgetService budgetService)
        {
            this.claim = claim;
            this.budgetService = budgetService;
        }

        /// <summary>
        /// 预算管理查询（包含金额、占比）
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("query")]
        public ApiResponse<List<BudgetResponse>> Query([FromBody]BudgetRequest request)
        {
            if (request.HospitalId == 0 || request.Year == 0)
                return new ApiResponse<List<BudgetResponse>>(ResponseType.ParameterError, "参数无效");

            var result = budgetService.QueryBudgetByYear(request.HospitalId, request.Year);
            return new ApiResponse<List<BudgetResponse>>(ResponseType.OK, "", result);
        }

        /// <summary>
        /// 保存预算管理数据
        /// </summary>
        /// <param name="mainYear"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("save/{mainYear}")]
        public ApiResponse Save(int mainYear, [FromBody]List<BudgetResponse> request)
        {
            var userId = claim.GetUserId();
            var result = false;
            if (request != null && request.Any(t => t.Id == 0))
                result = budgetService.SaveBudgetData(mainYear, request, userId);
            else
                result = budgetService.ModifyBudgetData(request);
            return result ? new ApiResponse(ResponseType.OK, "保存成功") : new ApiResponse(ResponseType.Fail, "保存失败");
        }

        /// <summary>
        /// 修改预算管理数据
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify")]
        public ApiResponse Modify([FromBody]List<BudgetResponse> request)
        {
            //var result = budgetService.ModifyBudgetData(request);
            //return result ? new ApiResponse(ResponseType.OK, "修改成功") : new ApiResponse(ResponseType.Fail, "修改失败");
            return new ApiResponse();
        }

        /// <summary>
        /// 预算合计数据查询
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("result/query")]
        public ApiResponse<List<per_budget_result>> Result([FromBody]BudgetRequest request)
        {
            if (request.HospitalId == 0 || request.Year == 0)
                return new ApiResponse<List<per_budget_result>>(ResponseType.ParameterError, "参数无效");

            var result = budgetService.QueryResultByYear(request.HospitalId, request.Year);
            return new ApiResponse<List<per_budget_result>>(ResponseType.OK, "", result);
        }

        /// <summary>
        /// 预算比例、历史比例查询
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("result/ratio")]
        public ApiResponse<List<BudgetRatioResponse>> Ratio([FromBody]BudgetRequest request)
        {
            if (request.HospitalId == 0 || request.Year == 0)
                return new ApiResponse<List<BudgetRatioResponse>>(ResponseType.ParameterError, "参数无效");

            var result = budgetService.QueryBudgetRatio(request.HospitalId, request.Year);
            return new ApiResponse<List<BudgetRatioResponse>>(ResponseType.OK, "", result);
        }

        /// <summary>
        /// 保存预算合计数据
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("result/save")]
        public ApiResponse ResultSave([FromBody]List<per_budget_result> request)
        {
            var userId = claim.GetUserId();
            var result = budgetService.SaveBudgetRatio(request, userId);
            return result ? new ApiResponse(ResponseType.OK, "保存成功") : new ApiResponse(ResponseType.Fail, "保存失败");
        }

        /// <summary>
        /// 使用预算比例
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("result/use/{id}")]
        public ApiResponse UseResult(int id)
        {
            var result = budgetService.UseResult(id);
            return result ? new ApiResponse(ResponseType.OK, "操作成功") : new ApiResponse(ResponseType.Fail, "操作失败");
        }

        /// <summary>
        /// 使用预算比例
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("result/cancel/{id}")]
        public ApiResponse CancelResult(int id)
        {
            var result = budgetService.CancelResult(id);
            return result ? new ApiResponse(ResponseType.OK, "操作成功") : new ApiResponse(ResponseType.Fail, "操作失败");
        }
    }
}