﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Logging;
using Performance.DtoModels;
using Performance.DtoModels.Response;
using Performance.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace Performance.Api.Controllers
{
    [Route("api/collect")]
    [ApiController]
    public class CollectController : ControllerBase
    {
        private readonly ILogger<CollectController> logger;
        private readonly ClaimService claim;
        private readonly CollectService collectService;

        public CollectController(
            ILogger<CollectController> logger,
            ClaimService claim,
            CollectService collectService)
        {
            this.logger = logger;
            this.claim = claim;
            this.collectService = collectService;
        }

        /// <summary>
        /// 查询采集内容
        /// </summary>
        [HttpPost]
        [Route("getcollectcontent")]
        public ApiResponse GetCollectContent([FromQuery] int hospitalId, [FromQuery] int userId)
        {
            if (hospitalId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            var result = collectService.GetCollectContent(hospitalId, userId);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 设置采集录入权限
        /// </summary>
        [HttpPost]
        [Route("setpermission/{userid}")]
        public ApiResponse SetPermission(int userid, [FromBody] IEnumerable<CollectPermission> collects)
        {
            if (collects == null || !collects.Any())
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            collectService.SetPermission(userid, collects);
            return new ApiResponse(ResponseType.OK);
        }

        /// <summary>
        /// 获取采集SHEET名称
        /// </summary>
        [HttpPost]
        [Route("getcollectsheet/{hospitalId}")]
        public ApiResponse GetCollectSheet(int hospitalId)
        {
            if (hospitalId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");

            var regex = new Regex("^[0-9]");
            var userId = claim.GetUserId();
            var result = collectService.GetCollectSheet(hospitalId, userId)
                 ?.Select(w => new { w.SheetType, w.SheetName })
                 .Distinct()
                 .ToList();
            //result?.Add(new { SheetType = -1, SheetName = "预留比例" });

            if (result == null || !result.Any()) return new ApiResponse(ResponseType.OK, result);

            result = result.OrderBy(w => regex.IsMatch(w.SheetName) ? w.SheetName : $"0{w.SheetName}")
                 .ThenBy(w => w.SheetType)
                 .ToList();

            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 加载采集内容
        /// </summary>
        [HttpPost]
        [Route("getcollectdata/{allotId}")]
        public ApiResponse GetCollectData(int allotId, [FromQuery] string sheetName)
        {
            if (allotId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");
            var userId = claim.GetUserId();
            var result = collectService.GetCollectData(userId, allotId, sheetName);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 保存采集内容
        /// </summary>
        [HttpPost]
        [Route("savecollectdata/{allotId}")]
        public ApiResponse SaveCollectData(int allotId, [FromBody] SaveCollectData request)
        {
            if (allotId <= 0)
                return new ApiResponse(ResponseType.ParameterError, "参数无效");
            collectService.SaveCollectData(allotId, request);

            return new ApiResponse(ResponseType.OK);
        }
    }

}