﻿using FluentValidation.AspNetCore;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Options;
using Newtonsoft.Json.Linq;
using Performance.DtoModels;
using Performance.DtoModels.AppSettings;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Performance.Api.Controllers
{
    [Route("api/[controller]")]
    public class ComputeController : Controller
    {
        private ComputeService _computeService;
        private AllotService _allotService;
        private ClaimService _claim;
        public ComputeController(AllotService allotService,
            ComputeService computeService,
            ClaimService claim)
        {
            _allotService = allotService;
            _computeService = computeService;
            _claim = claim;
        }

        /// <summary>
        /// 获取绩效发放列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getcompute")]
        [HttpPost]
        public ApiResponse GetCompute([CustomizeValidator(RuleSet = "Select"), FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetCompute(request.AllotId, request.Type);
            return new ApiResponse(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 特殊科室发放列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getspecial")]
        [HttpPost]
        public ApiResponse<List<res_specialunit>> GetSpecial([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetSpecial(request.AllotId);
            return new ApiResponse<List<res_specialunit>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 医生组科室绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getdoctordata")]
        [HttpPost]
        public ApiResponse<List<DeptResponse>> GetDoctor([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetDoctorPerformance(request.AllotId);
            return new ApiResponse<List<DeptResponse>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 护理组科室绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getnursedata")]
        [HttpPost]
        public ApiResponse<List<DeptResponse>> GetNurse([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetNursePerformance(request.AllotId);
            return new ApiResponse<List<DeptResponse>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 其他组科室绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getotherdata")]
        [HttpPost]
        public ApiResponse<List<DeptResponse>> GetOther([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetOtherPerformance(request.AllotId);
            return new ApiResponse<List<DeptResponse>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 行政科室绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getofficedata")]
        [HttpPost]
        public ApiResponse<List<DeptResponse>> GetOffice([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetOfficePerformance(request.AllotId);
            return new ApiResponse<List<DeptResponse>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 返回院领导、中层、工勤组绩效
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("gethosdata")]
        [HttpPost]
        public ApiResponse<List<DeptResponse>> GetAdminPerformance([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetAdminPerformance(request.AllotId);
            return new ApiResponse<List<DeptResponse>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 科室绩效详情
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("deptdetail")]
        [HttpPost]
        public ApiResponse DeptDetail([FromBody] DeptDetailRequest request)
        {
            ag_secondallot second = null;
            if (request.AccountID == 0)
            {
                if (request.SecondId == 0)
                    return new ApiResponse(ResponseType.ParameterError, "参数 AccountID或SecondId 无效");
                else
                {
                    second = _computeService.GetAccountId(request.SecondId, out int accountId);
                    request.AccountID = accountId;
                }
            }

            if (second != null && second.UnitType == UnitType.特殊核算组.ToString())
            {
                var response = _computeService.SpecialDeptDetail(second);
                return new ApiResponse(ResponseType.OK, response);
            }
            else
            {
                var response = _computeService.DeptDetail(request.AccountID);
                return new ApiResponse(ResponseType.OK, response);
            }
        }

        /// <summary>
        /// 获取全院绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("allcompute")]
        [HttpPost]
        public ApiResponse AllCompute([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var isShowManage = _computeService.IsShowManage(request.AllotId);
            var list = isShowManage == 1 ? _computeService.AllCompute(request.AllotId) : _computeService.AllManageCompute(request.AllotId);
            return new ApiResponse(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 获取全院管理绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("allcompute/management")]
        [HttpPost]
        public ApiResponse AllManageCompute([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.AllManageCompute(request.AllotId);
            return new ApiResponse(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 修改实发绩效
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("updatereal")]
        [HttpPost]
        public ApiResponse UpdateRealfee([CustomizeValidator(RuleSet = "UpdateReal"), FromBody] ComputerRequest request)
        {
            var userId = _claim.GetUserId();
            var realName = _claim.GetUserClaim(JwtClaimTypes.RealName);
            var compute = _computeService.GetComputeSingle(request.ComputeId);
            if (null == compute)
                throw new PerformanceException("当前数据记录不存在");
            compute = _computeService.UpdateRealfee(request, userId, realName);
            return new ApiResponse(ResponseType.OK, "修改成功", compute);
        }

        /// <summary>
        /// 获取全院绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getbaiscnorm")]
        [HttpPost]
        public ApiResponse GetBaiscnorm([FromBody] ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetBaiscnorm(request.AllotId);
            return new ApiResponse(ResponseType.OK, "ok", list);
        }
    }
}