﻿using FluentValidation.AspNetCore;
using Microsoft.AspNetCore.Mvc;
using Performance.DtoModels;
using Performance.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Performance.Api.Controllers
{
    /// <summary>
    /// 报表
    /// </summary>
    [Route("api/report")]
    public class ReportController : Controller
    {
        private AllotService allotService;
        private ReportService reportService;
        private ReportDataService reportDataService;
        private ClaimService claimService;
        public ReportController(
            ClaimService claimService,
            AllotService allotService,
            ReportService reportService,
            ReportDataService reportDataService)
        {
            this.allotService = allotService;
            this.reportService = reportService;
            this.reportDataService = reportDataService;
            this.claimService = claimService;
        }

        [Route("rank")]
        [HttpPost]
        public ApiResponse Rank([FromBody] HospitalIdRequest request)
        {
            var allots = allotService.GetAllotList(request.HospitalId);

            int[] states = new int[] { 6, 8 };
            var result = allots?.Where(w => states.Contains(w.States))
                .Select(w => new { w.Year, w.Month })
                .OrderByDescending(w => w.Year)
                .ThenByDescending(w => w.Month);

            return new ApiResponse(ResponseType.OK, result);
        }

        [Route("selection")]
        [HttpPost]
        public ApiResponse Selection([FromBody] SelectionRequest report)
        {
            var userId = claimService.GetUserId();
            var result = reportDataService.GetReportSelection(report.GroupId, userId);
            return new ApiResponse(ResponseType.OK, result);
        }

        [Route("info")]
        [HttpPost]
        public ApiResponse Info([FromBody] SelectionRequest report)
        {
            var result = reportDataService.GetReportInfo(report.GroupId);
            return new ApiResponse(ResponseType.OK, result);
        }

        [Route("search")]
        [HttpPost]
        public ApiResponse Search([FromBody] SearchReportRequest report)
        {
            if (report.Values == null || !report.Values.Any())
                return new ApiResponse(ResponseType.OK, new List<ReportData>());

            string[] keys = new string[] { "year", "month", };
            foreach (var item in report.Values.Where(t => keys.Contains(t.Title)))
            {
                if (item.Values == null || !item.Values.Any(t => !string.IsNullOrEmpty(t)))
                    return new ApiResponse(ResponseType.OK, new List<ReportData>());
            }

            var userId = claimService.GetUserId();
            var result = reportDataService.GetReportData(report.HospitalId, report.GroupId, report.ReportId, report.Values ?? new List<SelectionValues>(), userId);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 首页数据概况
        /// </summary>
        /// <returns></returns>
        [Route("survey")]
        [HttpPost]
        public ApiResponse Survey([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var result = reportService.Survey(request.HospitalId);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 科室医生人均绩效（含科主任）
        /// </summary>
        /// <returns></returns>
        [Route("doctoravg")]
        [HttpPost]
        public ApiResponse DoctorAvg([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var result = reportService.DoctorAvg(request.HospitalId, request.IsIndex);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 科室护士人均绩效（含护士长）
        /// </summary>
        /// <returns></returns>
        [Route("nurseavg")]
        [HttpPost]
        public ApiResponse NurseAvg([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var result = reportService.NurseAvg(request.HospitalId, request.IsIndex);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 门诊患者均次费用
        /// </summary>
        /// <returns></returns>
        [Route("outfeeavg")]
        [HttpPost]
        public ApiResponse OutFeeAvg([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var list = reportService.OutFeeAvg(request.HospitalId);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 住院患者均次费用
        /// </summary>
        /// <returns></returns>
        [Route("inpatfeeavg")]
        [HttpPost]
        public ApiResponse InpatFeeAvg([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var list = reportService.InpatFeeAvg(request.HospitalId);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 科室药占比
        /// </summary>
        /// <returns></returns>
        [Route("medicine")]
        [HttpPost]
        public ApiResponse Medicine([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var list = reportService.Medicine(request.HospitalId, request.IsIndex);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 科室有效收入占比
        /// </summary>
        /// <returns></returns>
        [Route("income")]
        [HttpPost]
        public ApiResponse Income([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var list = reportService.Income(request.HospitalId, request.IsIndex);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 月群体人均绩效
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getperforavg")]
        [HttpPost]
        public ApiResponse AvgPerfor([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var list = reportService.GetAvgPerfor(request.HospitalId);
            return new ApiResponse(ResponseType.OK, "", list);
        }


        /// <summary>
        /// 人群绩效比
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("avgratio")]
        [HttpPost]
        public ApiResponse AvgRatio([CustomizeValidator(RuleSet = "Query"), FromBody] ReportRequest request)
        {
            var list = reportService.AvgRatio(request.HospitalId);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 首页报表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("index")]
        [HttpPost]
        public ApiResponse IndexReport([CustomizeValidator(RuleSet = "Index"), FromBody] ReportRequest request)
        {
            var list = reportService.IndexReport(request.HospitalId, request.Source);
            return new ApiResponse(ResponseType.OK, "", list);
        }

        /// <summary>
        /// 菜单报表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("menu")]
        [HttpPost]
        public ApiResponse MenuReport([CustomizeValidator(RuleSet = "Menu"), FromBody] ReportRequest request)
        {
            var list = reportService.MenuReport(request);
            return new ApiResponse(ResponseType.OK, "", list);
        }
    }
}