﻿using Dapper;
using Microsoft.Extensions.Logging;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace Performance.Services.ExtractExcelService
{
    public enum ExDataDict
    {
        ExModule = 1,
        ExItem = 2,
        ExSpecial = 3,
    }

    public class QueryService : IAutoInjection
    {
        private readonly ILogger logger;
        private readonly PerforExmoduleRepository exmoduleRepository;
        private readonly PerforExitemRepository exitemRepository;
        private readonly PerforExspecialRepository exspecialRepository;
        private readonly PerforExscriptRepository exscriptRepository;
        private readonly PerforExresultRepository exresultRepository;
        private readonly PerforHospitalconfigRepository hosconfigRepository;
        private readonly PerforPerallotRepository perallotRepository;

        public QueryService(
            ILogger<QueryService> logger,
            PerforExmoduleRepository exmoduleRepository,
            PerforExitemRepository exitemRepository,
            PerforExspecialRepository exspecialRepository,
            PerforExscriptRepository exscriptRepository,
            PerforExresultRepository exresultRepository,
            PerforHospitalconfigRepository hosconfigRepository,
            PerforPerallotRepository perallotRepository
            )
        {
            this.logger = logger;
            this.exmoduleRepository = exmoduleRepository;
            this.exitemRepository = exitemRepository;
            this.exspecialRepository = exspecialRepository;
            this.exscriptRepository = exscriptRepository;
            this.exresultRepository = exresultRepository;
            this.hosconfigRepository = hosconfigRepository;
            this.perallotRepository = perallotRepository;
        }

        private readonly DateTime CreateTime = DateTime.Now;

        /// <summary>
        /// 获取抽取数据
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="allot"></param>
        /// <returns></returns>
        public List<ex_result> Handler(int hospitalId, per_allot allot, ref Dictionary<ExDataDict, object> dict)
        {
            try
            {
                var configs = hosconfigRepository.GetEntities(t => t.HospitalId == hospitalId);
                if (configs == null || !configs.Any()) throw new Exception("医院未配置绩效抽取信息");

                dict = new Dictionary<ExDataDict, object>
                {
                    { ExDataDict.ExModule, new List<ex_module>() },
                    { ExDataDict.ExItem, new List<ex_item>() },
                    { ExDataDict.ExSpecial, new List<ex_script>() }
                };
                var extypeIds = GetQueryScriptIds(hospitalId, ref dict);

                ClearHistoryData(allot.ID);

                var data = new List<ex_result>();
                var scripts = exscriptRepository.GetEntities(t => extypeIds.Contains(t.TypeId));
                if (scripts != null && scripts.Any())
                {
                    var allmodules = dict[ExDataDict.ExModule] as List<ex_module>;

                    foreach (var pair in dict)
                    {
                        switch (pair.Key)
                        {
                            case ExDataDict.ExModule:
                                data.AddRange(ExtractModuleData(allot, scripts, configs, pair.Value));
                                break;
                            case ExDataDict.ExItem:
                                data.AddRange(ExtractItemData(allot, scripts, configs, allmodules, pair.Value));
                                break;
                            case ExDataDict.ExSpecial:
                                data.AddRange(ExtractSpecialData(allot, scripts, configs, pair.Value));
                                break;
                        }
                    }
                }
                return data;
            }
            catch (Exception ex)
            {
                logger.LogError("获取数据时发生异常");
                throw ex;
            }
        }

        /// <summary>
        /// 获取医院抽取项配置
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="dict"></param>
        /// <returns></returns>
        private List<int> GetQueryScriptIds(int hospitalId, ref Dictionary<ExDataDict, object> dict)
        {
            var extypeIds = new List<int>();

            var modules = exmoduleRepository.GetEntities(t => t.HospitalId == hospitalId);
            var items = new List<ex_item>();
            if (modules != null && modules.Any())
            {
                dict[ExDataDict.ExModule] = modules;
                extypeIds.AddRange(modules.Select(t => t.TypeId ?? 0));

                items = exitemRepository.GetEntities(t => t.ModuleId.HasValue && modules.Select(m => m.Id).Contains(t.ModuleId.Value));
                if (items != null && items.Any(t => t.TypeId.HasValue))
                {
                    dict[ExDataDict.ExItem] = items;
                    extypeIds.AddRange(items.Select(t => t.TypeId ?? 0));
                }
            }

            var specials = exspecialRepository.GetEntities(t => t.HospitalId == hospitalId);
            if (specials != null && specials.Any())
            {
                dict[ExDataDict.ExSpecial] = specials;
                extypeIds.AddRange(specials.Select(t => t.TypeId ?? 0));
            }

            var exids = extypeIds.Where(t => t > 0);
            if (exids == null || !exids.Any()) return new List<int>();

            return exids.Distinct().ToList();
        }

        /// <summary>
        /// 清除历史抽取数据
        /// </summary>
        /// <param name="allotId"></param>
        private void ClearHistoryData(int allotId)
        {
            logger.LogInformation($"开始清除历史提取数据");

            perallotRepository.ClearResultData(allotId);

            logger.LogInformation($"清除历史提取数据已完成");
        }

        #region ExResultData

        /// <summary>
        /// 获取收入费用
        /// </summary>
        /// <param name="allot"></param>
        /// <param name="scripts"></param>
        /// <param name="configs"></param>
        /// <param name="dictValue"></param>
        /// <returns></returns>
        private List<ex_result> ExtractModuleData(per_allot allot, List<ex_script> scripts, List<sys_hospitalconfig> configs, object dictValue)
        {
            var data = new List<ex_result>();

            if (dictValue is List<ex_module> modules && modules != null && modules.Any(t => t.TypeId.HasValue && t.TypeId > 0))
            {
                var typeIds = modules.Where(t => t.TypeId.HasValue && t.TypeId > 0).Select(t => t.TypeId.Value).Distinct().ToList();

                if (typeIds == null || typeIds.Count == 0) return new List<ex_result>();

                foreach (var typeId in typeIds)
                {
                    foreach (var script in scripts.Where(t => t.TypeId == typeId))
                    {
                        var config = configs.FirstOrDefault(t => t.Id == script.ConfigId);
                        if (config == null) continue;

                        var querydata = QueryData(config, allot, script.ExecScript);
                        if (querydata != null && querydata.Any())
                        {
                            var thisModules = modules.Where(t => t.TypeId == typeId).ToList();

                            thisModules.ForEach(f =>
                            {
                                var result = querydata.Select(t => new ex_result
                                {
                                    Department = t.Department,
                                    Category = t.Category?.Trim(),
                                    Fee = t.Value,
                                    DoctorName = t.DoctorName,
                                    PersonnelNumber = t.PersonnelNumber,
                                    Source = f.ModuleName,
                                    DatabaseType = config.DataBaseType,
                                    ConfigId = config.Id,
                                    AllotId = allot.ID,
                                    CreateTime = CreateTime,
                                }).ToList();
                                exresultRepository.AddRange(result.ToArray());
                                data.AddRange(result);
                            });
                        }
                    }
                }
            }

            return data;
        }

        /// <summary>
        /// 获取工作量、成本等费用
        /// </summary>
        /// <param name="allot"></param>
        /// <param name="scripts"></param>
        /// <param name="configs"></param>
        /// <param name="modules"></param>
        /// <param name="dictValue"></param>
        /// <returns></returns>
        private List<ex_result> ExtractItemData(per_allot allot, List<ex_script> scripts, List<sys_hospitalconfig> configs, List<ex_module> modules, object dictValue)
        {
            var data = new List<ex_result>();

            if (dictValue is List<ex_item> items && items != null && items.Any(t => t.TypeId.HasValue && t.TypeId > 0))
            {
                var typeIds = items.Where(t => t.TypeId.HasValue && t.TypeId > 0).Select(t => t.TypeId.Value).Distinct().ToList();

                foreach (var typeId in typeIds)
                {
                    foreach (var script in scripts.Where(t => t.TypeId == typeId))
                    {
                        var config = configs.FirstOrDefault(t => t.Id == script.ConfigId);
                        if (config == null) continue;

                        var querydata = QueryData(config, allot, script.ExecScript);
                        if (querydata != null && querydata.Any())
                        {
                            var thisItems = items.Where(t => t.TypeId == typeId).ToList();
                            var modulename = modules.FirstOrDefault(t => t.Id == thisItems.First().ModuleId)?.ModuleName;

                            thisItems.ForEach(f =>
                            {
                                var result = querydata.Select(t => new ex_result
                                {
                                    Department = t.Department,
                                    Category = f.ItemName,
                                    Fee = t.Value,
                                    DoctorName = t.DoctorName,
                                    PersonnelNumber = t.PersonnelNumber,
                                    Source = modulename,
                                    DatabaseType = config.DataBaseType,
                                    ConfigId = config.Id,
                                    AllotId = allot.ID,
                                    CreateTime = CreateTime,
                                }).ToList();
                                exresultRepository.AddRange(result.ToArray());
                                data.AddRange(result);
                            });
                        }
                    }
                }
            }

            return data;
        }

        /// <summary>
        /// 获取特殊核算费用
        /// </summary>
        /// <param name="allot"></param>
        /// <param name="scripts"></param>
        /// <param name="configs"></param>
        /// <param name="dictValue"></param>
        /// <returns></returns>
        private List<ex_result> ExtractSpecialData(per_allot allot, List<ex_script> scripts, List<sys_hospitalconfig> configs, object dictValue)
        {
            var data = new List<ex_result>();

            if (dictValue is List<ex_special> specials && specials != null && specials.Any(t => t.TypeId.HasValue && t.TypeId > 0))
            {
                var typeIds = specials.Where(t => t.TypeId.HasValue && t.TypeId > 0).Select(t => t.TypeId.Value).Distinct().ToList();

                foreach (var typeId in typeIds)
                {
                    foreach (var script in scripts.Where(t => t.TypeId == typeId))
                    {
                        var config = configs.FirstOrDefault(t => t.Id == script.ConfigId);
                        if (config == null) continue;

                        var querydata = QueryData(config, allot, script.ExecScript);
                        if (querydata != null && querydata.Any())
                        {
                            var thisSpecials = specials.Where(t => t.TypeId == typeId).ToList();

                            thisSpecials.ForEach(f =>
                            {
                                var result = querydata.Select(t => new ex_result
                                {
                                    Department = f.Department,
                                    Category = f.Target,
                                    Fee = t.Value,
                                    DoctorName = t.DoctorName,
                                    PersonnelNumber = t.PersonnelNumber,
                                    Source = "4.2 特殊核算单元绩效测算表",
                                    DatabaseType = config.DataBaseType,
                                    ConfigId = config.Id,
                                    AllotId = allot.ID,
                                    CreateTime = CreateTime,
                                }).ToList();
                                exresultRepository.AddRange(result.ToArray());
                                data.AddRange(result);
                            });
                        }
                    }
                }
            }

            return data;
        }

        #endregion

        #region QueryData

        /// <summary>
        /// 查询数据
        /// </summary>
        /// <param name="config"></param>
        /// <param name="allot"></param>
        /// <param name="execsql"></param>
        /// <param name="source"></param>
        /// <param name="category"></param>
        /// <returns></returns>
        private IEnumerable<ExtractDto> QueryData(sys_hospitalconfig config, per_allot allot, string execsql)
        {
            var parameters = GetParameters(allot);
            using (var connection = ConnectionBuilder.Create((DatabaseType)config.DataBaseType, config.DbSource, config.DbName, config.DbUser, config.DbPassword))
            {
                foreach (var item in parameters)
                {
                    execsql = Regex.Replace(execsql, item.Key, item.Value, RegexOptions.IgnoreCase);
                }

                logger.LogInformation($"提取绩效数据SQL脚本{execsql}");
                var result = connection.Query<ExtractDto>(execsql, commandTimeout: 20000);
                logger.LogInformation($"提取绩效数据执行脚本获取数据{result?.Count() ?? 0}条记录");

                return result;
            }
        }

        /// <summary>
        /// 获取参数
        /// </summary>
        /// <param name="allot"></param>
        /// <returns></returns>
        private Dictionary<string, string> GetParameters(per_allot allot)
        {
            DateTime beginTime = new DateTime(allot.Year, allot.Month, 1);

            Dictionary<string, string> pairs = new Dictionary<string, string>
            {
                { "@beginTime", $"'{beginTime.ToString("yyyy-MM-dd")}'" },
                { "@endTime", $"'{beginTime.AddMonths(1).ToString("yyyy-MM-dd")}'"},
            };
            return pairs;
        }

        #endregion
    }
}
