﻿using FluentValidation.AspNetCore;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Options;
using Newtonsoft.Json.Linq;
using Performance.DtoModels;
using Performance.DtoModels.AppSettings;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Performance.Api.Controllers
{
    [Route("api/[controller]")]
    public class ComputeController : Controller
    {
        private ComputeService _computeService;
        private AllotService _allotService;
        private ClaimService _claim;
        public ComputeController(AllotService allotService,
            ComputeService computeService,
            ClaimService claim)
        {
            _allotService = allotService;
            _computeService = computeService;
            _claim = claim;
        }

        /// <summary>
        /// 获取绩效发放列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getcompute")]
        [HttpPost]
        public ApiResponse GetCompute([CustomizeValidator(RuleSet = "Select"), FromBody]ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetCompute(request.AllotId, request.Type);
            return new ApiResponse(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 特殊科室发放列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getspecial")]
        [HttpPost]
        public ApiResponse<List<res_specialunit>> GetSpecial([FromBody]ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetSpecial(request.AllotId);
            return new ApiResponse<List<res_specialunit>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 医生组科室绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getdoctordata")]
        [HttpPost]
        public ApiResponse<List<DoctorResponse>> GetDoctor([FromBody]ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetDoctorPerformance(request.AllotId);
            return new ApiResponse<List<DoctorResponse>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 护理组科室绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getnursedata")]
        [HttpPost]
        public ApiResponse<List<NurseResponse>> GetNurse([FromBody]ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetNursePerformance(request.AllotId);
            return new ApiResponse<List<NurseResponse>>(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 科室绩效详情
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("deptdetail")]
        [HttpPost]
        public ApiResponse<DeptDetailResponse> DeptDetail([FromBody]DeptDetailRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");

            DeptDetailResponse response = _computeService.GetDepartmentDetail(request.AllotId, request.AccountID, request.Type);
            return new ApiResponse<DeptDetailResponse>(ResponseType.OK, response);
        }

        /// <summary>
        /// 获取全院绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("allcompute")]
        [HttpPost]
        public ApiResponse AllCompute([FromBody]ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.AllCompute(request.AllotId);
            return new ApiResponse(ResponseType.OK, "ok", list);
        }

        /// <summary>
        /// 修改实发绩效
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("updatereal")]
        [HttpPost]
        public ApiResponse UpdateRealfee([CustomizeValidator(RuleSet = "UpdateReal"), FromBody] ComputerRequest request)
        {
            var user = _claim.At(request);
            var compute = _computeService.GetComputeSingle(request.ComputeId);
            if (null == compute)
                throw new PerformanceException("当前数据记录不存在");
            compute = _computeService.UpdateRealfee(request, user);
            return new ApiResponse(ResponseType.OK, "修改成功", compute);
        }

        /// <summary>
        /// 获取全院绩效列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("getbaiscnorm")]
        [HttpPost]
        public ApiResponse GetBaiscnorm([FromBody]ComputerRequest request)
        {
            var allot = _allotService.GetAllot(request.AllotId);
            if (null == allot)
                throw new PerformanceException("当前绩效记录不存在");
            var list = _computeService.GetBaiscnorm(request.AllotId);
            return new ApiResponse(ResponseType.OK, "ok", list);
        }
    }
}