﻿using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Repository;
using Performance.Services.AllotCompute;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services
{
    /// <summary>
    /// 临床科室主任
    /// </summary>
    public class ComputeDirector : IAutoInjection
    {
        private readonly BaiscNormService baiscNormService;
        private readonly PerforCofdirectorRepository perforCofdirectorRepository;
        private readonly PerforCofworkyearRepository perforCofworkyearRepository;
        private readonly PerforResaccountnurseRepository perforResAccountnurseRepository;
        private readonly PerforResaccountdoctorRepository perforResAccountdoctorRepository;

        public ComputeDirector(BaiscNormService baiscNormService,
            PerforCofdirectorRepository perforCofdirectorRepository,
            PerforCofworkyearRepository perforCofworkyearRepository,
            PerforResaccountnurseRepository perforResAccountnurseRepository,
            PerforResaccountdoctorRepository perforResAccountdoctorRepository)
        {
            this.baiscNormService = baiscNormService;
            this.perforCofdirectorRepository = perforCofdirectorRepository;
            this.perforCofworkyearRepository = perforCofworkyearRepository;
            this.perforResAccountnurseRepository = perforResAccountnurseRepository;
            this.perforResAccountdoctorRepository = perforResAccountdoctorRepository;
        }

        /// <summary>
        /// 临床科室主任、临床科室副主任、临床科室护士长 计算
        /// </summary>
        /// <param name="empolyeeList"></param>
        /// <param name="nurseList"></param>
        /// <param name="directorList"></param>
        /// <returns></returns>
        public List<ComputeResult> Compute(List<ComputeEmployee> empolyeeList, per_allot allot)
        {
            //规模绩效和效率绩效配置表
            var directorList = perforCofdirectorRepository.GetEntities(t => t.AllotID == allot.ID);
            //取出医生科室
            var doctorList = perforResAccountdoctorRepository.GetEntities(t => t.AllotID == allot.ID);
            //取出护士科室
            var nurseList = perforResAccountnurseRepository.GetEntities(t => t.AllotID == allot.ID);

            List<ComputeResult> computeList = new List<ComputeResult>();
            // 主任及护士长 对应 效率绩效/规模绩效 关系
            // 科室名称 确定绩效计算人员中的人群
            // 绩效考核基数参考对象  确定科室下那些考核对象归纳为 科室主任  护士长
            // 数据库存放的文本 与枚举 全文字中文匹配
            var basicRuleList = new[]
            {
                new { AccountUnitType = AccountUnitType.临床科室, PerforType = PerforType.临床主任, DirectorType = DirectorType.临床科室主任 },
                new { AccountUnitType = AccountUnitType.临床科室, PerforType = PerforType.临床副主任, DirectorType = DirectorType.临床科室副主任 },
                new { AccountUnitType = AccountUnitType.医技科室, PerforType = PerforType.医技主任, DirectorType = DirectorType.医技科室主任 },
                new { AccountUnitType = AccountUnitType.医技科室, PerforType = PerforType.医技副主任, DirectorType = DirectorType.医技科室副主任 },
                new { AccountUnitType = AccountUnitType.临床科室, PerforType = PerforType.护士长, DirectorType = DirectorType.临床科室护士长 },
            };
            foreach (var basicRule in basicRuleList)
            {
                var needCompute = empolyeeList.Where(t => t.AccountType == basicRule.AccountUnitType.ToString() && t.FitPeople.Trim() == EnumHelper.GetDescription(basicRule.PerforType));
                foreach (var item in needCompute)
                {
                    //分别取出对应的 计算 人员 平均值等信息
                    decimal? number, perforTotal, avg;
                    if (basicRule.PerforType == PerforType.护士长)
                    {
                        var resAccount = nurseList.FirstOrDefault(t => t.AccountingUnit == item.AccountingUnit);
                        number = resAccount?.Number;
                        perforTotal = resAccount?.PerforTotal;
                        avg = resAccount?.Avg;
                    }
                    else
                    {
                        var resAccount = doctorList.FirstOrDefault(t => t.AccountingUnit == item.AccountingUnit);
                        number = resAccount?.Number;
                        perforTotal = resAccount?.PerforTotal;
                        avg = resAccount?.Avg;
                    }

                    var efficiency = directorList.FirstOrDefault(t => t.TypeName == "效率绩效" && t.JobTitle == basicRule.DirectorType.ToString());
                    var scale = directorList.FirstOrDefault(t => t.TypeName == "规模绩效" && t.JobTitle == basicRule.DirectorType.ToString());

                    // 科室单独规模绩效 效率绩效 系数
                    var efficAccount = directorList.FirstOrDefault(t => t.TypeName == "效率绩效" && t.JobTitle == item.AccountingUnit);
                    var scaleAccount = directorList.FirstOrDefault(t => t.TypeName == "规模绩效" && t.JobTitle == item.AccountingUnit);

                    var compute = new ComputeResult
                    {
                        AccountType = item.AccountType,
                        AccountingUnit = item.AccountingUnit,
                        EmployeeName = item.DoctorName,
                        FitPeople = item.FitPeople,
                        JobTitle = item.JobTitle,

                        Number = number,
                        PerforTotal = perforTotal,
                        Avg = avg,
                        Efficiency = avg * efficiency?.Value * (efficAccount?.Value ?? 1),
                        Scale = perforTotal * scale?.Value * (scaleAccount?.Value ?? 1),
                        Grant = item.Grant ?? 1,
                        ScoreAverageRate = item.ScoreAverageRate,
                        Punishment = item.Punishment,
                        OtherPerfor = item.OtherPerfor,
                        Adjust = item.Adjust,
                        Workload = item.Workload
                    };
                    //应发管理绩效
                    compute.ShouldGiveFee = (compute.Efficiency + compute.Scale) * (item.Grant ?? 1);
                    //绩效合计
                    var perforSumFee = (compute.Avg + compute.Efficiency + compute.Scale) * (item.Grant ?? 1);
                    // 在科主任护士长绩效测试时，有特殊情况需要乘一个基数
                    //if (perforSumFee > (compute.Avg ?? 0) * 2.5m)
                    //    compute.PerforSumFee = (compute.Avg ?? 0) * 2.5m;
                    //else
                    compute.PerforSumFee = perforSumFee;
                    //应发绩效
                    compute.GiveFee = compute.Avg * (item.Grant ?? 1) + (compute.Efficiency + compute.Scale) * (item.Grant ?? 1) * compute.ScoreAverageRate + (item.Punishment ?? 0) + (item.OtherPerfor ?? 0);
                    //实发绩效
                    //compute.RealGiveFee = compute.GiveFee * (item.Adjust ?? 1m) - compute.ShouldGiveFee * item.ScoreAverageRate * (item.Grant ?? 1) * 0.1m;
                    compute.RealGiveFee = compute.GiveFee;
                    computeList.Add(compute);
                }

            }
            return computeList;
        }

        /// <summary>
        /// 院领导、业务中层、工勤人员 计算
        /// </summary>
        /// <param name="empolyeeList"></param>
        /// <param name="baiscnormList"></param>
        /// <returns></returns>
        public List<ComputeResult> Compute(List<ComputeEmployee> empolyeeList, per_allot allot, List<res_baiscnorm> baiscnormList)
        {
            //年资系数
            var workyearList = perforCofworkyearRepository.GetEntities(t => t.AllotID == allot.ID);

            List<ComputeResult> computeList = new List<ComputeResult>();

            List<string> involve = new List<string> { AccountUnitType.行政高层.ToString(), AccountUnitType.行政中层.ToString(), AccountUnitType.行政工勤.ToString() };
            var needCompute = empolyeeList.Where(t => involve.Contains(t.AccountType));

            if (!needCompute.Any()) return computeList;

            var perforTypeArray = EnumHelper.GetItems<PerforType>();
            foreach (var item in needCompute)
            {
                var compute = new ComputeResult
                {
                    AccountType = item.AccountType,
                    AccountingUnit = item.AccountingUnit,
                    EmployeeName = item.DoctorName,
                    FitPeople = item.FitPeople,
                    Grant = item.Grant ?? 1,
                    WorkTime = item.WorkTime,
                    PostCoefficient = item.PostCoefficient,
                    Attendance = item.Attendance,
                    ScoreAverageRate = item.ScoreAverageRate,
                    Punishment = item.Punishment,
                    OtherPerfor = item.OtherPerfor,
                    JobTitle = item.JobTitle,
                    Adjust = item.Adjust,
                    Workload = item.Workload

                };
                decimal? baiscnorm = 1;
                var perforTypeItem = perforTypeArray.FirstOrDefault(t => t.Description == item.FitPeople);
                if (perforTypeItem != null)
                {
                    var perforType = (PerforType)perforTypeItem.Value;
                    baiscnorm = baiscNormService.GetBaiscNorm(baiscnormList, perforType);

                    //年资系数
                    if (item.FitPeople == AccountUnitType.行政工勤.ToString() && item.WorkTime.HasValue && item.WorkTime.Value > new DateTime(1970, 1, 1))
                    {
                        var years = ((DateTime.Now.Year - item.WorkTime.Value.Year) * 12 + (DateTime.Now.Month - item.WorkTime.Value.Month)) / 12.0m;
                        var value = workyearList.FirstOrDefault(t => t.MinRange < years && years <= t.MaxRange)?.Value;
                        compute.WorkYear = value;
                    }
                }
                //添加参数计算
                compute.BaiscNormValue = baiscnorm * (item.FitPeopleRatio ?? 1);
                //应发绩效
                compute.GiveFee = compute.BaiscNormValue * (compute.PostCoefficient + (item.Adjust ?? 0))
                   * (item.Grant ?? 1) * compute.Attendance * compute.ScoreAverageRate * (compute.WorkYear ?? 1)
                   + (compute.OtherPerfor ?? 0) + (item.Workload ?? 0) + (compute.Punishment ?? 0);
                //实发绩效
                compute.RealGiveFee = compute.GiveFee;
                computeList.Add(compute);
            }

            return computeList;
        }
    }


}
