﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using Masuit.Tools;
using Masuit.Tools.Systems;
using Microsoft.Extensions.Configuration;
using OfficeOpenXml;
using OfficeOpenXml.Style;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Services.ExtractExcelService.SheetDataWrite;

namespace Performance.Services
{
    public partial class ComputeService : IAutoInjection
    {
        /// <summary>
        /// 查询科室详情，返回统一格式
        /// </summary>
        /// <param name="allotId"></param>
        /// <param name="unitType"></param>
        /// <param name="accountingUnit"></param>
        /// <returns></returns>
        /// <exception cref="NotImplementedException"></exception>
        public UniteDeptDetailResponse UniteDeptDetail(int allotId, UnitType unitType, string accountingUnit, int userId)
        {
            var allot = perforPerallotRepository.GetEntity(t => t.ID == allotId);
            if (allot == null) throw new PerformanceException("绩效月信息错误");
            //当角色对应时过滤
            var userInfo = _userRepository.GetUser(userId);

            var account = perforResaccountRepository.GetEntity(t => t.AllotID == allotId && t.UnitType == (int)unitType && t.AccountingUnit == accountingUnit);
            account ??= new res_account
            {
                AllotID = allotId,
                UnitType = (int)unitType,
                AccountingUnit = accountingUnit,
                Number = 0,
                BasicFactor = 0,
                Extra = 0,
                AssessBeforeOtherFee = 0,
                MedicineExtra = 0,
                MaterialsExtra = 0,
                ScoringAverage = 0,
                AssessLaterOtherFee = 0,
                AdjustFactor = 0,
                AdjustLaterOtherFee = 0,
                RealGiveFee = 0,
            };

            var ignore = _configuration.GetSection("UniteDeptDetailIgnore").Get<string[]>() ?? Array.Empty<string>();
            var persheet = _perforPerSheetRepository.GetEntities(t => t.AllotID == allotId) ?? new List<per_sheet>();
            var headers = _perforImheaderRepository.GetEntities(t => t.AllotID == allotId) ?? new List<im_header>();
            var basicData = _perforImDataRepository.GetEntities(t => t.AllotID == allotId && t.AccountingUnit == accountingUnit) ?? new List<im_data>();

            var response = new UniteDeptDetailResponse
            {
                Title = $"{allot.Name ?? $"{allot.Year}年{allot.Month}月"} {unitType} {accountingUnit}",
                UnitType = unitType.ToString(),
                AccountingUnit = accountingUnit,
                DetailItems = new List<DeptDetailItem>()
            };
            // 行政工勤
            if (UnitTypeUtil.IsOffice(response.UnitType))
            {
                var detailItems = new List<DeptDetailItem>();
                #region 补全行政工勤人员明细
                {
                    var items = new List<Dictionary<string, object>>();
                    var computes = _perforRescomputeRepository.GetEntities(t => t.AllotID == allotId && t.AccountType == AccountUnitType.行政工勤.ToString() && t.AccountingUnit == accountingUnit) ?? new List<res_compute>();
                    foreach (var item in computes)
                    {
                        items.Add(new Dictionary<string, object>
                        {
                            { "工号", item.JobNumber },
                            { "姓名", item.EmployeeName },
                            { "系数", ValueFormating(item.PostCoefficient,"") },
                            { "出勤", ValueFormating(item.Attendance, UniteDept.Format.两位小数百分比.ToString()) },
                            { "其他绩效", ValueFormating(item.OtherPerfor,"") },
                            { "绩效合计",  ValueFormating(item.PerforTotal,UniteDept.Format.两位小数.ToString()) },
                        });
                    }
                    detailItems.Add(new DeptDetailItem { Title = "行政工勤", Total = account.PerforTotal ?? 0, TotalFormat = ValueFormating(account.PerforTotal ?? 0, ""), Items = items });
                }
                #endregion

                #region 补全6.11个人岗位系数
                {
                    var items = new List<Dictionary<string, object>>();
                    var postSheet = persheet.FirstOrDefault(t => t.SheetType == (int)SheetType.PersonPostCoefficient);
                    if (postSheet != null)
                    {
                        var postDatas = basicData?.Where(w => w.SheetID == postSheet.ID && w.AccountingUnit == account?.AccountingUnit);
                        foreach (var post in postDatas?.GroupBy(w => new { w.JobNumber }))
                        {
                            var row = new Dictionary<string, object>
                            {
                                { "工号", post.Key.JobNumber },
                                { "姓名", post.FirstOrDefault()?.EmployeeName },
                            };
                            foreach (var item in post.Where(w => w.IsTotal != 1))
                            {
                                var val = item.IsFactor == 1 ? (item.CellValue * (item.FactorValue ?? 0)) : item.CellValue;
                                row.AddOrUpdate(item.TypeName, ValueFormating(val, ""), ValueFormating(val, ""));
                            }
                            var amount = post.FirstOrDefault(w => w.IsTotal == 1)?.CellValue ?? 0;
                            row.AddOrUpdate("合计", ValueFormating(amount, ""), ValueFormating(amount, ""));
                            items.Add(row);
                        }
                        var totalValue = basicData.Where(t => t.SheetID == postSheet.ID && t.UnitType == (int)SheetType.PersonPostCoefficient && t.IsTotal == 1)?.Sum(t => t.CellValue) ?? 0;
                        detailItems.Add(new DeptDetailItem { Title = "岗位系数", Total = totalValue, Items = items });
                    }
                }
                #endregion

                #region 特殊处理
                var total = detailItems.FirstOrDefault(w => w.Title == "行政工勤")?.Total ?? 0;

                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.业绩分值.ToString(), Total = 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.在册人数.ToString(), Total = account.PermanentStaff ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.DRG绩效.ToString(), Total = account.Drg ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.业绩绩效.ToString(), Total = total, Children = detailItems });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.工作量绩效.ToString(), Total = 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核前绩效.ToString(), Total = total, StandOut = true });

                // 只显示合计没有详情
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核后绩效.ToString(), Total = account.AssessLaterPerforTotal ?? 0, StandOut = true });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核后其他绩效.ToString(), Total = account.AssessLaterOtherFee ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.调节系数.ToString(), Total = account.AdjustFactor ?? 1 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.实发绩效.ToString(), Total = account.RealGiveFee ?? 0, StandOut = true });
                #endregion
            }
            else if (UnitTypeUtil.Is(response.UnitType, UnitType.特殊核算组.ToString()))
            {
                var special = _perforResspecialunitRepository.GetEntities(t => t.AllotID == allotId && t.Department == accountingUnit);
                var imAccountbasics = _perforImaccountbasicRepository.GetEntities(t => t.AllotID == allotId && t.DoctorAccountingUnit == accountingUnit);
                if (special?.Any() != true) return response;
                account.AdjustLaterOtherFee = special.FirstOrDefault()?.AdjustLaterOtherFee ?? 0;
                account.AssessLaterOtherFee = special.FirstOrDefault()?.AssessLaterOtherFee ?? 0;
                account.AssessBeforeOtherFee = special.FirstOrDefault()?.AssessBeforeOtherFee ?? 0;
                account.MedicineExtra = special.FirstOrDefault()?.MedicineExtra ?? 0;
                account.MaterialsExtra = special.FirstOrDefault()?.MaterialsExtra ?? 0;
                account.ScoringAverage = special.FirstOrDefault()?.ScoringAverage ?? 0;
                account.Extra = imAccountbasics.FirstOrDefault()?.DoctorExtra ?? 0;

                var detailItems = new DeptDetailItem();
                #region 补全特殊核算组量化指标明细
                {
                    var items = new List<Dictionary<string, object>>();
                    foreach (var item in special)
                    {
                        items.Add(new Dictionary<string, object>
                        {
                            { "项目", item.QuantitativeIndicators },
                            { "数量", ValueFormating(item.Quantity, "")  },
                            { "系数", ValueFormating(item.QuantitativeIndicatorsValue, "") },
                            { "金额", ValueFormating(item.QuantitativeFee, UniteDept.Format.两位小数.ToString()) },
                        });
                    }

                    detailItems.Total = special.First().ResultsTotalFee ?? 0;
                    detailItems.TotalFormat = ValueFormating(special.First().ResultsTotalFee ?? 0, "");
                    detailItems.Items = items;
                }
                #endregion

                #region 特殊处理 
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.业绩分值.ToString(), Total = 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.在册人数.ToString(), Total = special.First().PermanentStaff ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.DRG绩效.ToString(), Total = special.First().Drg ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.业绩绩效.ToString(), Total = 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.工作量绩效.ToString(), Total = detailItems.Total, TotalFormat = detailItems.TotalFormat, Items = detailItems.Items });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核前绩效.ToString(), Total = special.First().PerforTotal ?? 0, StandOut = true });

                // 只显示合计没有详情
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核后绩效.ToString(), Total = special.First().GiveFee ?? 0, StandOut = true });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核后其他绩效.ToString(), Total = special.First().AssessLaterOtherFee ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.调节系数.ToString(), Total = special.First().Adjust ?? 1 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.实发绩效.ToString(), Total = special.First().RealGiveFee ?? 0, StandOut = true });
                #endregion
            }
            else
            {
                var detailItems = new List<DeptDetailItem>();
                #region 补全临床科室收入明细
                {
                    var leftSheetTypes = new List<int> { (int)SheetType.Income, (int)SheetType.OtherIncome, (int)SheetType.Expend, };
                    foreach (var sheetType in leftSheetTypes)
                    {
                        foreach (var sheet in persheet.Where(t => t.SheetType == sheetType).OrderBy(w => w.SheetName))
                        {
                            var type = TypeConversion(account.UnitType);
                            var sheetName = BeautifulName(sheet.SheetName);


                            var items = new List<Dictionary<string, object>>();
                            var postDatas = basicData.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1 && t.UnitType == (int)type);
                            if (postDatas?.Any() != true) continue;
                            var headersDatas = headers.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1).ToList();
                            postDatas = from temp1 in postDatas
                                        join temp2 in headersDatas on temp1.TypeName equals temp2.CellValue
                                        orderby temp2.PointCell ascending
                                        select temp1;

                            foreach (var post in postDatas.GroupBy(t => new { t.TypeName }))
                            {
                                if (ignore.Contains(post.Key.TypeName)) continue;
                                var lhfz = post.Sum(group => group.CellValue);
                                var hsfz = post.Sum(group => group.IsFactor == 1 ? (group.CellValue * (group.FactorValue ?? 0)) : group.CellValue);
                                var row = new Dictionary<string, object>
                                {
                                    { "量化指标", post.Key.TypeName },
                                    { "量化分值", ValueFormating(lhfz, "") },
                                    { "核算比例", post.FirstOrDefault().FactorValue},
                                    { "核算分值", ValueFormating(hsfz, "") },
                                };
                                if (post.FirstOrDefault().FactorValue != null)
                                {
                                    items.Add(row);
                                }
                            }
                            #region 添加合计
                            {
                                var lhfz = postDatas.Sum(w => w.CellValue);
                                var hsfz = postDatas.Sum(w => w.IsFactor == 1 ? (w.CellValue * (w.FactorValue ?? 0)) : w.CellValue);
                                items.Add(new Dictionary<string, object>
                                {
                                    {  "量化指标", "原始分值合计"},
                                    {  "量化分值", ValueFormating(lhfz, "") },
                                    {  "核算比例", "核算分值合计"},
                                    {  "核算分值", ValueFormating(hsfz, "") },
                                });
                            }
                            #endregion

                            var amount = basicData.Where(t => t.SheetID == sheet.ID && t.UnitType == (int)type && t.IsTotal != 1)?.Sum(t => t.IsFactor == 1 ? t.CellValue * (t.FactorValue ?? 0) : t.CellValue) ?? 0;
                            detailItems.Add(new DeptDetailItem { Title = sheetName, Total = amount, TotalFormat = ValueFormating(amount, ""), Items = items });
                        }
                    }
                }
                #endregion

                var detailItemWorkloads = new DeptDetailItem { };
                #region 补全临床科室工作量明细
                {
                    foreach (var sheet in persheet.Where(t => t.SheetType == (int)SheetType.Workload))
                    {
                        if ((sheet.SheetName.Contains("医生") && account.UnitType == (int)UnitType.护理组) || (sheet.SheetName.Contains("护理") && account.UnitType != (int)UnitType.护理组))
                            continue;

                        var type = TypeConversion(account.UnitType);

                        var workitems = cofworkitemRepository.GetEntities(t => t.AllotID == allotId);
                        var baseMedicineFactor = GetFactors(persheet, basicData, type, SheetType.WorkloadMedicineProp);
                        var baseCmiFactor = GetFactors(persheet, basicData, type, SheetType.WorkloadCMI);
                        var baseInclineFactor = GetFactors(persheet, basicData, type, SheetType.WorkloadIncline);

                        var items = new List<Dictionary<string, object>>();

                        var unitTypes = new int[] { (int)UnitType.医生组, (int)UnitType.医技组, (int)UnitType.其他医技组 };
                        var postDatas = (unitTypes.Contains((int)type))
                            ? basicData.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1 && unitTypes.Contains(t.UnitType.Value))
                            : basicData.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1 && t.UnitType == (int)type);
                        if (postDatas?.Any() != true) continue;
                        var headersDatas = headers.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1).ToList();
                        postDatas = from temp1 in postDatas
                                    join temp2 in headersDatas on temp1.TypeName equals temp2.CellValue
                                    orderby temp2.PointCell ascending
                                    select temp1;
                        foreach (var post in postDatas.GroupBy(t => new { t.TypeName }))
                        {
                            if (ignore.Contains(post.Key.TypeName)) continue;
                            var row = new Dictionary<string, object>
                            {
                                { "项目", post.Key.TypeName },
                                { "数量", ValueFormating(post.Sum(group => group.CellValue), "") },
                                { "系数", ValueFormating(post.FirstOrDefault().FactorValue, "") },
                                { "药占比系数", "" },
                                { "CMI系数", "" },
                                { "工作量倾斜", "" },
                                { "金额", "" }
                            };
                            decimal medicineFactor = 1m, cmiFactor = 1m, inclineFactor = 1m;

                            if (workitems != null && workitems.Any(a => a.Type == 1 && a.Item == post.Key.TypeName) && baseMedicineFactor != null)
                            {
                                medicineFactor = baseMedicineFactor.Value;
                                row.AddOrUpdate("药占比系数", ValueFormating(medicineFactor, ""));
                            }
                            if (workitems != null && workitems.Any(a => a.Type == 2 && a.Item == post.Key.TypeName) && baseCmiFactor != null)
                            {
                                cmiFactor = baseCmiFactor.Value;
                                row.AddOrUpdate("CMI系数", ValueFormating(cmiFactor, ""));
                            }
                            if (workitems != null && workitems.Any(a => a.Type == 3 && a.Item == post.Key.TypeName) && baseInclineFactor != null)
                            {
                                inclineFactor = baseInclineFactor.Value;
                                row.AddOrUpdate("工作量倾斜", ValueFormating(inclineFactor, ""));
                            }
                            var itemValue = post.Sum(group => group.IsFactor == 1 ? (group.CellValue * (group.FactorValue ?? 0)) : group.CellValue);
                            itemValue = itemValue * medicineFactor * cmiFactor * inclineFactor;
                            row.AddOrUpdate("金额", ValueFormating(itemValue, ""));


                            items.Add(row);
                        }
                        items.Add(new Dictionary<string, object>
                        {
                            { "项目", "核算分值合计"},
                            { "数量", "" },
                            { "系数", "" },
                            { "药占比系数", "" },
                            { "CMI系数", "" },
                            { "工作量倾斜", "" },
                            { "金额", ValueFormating(account.WorkloadFee,"") },
                        });
                        DeleteBlankColumn(items);

                        //var amount = basicData.Where(t => t.SheetID == sheet.ID && t.UnitType == (int)type && t.IsTotal != 1)?.Sum(t => t.IsFactor == 1 ? t.CellValue * (t.FactorValue ?? 0) : t.CellValue) ?? 0;
                        detailItemWorkloads.Total = account.WorkloadFee ?? 0m;
                        detailItemWorkloads.TotalFormat = ValueFormating(account.WorkloadFee, "");
                        detailItemWorkloads.Items = items;
                    }
                }
                #endregion

                #region 特殊处理
                string remark = "";
                if (allot.ShowFormula == 1)
                {
                    var sr = string.Join(" + ", persheet.Where(t => t.SheetType == (int)SheetType.Income).Select(w => BeautifulName(w.SheetName)));
                    var ewsr = string.Join(" + ", persheet.Where(t => t.SheetType == (int)SheetType.OtherIncome).Select(w => BeautifulName(w.SheetName)));
                    var zc = string.Join(" - ", persheet.Where(t => t.SheetType == (int)SheetType.Expend).Select(w => BeautifulName(w.SheetName)));
                    remark = $"{sr} + {ewsr} - {zc}";
                }
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.业绩分值.ToString(), Total = account.Income ?? 0, Children = detailItems });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.在册人数.ToString(), Total = account.PermanentStaff ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.DRG绩效.ToString(), Total = account.Drg ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.业绩绩效.ToString(), Total = account.PerforFee ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.工作量绩效.ToString(), Total = detailItemWorkloads.Total, TotalFormat = detailItemWorkloads.TotalFormat, Items = detailItemWorkloads.Items });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核前绩效.ToString(), Total = account.PerforTotal ?? 0, StandOut = true });

                // 只显示合计没有详情
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核后绩效.ToString(), Total = account.AssessLaterPerforTotal ?? 0, StandOut = true });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.考核后其他绩效.ToString(), Total = account.AssessLaterOtherFee ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.调节系数.ToString(), Total = account.AdjustFactor ?? 1 });
                response.DetailItems.Add(new DeptDetailItem { Title = UniteDept.Title.实发绩效.ToString(), Total = account.RealGiveFee ?? 0, StandOut = true });
                #endregion
            }

            // 加载用户设置
            var settings = GetUniteDeptDetailSetting(allotId);
            // 显示详情 
            #region 补全公共附属表明细
            {
                var rightSheetTypeMaps = new DeptDetailItem[]
                {
                    new DeptDetailItem { Title = UniteDept.Title.预算比例.ToString(), SheetType = SheetType.BudgetRatio, Total = account.BasicFactor ?? 0, },
                    new DeptDetailItem { Title = UniteDept.Title.医院奖罚.ToString(), SheetType = SheetType.AccountExtra, Total = account.Extra ?? 0, },
                    new DeptDetailItem { Title = UniteDept.Title.考核前其他绩效.ToString(), SheetType = SheetType.AssessBeforeOtherFee, Total = account.AssessBeforeOtherFee ?? 0, },
                    new DeptDetailItem { Title = UniteDept.Title.药占比奖罚.ToString(), SheetType = SheetType.AccountDrugAssess, Total = account.MedicineExtra ?? 0, },
                    new DeptDetailItem { Title = UniteDept.Title.材料占比奖罚.ToString(), SheetType = SheetType.AccountMaterialsAssess, Total = account.MaterialsExtra ?? 0, },
                    new DeptDetailItem { Title = UniteDept.Title.考核得分率.ToString(), SheetType = SheetType.AccountScoreAverage, Total = account.ScoringAverage ?? 0, },
                    new DeptDetailItem { Title = UniteDept.Title.调节后其他绩效.ToString(), SheetType = SheetType.AccountAdjustLaterOtherFee, Total = account.AdjustLaterOtherFee ?? 0, },
                };
                foreach (var detail in rightSheetTypeMaps)
                {
                    response.DetailItems.Add(detail);

                    foreach (var sheet in persheet.Where(t => t.SheetType == (int)detail.SheetType).Select(s => s.ID))
                    {
                        var type = TypeConversion(account.UnitType);
                        var items = new List<Dictionary<string, object>>();
                        var postDatas = basicData.Where(t => t.SheetID == sheet && t.IsTotal != 1 && t.UnitType == (int)type);

                        var headersDatas = headers.Where(t => t.SheetID == sheet && t.IsTotal != 1).ToList();
                        postDatas = from temp1 in postDatas
                                    join temp2 in headersDatas on temp1.TypeName equals temp2.CellValue
                                    orderby temp2.PointCell ascending
                                    select temp1;
                        foreach (var post in postDatas.GroupBy(t => new { t.TypeName }))
                        {
                            if (ignore.Contains(post.Key.TypeName)) continue;

                            var settingItem = settings.FirstOrDefault(w => w.Title == detail.Title) ?? new UniteDeptDetailItem() { Children = new List<UniteDeptDetailItem>() };
                            var gs = settingItem.Children.FirstOrDefault(w => w.Title == post.Key.TypeName) ?? new UniteDeptDetailItem { Format = "" };
                            if (string.IsNullOrEmpty(gs.Display))
                                gs.Display = UniteDept.DisplayRule.始终显示.ToString();

                            if (gs.Display == UniteDept.DisplayRule.始终隐藏.ToString()) continue;

                            var states = new int[] { (int)AllotStates.绩效下发, (int)AllotStates.归档 };
                            if (gs.Display == UniteDept.DisplayRule.下发显示.ToString() && userInfo.IsSecondAdmin && !states.Contains(allot.States)) continue;

                            var fz = post.Sum(group => group.IsFactor == 1 ? (group.CellValue * (group.FactorValue ?? 0)) : group.CellValue);
                            var row = new Dictionary<string, object>
                            {
                                { "指标", post.Key.TypeName },
                                { "分值", ValueFormating(fz, gs.Format) },
                                { "说明", gs.Remark },
                            };
                            items.Add(row);
                        }
                        DeleteBlankColumn(items);
                        //var amount = basicData?.Where(t => t.SheetID == sheet.ID && t.UnitType == (int)type && t.IsTotal == 1)?.FirstOrDefault()?.CellValue ?? 0;
                        //detail.Total = amount;
                        detail.Items = items;
                    }
                }
            }
            #endregion


            // 应用用户设置的显示规则
            RecursionUseSettings(response.DetailItems, settings, userInfo.IsSecondAdmin, allot);

            response.DetailItems = response.DetailItems.Where(w => w.Display).OrderBy(w => w.Sort).ToList();
            RecursiveSorts(response.DetailItems);
            return response;
        }
        /// <summary>
        /// 详情结果（排序）
        /// </summary>
        /// <param name="detailItems"></param>
        private void RecursiveSorts(List<DeptDetailItem> detailItems)
        {
            foreach (var item in detailItems)
            {
                if (item.Children?.Any() == true)
                {
                    item.Children = item.Children.OrderBy(w => w.Sort).ThenBy(x => x.Sort).ToList();
                    RecursiveSorts(item.Children);
                }
            }
        }

        #region 科室详情辅助方法

        private void RecursionUseSettings(List<DeptDetailItem> detailItems, List<UniteDeptDetailItem> userDetailItems, bool isSecondAdmin, per_allot allot)
        {
            detailItems ??= new List<DeptDetailItem>();
            userDetailItems ??= new List<UniteDeptDetailItem>();

            foreach (var detail in detailItems)
            {
                detail.Display = true;

                var cfg = userDetailItems.Find(w => w.Title == detail.Title);
                // 部分左右侧显示全部格式化
                if (detail?.Total == 0)
                {
                    detail.TotalFormat = detail.TotalFormat == null ? "0" : detail.TotalFormat;
                }
                else
                {
                    detail.TotalFormat = ValueFormating(detail.Total, cfg?.Format);
                }

                // 只对右侧设置排序
                detail.Sort = cfg == null ? detail.Sort : cfg.Sort;
                // 备注
                detail.Remark = allot.ShowFormula == 1 ? cfg?.Remark : null;
                // 只对科室查看时右侧并且未下发隐藏内容
                if (cfg?.Display == UniteDept.DisplayRule.始终隐藏.ToString())
                {
                    detail.Display = false;
                }
                else if (cfg?.Display == UniteDept.DisplayRule.下发显示.ToString())
                {
                    if (isSecondAdmin && allot.States != (int)AllotStates.绩效下发 && allot.States != (int)AllotStates.归档)
                    {
                        detail.Display = false;
                    }
                }
                else if (cfg?.Display == UniteDept.DisplayRule.总额下发后显示.ToString())
                {
                    if (isSecondAdmin && allot.States != (int)AllotStates.绩效下发 && allot.States != (int)AllotStates.归档)
                    {
                        detail.TotalFormat = null;
                    }
                }
                detail.Title = string.IsNullOrEmpty(cfg?.Alias?.Trim()) ? detail.Title : cfg?.Alias?.Trim();
                // Item里面需要隐藏和格式化的操作
                if (detail.Children?.Any() != true && detail.Items?.Any() == true && cfg?.Children?.Any() == true)
                {
                    var keySort = cfg.Children.Select(w => w.Title).ToList();

                    List<Dictionary<string, object>> sortedItems = new List<Dictionary<string, object>>();
                    detail.Items.ForEach(item =>
                    {
                        var temp = item.OrderBy(row => keySort.IndexOf(row.Key)).ToDictionary(t => t.Key, t => t.Value);
                        sortedItems.Add(temp);
                    });

                    // 更新 Items 列表
                    detail.Items = sortedItems;

                    foreach (var item in cfg.Children)
                    {
                        foreach (var row in detail.Items)
                        {
                            if (!string.IsNullOrEmpty(item.Format))
                            {
                                if (row.ContainsKey(item.Title) && decimal.TryParse(row[item.Title]?.ToString() ?? "", out decimal value))
                                {
                                    row.AddOrUpdate(item.Title, ValueFormating(value, item.Format));
                                }
                            }
                            if (item.Display == UniteDept.DisplayRule.始终隐藏.ToString())
                            {
                                row.Remove(item.Title);
                            }
                            else if (item.Display == UniteDept.DisplayRule.下发显示.ToString())
                            {
                                if (isSecondAdmin && allot.States != (int)AllotStates.绩效下发 && allot.States != (int)AllotStates.归档)
                                {
                                    row.Remove(item.Title);
                                }
                            }
                            else if (item.Display == UniteDept.DisplayRule.总额下发后显示.ToString())
                            {
                                row.Remove(item.Title);
                            }
                        }
                    }
                }

                RecursionUseSettings(detail.Children, cfg?.Children, isSecondAdmin, allot);
            }
        }

        /// <summary>
        /// 删除空白列
        /// </summary>
        /// <param name="items"></param>
        private static void DeleteBlankColumn(List<Dictionary<string, object>> items)
        {
            // 保证Key数量一致
            var keys = items.SelectMany(dict => dict.Keys).Distinct();
            foreach (var k in keys)
            {
                var values = items.SelectMany(dict => dict.Where(kvp => kvp.Key == k).Select(kvp => kvp.Value));
                if (!values.Any(w => !string.IsNullOrEmpty(w?.ToString())))
                {
                    foreach (var item in items)
                    {
                        item.Remove(k);
                    }
                }
            }
        }

        // 详情页面数值格式化
        private string ValueFormating(decimal? value, string format)
        {
            var total = value ?? 0;
            if (total == 0)
            {
                return string.Format("{0:0.####}", total);
            }
            else
            {
                if (format == UniteDept.Format.整数百分比.ToString())
                {
                    return string.Format("{0:0%}", total);
                }
                else if (format == UniteDept.Format.一位小数百分比.ToString())
                {
                    return string.Format("{0:0.#%}", total);
                }
                else if (format == UniteDept.Format.两位小数百分比.ToString())
                {
                    return string.Format("{0:0.##%}", total);
                }
                else if (format == UniteDept.Format.三位小数百分比.ToString())
                {
                    return string.Format("{0:0.###%}", total);
                }
                else if (format == UniteDept.Format.四位小数百分比.ToString())
                {
                    return string.Format("{0:0.####%}", total);
                }
                else if (Math.Abs(total) < 1.5m)
                {
                    return string.Format("{0:0.####}", total);
                }
                else if (format == UniteDept.Format.整数.ToString())
                {
                    return Math.Round(total, 0, MidpointRounding.AwayFromZero).ToString();
                }
                else if (format == UniteDept.Format.一位小数.ToString())
                {
                    total = Math.Round(total, 1, MidpointRounding.AwayFromZero);
                    return string.Format("{0:0.####}", total);
                }
                else if (format == UniteDept.Format.两位小数.ToString())
                {
                    total = Math.Round(total, 2, MidpointRounding.AwayFromZero);
                    return string.Format("{0:0.####}", total);
                }
                else if (format == UniteDept.Format.三位小数.ToString())
                {
                    total = Math.Round(total, 3, MidpointRounding.AwayFromZero);
                    return string.Format("{0:0.####}", total);
                }
                else if (format == UniteDept.Format.四位小数.ToString())
                {
                    total = Math.Round(total, 4, MidpointRounding.AwayFromZero);
                    return string.Format("{0:0.####}", total);
                }
                return string.Format("{0:0.####}", total);
            }
        }
        private string BeautifulName(string sheetName)
        {
            sheetName = new Regex("[0-9]*").Replace(sheetName, "", 5).Replace(".", "").Replace(" ", "");
            sheetName = sheetName != "科室考核" ? sheetName.Replace("科室绩效", "").Replace("科室", "") : sheetName;
            return sheetName;
        }
        #endregion

        /// <summary>
        /// 查询显示设置内容
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        /// <exception cref="PerformanceException"></exception>
        public List<UniteDeptDetailItem> GetUniteDeptDetailSetting(int allotId)
        {
            var allot = perforPerallotRepository.GetEntity(t => t.ID == allotId);
            if (allot == null) throw new PerformanceException("绩效月信息错误");

            var ignore = _configuration.GetSection("UniteDeptDetailIgnore").Get<string[]>() ?? Array.Empty<string>();
            var persheet = _perforPerSheetRepository.GetEntities(t => t.AllotID == allotId) ?? new List<per_sheet>();
            var basicData = _perforImDataRepository.GetEntities(t => t.AllotID == allotId) ?? new List<im_data>();

            var response = new List<UniteDeptDetailItem>() { };

            #region 临床科室收入
            List<UniteDeptDetailItem> lcUniteDepts = new List<UniteDeptDetailItem>();
            {
                var leftSheetTypes = new List<int> { (int)SheetType.Income, (int)SheetType.OtherIncome, (int)SheetType.Expend, };
                foreach (var sheetType in leftSheetTypes)
                {
                    foreach (var sheet in persheet.Where(t => t.SheetType == sheetType).OrderBy(w => w.SheetName))
                    {
                        var sheetName = BeautifulName(sheet.SheetName);
                        if (!lcUniteDepts.Any(w => w.Title == sheetName))
                        {
                            var lcyj = new UniteDeptDetailItem { Title = sheetName, Display = UniteDept.DisplayRule.始终显示.ToString(), Children = new List<UniteDeptDetailItem>() };

                            lcyj.Children.Add(new UniteDeptDetailItem { Title = "量化指标", IsAllowRemark = false, Format = UniteDept.Format.整数.ToString(), Display = UniteDept.DisplayRule.始终显示.ToString() });
                            lcyj.Children.Add(new UniteDeptDetailItem { Title = "量化分值", IsAllowRemark = false, Format = UniteDept.Format.整数.ToString(), Display = UniteDept.DisplayRule.始终显示.ToString() });
                            lcyj.Children.Add(new UniteDeptDetailItem { Title = "核算比例", IsAllowRemark = false, Format = UniteDept.Format.整数.ToString(), Display = UniteDept.DisplayRule.始终显示.ToString() });
                            lcyj.Children.Add(new UniteDeptDetailItem { Title = "核算分值", IsAllowRemark = false, Format = UniteDept.Format.整数.ToString(), Display = UniteDept.DisplayRule.始终显示.ToString() });
                            lcUniteDepts.Add(lcyj);
                        }
                    }
                }
            }
            #endregion 
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.业绩分值.ToString(), Children = lcUniteDepts });
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.在册人数.ToString() });
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.DRG绩效.ToString() });
            #region 行政工勤
            List<UniteDeptDetailItem> gqUniteDepts = new List<UniteDeptDetailItem>();
            {
                var gqyj = new UniteDeptDetailItem { Title = "行政工勤", Children = new List<UniteDeptDetailItem>() };

                gqyj.Children.Add(new UniteDeptDetailItem { Title = "工号", IsAllowRemark = false });
                gqyj.Children.Add(new UniteDeptDetailItem { Title = "姓名", IsAllowRemark = false });
                gqyj.Children.Add(new UniteDeptDetailItem { Title = "系数", IsAllowRemark = false });
                gqyj.Children.Add(new UniteDeptDetailItem { Title = "出勤", IsAllowRemark = false });
                gqyj.Children.Add(new UniteDeptDetailItem { Title = "其他绩效", IsAllowRemark = false });
                gqyj.Children.Add(new UniteDeptDetailItem { Title = "绩效合计", IsAllowRemark = false });

                gqUniteDepts.Add(gqyj);

                var gwxs = new UniteDeptDetailItem { Title = "岗位系数", Children = new List<UniteDeptDetailItem>() };

                gwxs.Children.Add(new UniteDeptDetailItem { Title = "工号", IsAllowRemark = false });
                gwxs.Children.Add(new UniteDeptDetailItem { Title = "姓名", IsAllowRemark = false });
                var postSheet = persheet.FirstOrDefault(t => t.SheetType == (int)SheetType.PersonPostCoefficient);
                if (postSheet != null)
                {
                    var typeNames = basicData?.Where(w => w.SheetID == postSheet.ID && w.IsTotal != 1).Select(w => w.TypeName).Distinct();
                    foreach (var item in typeNames)
                    {
                        gwxs.Children.Add(new UniteDeptDetailItem { Title = item, IsAllowRemark = false });
                    }
                }
                gwxs.Children.Add(new UniteDeptDetailItem { Title = "合计", IsAllowRemark = false });
                gqUniteDepts.Add(gwxs);
            }
            #endregion 
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.业绩绩效.ToString(), Remark = "业绩分值 * 预算比例", Children = gqUniteDepts });
            #region 工作量
            List<UniteDeptDetailItem> gzlUniteDepts = new List<UniteDeptDetailItem>();
            {
                gzlUniteDepts.Add(new UniteDeptDetailItem { Title = "项目", Display = UniteDept.DisplayRule.始终显示.ToString(), Format = UniteDept.Format.无.ToString(), IsAllowRemark = false });
                gzlUniteDepts.Add(new UniteDeptDetailItem { Title = "数量", Display = UniteDept.DisplayRule.始终显示.ToString(), IsAllowRemark = false });
                gzlUniteDepts.Add(new UniteDeptDetailItem { Title = "系数", Display = UniteDept.DisplayRule.始终显示.ToString(), Format = UniteDept.Format.两位小数.ToString(), IsAllowRemark = false });
                gzlUniteDepts.Add(new UniteDeptDetailItem { Title = "药占比系数", Display = UniteDept.DisplayRule.始终显示.ToString(), IsAllowRemark = false });
                gzlUniteDepts.Add(new UniteDeptDetailItem { Title = "CMI系数", Display = UniteDept.DisplayRule.始终显示.ToString(), IsAllowRemark = false });
                gzlUniteDepts.Add(new UniteDeptDetailItem { Title = "工作量倾斜", Display = UniteDept.DisplayRule.始终显示.ToString(), IsAllowRemark = false });
                gzlUniteDepts.Add(new UniteDeptDetailItem { Title = "金额", Display = UniteDept.DisplayRule.下发显示.ToString(), IsAllowRemark = false });
            }
            #endregion

            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.工作量绩效.ToString(), Children = gzlUniteDepts, Display = UniteDept.DisplayRule.始终隐藏.ToString() });
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.考核前绩效.ToString(), StandOut = true });
            // 只显示合计没有详情
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.考核后绩效.ToString(), StandOut = true, Remark = "考核前绩效 × 考核得分率 + 药占比奖罚 + 材料占比奖罚 + 医院奖罚 + 考核后其他绩效" });
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.考核后其他绩效.ToString(), Format = UniteDept.Format.整数.ToString() });
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.调节系数.ToString(), Format = UniteDept.Format.四位小数百分比.ToString() });
            response.Add(new UniteDeptDetailItem { Title = UniteDept.Title.实发绩效.ToString(), Display = UniteDept.DisplayRule.下发显示.ToString(), StandOut = true, Remark = "考核后绩效 × 调节系数 + 调节后其他绩效" });

            var entity = _perforCofDeptDetailRepository.GetEntity(w => w.AllotId == allotId);
            var cofDeptDetail = string.IsNullOrEmpty(entity?.Settings)
                ? new List<UniteDeptDetailItem>()
                : JsonHelper.Deserialize<List<UniteDeptDetailItem>>(entity.Settings);
            // 显示详情 
            #region 补全公共附属表明细  
            var rightSheetTypeMaps = new List<UniteDeptDetailItem>()
            {
                new UniteDeptDetailItem { Title = UniteDept.Title.预算比例.ToString(), SheetType = (int)SheetType.BudgetRatio, Format = UniteDept.Format.四位小数百分比.ToString() },
                new UniteDeptDetailItem { Title = UniteDept.Title.医院奖罚.ToString(), SheetType = (int)SheetType.AccountExtra},
                new UniteDeptDetailItem { Title = UniteDept.Title.考核前其他绩效.ToString(), SheetType = (int)SheetType.AssessBeforeOtherFee},
                new UniteDeptDetailItem { Title = UniteDept.Title.药占比奖罚.ToString(), SheetType = (int)SheetType.AccountDrugAssess},
                new UniteDeptDetailItem { Title = UniteDept.Title.材料占比奖罚.ToString(), SheetType = (int)SheetType.AccountMaterialsAssess},
                new UniteDeptDetailItem { Title = UniteDept.Title.考核得分率.ToString(), SheetType = (int)SheetType.AccountScoreAverage, Format = UniteDept.Format.四位小数百分比.ToString() },
                new UniteDeptDetailItem { Title = UniteDept.Title.调节后其他绩效.ToString(), SheetType = (int)SheetType.AccountAdjustLaterOtherFee},
            };
            foreach (var detail in rightSheetTypeMaps)
            {
                response.Add(detail);
                var items = new List<UniteDeptDetailItem>();
                var cofDisplay = UniteDept.DisplayRule.始终隐藏.ToString();
                if (cofDeptDetail.Any())
                {
                    cofDisplay = cofDeptDetail.FirstOrDefault(s => s.Title == detail.Title).Display ?? UniteDept.DisplayRule.始终隐藏.ToString();
                }

                foreach (var sheet in persheet.Where(t => t.SheetType == detail.SheetType))
                {
                    var postDatas = basicData.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1);

                    if (detail.Title == UniteDept.Title.预算比例.ToString() || detail.Title == UniteDept.Title.考核得分率.ToString() || detail.Title == UniteDept.Title.调节系数.ToString())
                    {
                        items.AddRange(postDatas.GroupBy(t => new { t.TypeName })
                                                .Where(post => !ignore.Contains(post.Key.TypeName))
                                                .Select(post => new UniteDeptDetailItem { Title = post.Key.TypeName, IsAllowRemark = false, Format = UniteDept.Format.四位小数百分比.ToString(), Display = cofDisplay }));
                    }
                    else
                    {
                        items.AddRange(postDatas.GroupBy(t => new { t.TypeName })
                                                .Where(post => !ignore.Contains(post.Key.TypeName))
                                                .Select(post => new UniteDeptDetailItem { Title = post.Key.TypeName, IsAllowRemark = false, Display = cofDisplay }));
                    }
                }
                detail.Children = items;

            }
            #endregion
            InitRecursion(response, null);
            #region 排序
            var enumItems = EnumHelper.GetItems<UniteDept.Title>();
            foreach (var detail in response)
            {
                var cfg = enumItems.FirstOrDefault(w => w.Name == detail.Title);
                detail.Sort = cfg == null ? detail.Sort : cfg.Value;
            }
            #endregion
            // 读取数据库保存内容
            GetRecursionUseSettings(response, cofDeptDetail);
            RecursiveSort(response);
            response = response.OrderBy(w => w.Sort).ToList();
            return response;
        }

        /// <summary>
        /// 配置结果（排序）
        /// </summary>
        /// <param name="detailItems"></param>
        private void RecursiveSort(List<UniteDeptDetailItem> detailItems)
        {
            foreach (var item in detailItems)
            {
                if (item.Children?.Any() == true)
                {
                    item.Children = item.Children.OrderBy(w => w.Sort).ThenBy(x => x.Sort).ToList();
                    RecursiveSort(item.Children);
                }
            }
        }

        #region 递归设置
        private void GetRecursionUseSettings(List<UniteDeptDetailItem> initDetailItems, List<UniteDeptDetailItem> userDetailItems)
        {
            initDetailItems ??= new List<UniteDeptDetailItem>();
            userDetailItems ??= new List<UniteDeptDetailItem>();

            foreach (var detail in initDetailItems)
            {
                var temp = userDetailItems.FirstOrDefault(w => w.Title == detail.Title);
                if (temp != null)
                {
                    detail.Alias = temp.Alias;
                    detail.Remark = temp.Remark;
                    detail.Format = temp.Format;
                    detail.Sort = temp.Sort;
                    detail.Display = temp.Display;
                    GetRecursionUseSettings(detail.Children, temp.Children);
                }
            }
            //foreach (var item in userDetailItems)
            //{
            //    if (!initDetailItems.Any(w => w.Title == item.Title))
            //        initDetailItems.Add(item);
            //}
        }

        /// <summary>
        /// 初始化显示配置信息
        /// </summary>
        /// <param name="detailItems"></param>
        /// <param name="parentId"></param>
        private void InitRecursion(List<UniteDeptDetailItem> detailItems, string parentId)
        {
            foreach (var detail in detailItems)
            {
                detail.Id = SnowFlake.GetInstance().GetLongId().ToString();
                detail.ParentId = parentId;
                detail.Format = detail.Format != null ? detail.Format : UniteDept.Format.整数.ToString();
                detail.Display = detail.Display != null ? detail.Display : UniteDept.DisplayRule.始终隐藏.ToString();
                detail.IsAllowAlias = string.IsNullOrEmpty(parentId) || detail.Children?.Any() == true;
                detail.Children ??= new List<UniteDeptDetailItem>();
                InitRecursion(detail.Children, detail.Id);
            }
        }
        #endregion
        /// <summary>
        /// 科室详情配置保存
        /// </summary>
        /// <param name="allotId"></param>
        /// <param name="request"></param>
        public void SaveUniteDeptDetailSetting(int allotId, List<UniteDeptDetailItem> request)
        {
            var entity = _perforCofDeptDetailRepository.GetEntity(w => w.AllotId == allotId);
            if (entity == null)
            {
                entity = new cof_dept_detail { AllotId = allotId, CreateTime = DateTime.Now, Settings = JsonHelper.Serialize(request) };
                _perforCofDeptDetailRepository.Add(entity);
            }
            else
            {
                entity.Settings = JsonHelper.Serialize(request);
                _perforCofDeptDetailRepository.Update(entity);
            }
        }

        #region 科室详情下载
        public string ExcelDownload(UniteDeptDetailResponse uniteDeptDetail, string name, int allotId)
        {
            var dpath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Files");
            if (!Directory.Exists(dpath)) Directory.CreateDirectory(dpath);

            string filepath = Path.Combine(dpath, $"{name}{DateTime.Now:yyyy年MM月dd日}");
            if (File.Exists(filepath)) File.Delete(filepath);

            using (FileStream fs = new FileStream(filepath, FileMode.OpenOrCreate))
            using (ExcelPackage package = new ExcelPackage(fs))
            {
                var worksheet = package.Workbook.Worksheets.Add(uniteDeptDetail.Title);
                worksheet.View.ShowGridLines = false;

                var maxColIndex = GetMaxColumnIndex(0, uniteDeptDetail.DetailItems);
                worksheet.SetValue(1, 1, uniteDeptDetail.Title);
                worksheet.Cells[1, 1, 1, maxColIndex].Merge = true;
                worksheet.Cells[1, 1, 1, maxColIndex].Style.Font.Bold = true;
                worksheet.Cells[1, 1, 1, maxColIndex].Style.Font.Size = 18;
                worksheet.Cells[1, 1, 1, maxColIndex].Style.Border.BorderAround(ExcelBorderStyle.Thin);
                worksheet.Cells[1, 1, 1, maxColIndex].Style.VerticalAlignment = ExcelVerticalAlignment.Center;
                worksheet.Cells[1, 1, 1, maxColIndex].Style.HorizontalAlignment = ExcelHorizontalAlignment.Center;
                worksheet.Cells[1, 1].Style.Fill.PatternType = ExcelFillStyle.Solid;
                worksheet.Cells[1, 1].Style.Fill.BackgroundColor.SetColor(1, 255, 255, 255);
                var rowIndex = 1;
                WriteSheetCells(worksheet, maxColIndex, rowIndex, uniteDeptDetail.DetailItems);
                worksheet.Cells.AutoFitColumns();
                int lastColumnIndex = worksheet.Dimension.End.Column;
                worksheet.Column(lastColumnIndex).Width = 20;
                package.Save();
            }
            return filepath;
        }

        private int WriteSheetCells(ExcelWorksheet worksheet, int maxColIndex, int rowIndex, List<DeptDetailItem> detailItems)
        {
            foreach (var item in detailItems.Where(w => w.Display))
            {
                ++rowIndex;
                worksheet.SetValue(rowIndex, 1, $"{item.Title} :  {item.TotalFormat}");
                if (!string.IsNullOrEmpty(item.Remark))
                {
                    worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].AddComment(item.Remark, "System");
                }
                worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Merge = true;
                worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Font.Bold = true;
                worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Font.Size = 14;
                worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Border.BorderAround(ExcelBorderStyle.Thin);
                worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.VerticalAlignment = ExcelVerticalAlignment.Center;
                worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.HorizontalAlignment = ExcelHorizontalAlignment.Left;
                worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Fill.PatternType = ExcelFillStyle.Solid;
                worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Fill.BackgroundColor.SetColor(1, 255, 255, 255);
                if (item.Children?.Any() == true)
                {
                    rowIndex = WriteSheetCells(worksheet, maxColIndex, rowIndex, item.Children);
                }
                else if (item.Items?.Any() == true)
                {
                    rowIndex = rowIndex + 1;
                    var headers = item.Items.FirstOrDefault().Keys.Select(w => w);
                    for (int colIndex = 0; colIndex < headers.Count(); colIndex++)
                    {
                        worksheet.SetValue(rowIndex, colIndex + 1, headers.ElementAt(colIndex));

                        worksheet.Cells[rowIndex, colIndex + 1].Style.Font.Bold = true;
                        worksheet.Cells[rowIndex, colIndex + 1].Style.Font.Size = 11;
                        worksheet.Cells[rowIndex, colIndex + 1].Style.Border.BorderAround(ExcelBorderStyle.Thin);
                        worksheet.Cells[rowIndex, colIndex + 1].Style.VerticalAlignment = ExcelVerticalAlignment.Center;
                        worksheet.Cells[rowIndex, colIndex + 1].Style.HorizontalAlignment = ExcelHorizontalAlignment.Center;
                        worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Fill.PatternType = ExcelFillStyle.Solid;
                        worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Fill.BackgroundColor.SetColor(1, 255, 255, 255);
                        if (colIndex + 1 == headers.Count())
                        {
                            worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Merge = true;

                            worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.Font.Bold = true;
                            worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.Font.Size = 11;
                            worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.Border.BorderAround(ExcelBorderStyle.Thin);
                            worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.VerticalAlignment = ExcelVerticalAlignment.Center;
                            worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.HorizontalAlignment = ExcelHorizontalAlignment.Center;
                            worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Fill.PatternType = ExcelFillStyle.Solid;
                            worksheet.Cells[rowIndex, 1, rowIndex, maxColIndex].Style.Fill.BackgroundColor.SetColor(1, 255, 255, 255);
                        }
                    }
                    foreach (var row in item.Items)
                    {
                        rowIndex = rowIndex + 1;
                        for (int colIndex = 0; colIndex < headers.Count(); colIndex++)
                        {
                            var cellValue = row[headers.ElementAt(colIndex)]?.ToString() ?? ""; // 获取单元格的值
                            double numericValue = 0;
                            if (cellValue.EndsWith("%") && double.TryParse(cellValue.Replace("%", ""), out numericValue))
                            {
                                worksheet.SetValue(rowIndex, colIndex + 1, numericValue / 100);
                                worksheet.Cells[rowIndex, colIndex + 1].Style.Numberformat.Format = "0.00%";
                            }
                            else if (double.TryParse(cellValue, out numericValue) && cellValue.Length != 18)
                            {
                                worksheet.SetValue(rowIndex, colIndex + 1, numericValue);
                                worksheet.Cells[rowIndex, colIndex + 1].Style.Numberformat.Format = "0.00";
                            }
                            else
                            {
                                worksheet.SetValue(rowIndex, colIndex + 1, cellValue);
                            }
                            worksheet.Cells[rowIndex, colIndex + 1].Style.Font.Size = 11;
                            worksheet.Cells[rowIndex, colIndex + 1].Style.Border.BorderAround(ExcelBorderStyle.Thin);
                            worksheet.Cells[rowIndex, colIndex + 1].Style.VerticalAlignment = ExcelVerticalAlignment.Center;
                            worksheet.Cells[rowIndex, colIndex + 1].Style.HorizontalAlignment = ExcelHorizontalAlignment.Center;
                            if (colIndex + 1 == headers.Count())
                            {
                                worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Merge = true;
                                worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.Font.Size = 11;
                                worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.Border.BorderAround(ExcelBorderStyle.Thin);
                                worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.VerticalAlignment = ExcelVerticalAlignment.Center;
                                worksheet.Cells[rowIndex, colIndex + 1, rowIndex, maxColIndex].Style.HorizontalAlignment = ExcelHorizontalAlignment.Center;
                            }
                        }
                    }
                }
            }
            return rowIndex;
        }

        private int GetMaxColumnIndex(int maxColIndex, List<DeptDetailItem> detailItems)
        {
            foreach (var item in detailItems)
            {
                if (item.Children?.Any() == true)
                {
                    var tempMaxColIndex = GetMaxColumnIndex(maxColIndex, item.Children);
                    if (tempMaxColIndex > maxColIndex)
                    {
                        maxColIndex = tempMaxColIndex;
                    }
                }
                else if (item.Items?.Any() == true)
                {
                    var tempMaxColIndex = item.Items.SelectMany(dict => dict.Keys).Distinct().Count();
                    if (tempMaxColIndex > maxColIndex)
                    {
                        maxColIndex = tempMaxColIndex;
                    }
                }
            }
            return maxColIndex;
        }


        #endregion

        #region 管理绩效详情显示隐藏设置--加载和保存
        public List<UniteDeptDetailItem> GetEmpDetailSetting(int allotId)
        {
            var allot = perforPerallotRepository.GetEntity(t => t.ID == allotId) ?? throw new PerformanceException("绩效月信息错误");

            var ignore = _configuration.GetSection("UniteDeptDetailIgnore").Get<string[]>() ?? Array.Empty<string>();
            var persheet = _perforPerSheetRepository.GetEntities(t => t.AllotID == allotId) ?? new List<per_sheet>();
            var basicData = _perforImDataRepository.GetEntities(t => t.AllotID == allotId) ?? new List<im_data>();

            var response = new List<UniteDeptDetailItem>() { };
            #region 行政工勤
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.考核前其他绩效.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.夜班费.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.考核后绩效.ToString() });
            #endregion
            #region 管理绩效
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.核算人数.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.人均绩效.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.实际人均.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.效率绩效人数.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.效率系数.ToString(), Format = UniteDept.Format.两位小数.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.效率绩效小计.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.规模绩效系数.ToString(), Format = UniteDept.Format.两位小数.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.规模绩效小计.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.管理绩效发放系数.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.考核前管理绩效.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.考核后管理绩效.ToString() });
            #endregion
            #region 共有
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.考核前绩效.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.出勤率.ToString(), Format = UniteDept.Format.整数百分比.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.考核得分率.ToString(), Format = UniteDept.Format.整数百分比.ToString() });
            //response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.医院奖罚.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.调节系数.ToString(), Format = UniteDept.Format.整数百分比.ToString() });
            response.Add(new UniteDeptDetailItem { Title = ManagementDept.TitleManage.绩效合计.ToString(), Display = UniteDept.DisplayRule.下发显示.ToString(), });
            #endregion
            #region 补全公共附属表明细  
            var rightSheetTypeMaps = new List<UniteDeptDetailItem>()
            {
                new UniteDeptDetailItem { Title = ManagementDept.TitleManage.基础绩效系数.ToString(), SheetType = (int)SheetType.PersonPostCoefficient},
                new UniteDeptDetailItem { Title = ManagementDept.TitleManage.医院奖罚.ToString(), SheetType = (int)SheetType.PersonExtra},
                new UniteDeptDetailItem { Title = ManagementDept.TitleManage.其他管理绩效.ToString(), SheetType = (int)SheetType.PersonOtherManagePerforFee},
                new UniteDeptDetailItem { Title = ManagementDept.TitleManage.调节后其他绩效.ToString(), SheetType = (int)SheetType.PersonAdjustLaterOtherFee},
                new UniteDeptDetailItem { Title = ManagementDept.TitleManage.调节后其他管理绩效.ToString(),SheetType=(int)SheetType.PersonAdjustLaterOtherManagePerforFee},
            };
            foreach (var detail in rightSheetTypeMaps)
            {
                response.Add(detail);
                var items = new List<UniteDeptDetailItem>();

                foreach (var sheet in persheet.Where(t => t.SheetType == detail.SheetType))
                {
                    var sheetName = BeautifulName(sheet.SheetName);
                    var postDatas = basicData.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1);

                    if (detail.Title == UniteDept.Title.预算比例.ToString() || detail.Title == UniteDept.Title.考核得分率.ToString() || detail.Title == UniteDept.Title.调节系数.ToString())
                    {
                        items.AddRange(postDatas.GroupBy(t => new { t.TypeName })
                                                .Where(post => !ignore.Contains(post.Key.TypeName))
                                                .Select(post => new UniteDeptDetailItem { Title = post.Key.TypeName, IsAllowRemark = false, Format = UniteDept.Format.四位小数百分比.ToString() }));
                    }
                    else
                    {
                        items.AddRange(postDatas.GroupBy(t => new { t.TypeName })
                                                .Where(post => !ignore.Contains(post.Key.TypeName))
                                                .Select(post => new UniteDeptDetailItem { Title = post.Key.TypeName, IsAllowRemark = false }));
                    }
                }
                detail.Children = items;

            }
            #endregion
            //初始化配置
            InitRecursion(response, null);
            #region 排序
            var enumItems = EnumHelper.GetItems<ManagementDept.TitleManage>();
            foreach (var detail in response)
            {
                var cfg = enumItems.FirstOrDefault(w => w.Name == detail.Title);
                detail.Sort = cfg == null ? detail.Sort : cfg.Value;
            }
            #endregion

            var entity = _perforCofEmpDetailRepository.GetEntity(w => w.AllotId == allotId);
            var cofDeptDetail = string.IsNullOrEmpty(entity?.Settings)
                ? new List<UniteDeptDetailItem>()
                : JsonHelper.Deserialize<List<UniteDeptDetailItem>>(entity.Settings);

            GetRecursionUseSettings(response, cofDeptDetail);
            RecursiveSort(response);
            response = response.OrderBy(w => w.Sort).ToList();
            return response;
        }

        public void SaveEmpDetailSetting(int allotId, List<UniteDeptDetailItem> request)
        {
            var entity = _perforCofEmpDetailRepository.GetEntity(w => w.AllotId == allotId);
            if (entity == null)
            {
                entity = new cof_emp_detail { AllotId = allotId, CreateTime = DateTime.Now, Settings = JsonHelper.Serialize(request) };
                _perforCofEmpDetailRepository.Add(entity);
            }
            else
            {
                entity.Settings = JsonHelper.Serialize(request);
                _perforCofEmpDetailRepository.Update(entity);
            }
        }
        /// <summary>
        /// 管理绩效详情
        /// </summary>
        /// <param name="request"></param>
        /// <param name="userid"></param>
        /// <returns></returns>
        /// <exception cref="NotImplementedException"></exception>
        /// <exception cref="PerformanceException"></exception>
        public List<UniteDeptDetailResponse> EmpDetail(EmpDetailRequest request, int userid)
        {
            //当角色对应时过滤
            var userInfo = _userRepository.GetUser(userid);
            if (userInfo?.User == null) throw new NotImplementedException("当前用户不存在");
            if (userInfo.URole == null) throw new NotImplementedException("当前用户暂未分配角色");
            if (!userInfo.Hospitals.NotNullOrEmpty()) throw new NotImplementedException("当前用户暂未分配医院");

            var responses = new List<UniteDeptDetailResponse>();
            var resComputes = new List<res_compute>();
            if (request.ComputeId > 0)
            {
                var temp = _perforRescomputeRepository.GetEntity(t => t.ID == request.ComputeId);
                if (temp == null) return new List<UniteDeptDetailResponse>();
                resComputes.Add(temp);
            }
            else
            {
                // 查询当前角色下科室的绩效
                UnitTypeUtil.Maps.TryGetValue(userInfo?.URole.Type ?? 0, out string[] unitTypes);
                if (unitTypes == null || !unitTypes.Any()) return new List<UniteDeptDetailResponse>();

                var accountingUnit = userInfo.User.Department;
                if (userInfo.IsSecondAdmin)
                {
                    var secondList = _agsecondallotRepository.Get(request.AllotId, unitTypes, userInfo.User.Department);
                    accountingUnit = secondList.FirstOrDefault()?.Department;
                    unitTypes = new string[] { secondList.FirstOrDefault()?.UnitType };
                }

                Dictionary<int, List<string>> dict = new Dictionary<int, List<string>>
            {
                { (int)Role.科主任, new List<string>{ AccountUnitType.科主任.ToString()} },
                { (int)Role.护士长, new List<string>{ AccountUnitType.护士长.ToString() } },
                { (int)Role.行政科室, new List<string> { AccountUnitType.行政中层.ToString() } },
                { (int)Role.特殊科室, new List<string> { AccountUnitType.科主任.ToString() , AccountUnitType.护士长.ToString() } },
            };

                int roleType = userInfo?.URole.Type ?? 0;
                var accountTypes = userInfo != null && dict.ContainsKey(roleType) ? dict[roleType] : new List<string>();

                var temp = _rescomputeRepository.GetEntities(t => t.AllotID == request.AllotId && t.AccountingUnit == accountingUnit && (unitTypes.Contains(t.UnitType) || accountTypes.Contains(t.AccountType)));
                if (temp == null) return new List<UniteDeptDetailResponse>();
                resComputes.AddRange(temp);
            }
            foreach (var resCompute in resComputes)
            {
                var allot = perforPerallotRepository.GetEntity(t => t.ID == resCompute.AllotID) ?? throw new PerformanceException("绩效月信息错误");
                if (!Enum.TryParse(resCompute.UnitType ?? resCompute.AccountType, ignoreCase: true, out UnitType unitType))
                    throw new PerformanceException("核算组别错误");


                var ignore = _configuration.GetSection("UniteDeptDetailIgnore").Get<string[]>() ?? Array.Empty<string>();
                var persheet = _perforPerSheetRepository.GetEntities(t => t.AllotID == resCompute.AllotID) ?? new List<per_sheet>();
                var headers = _perforImheaderRepository.GetEntities(t => t.AllotID == resCompute.AllotID) ?? new List<im_header>();
                var basicData = _perforImDataRepository.GetEntities(t => t.AllotID == resCompute.AllotID && t.AccountingUnit == resCompute.AccountingUnit && t.JobNumber == resCompute.JobNumber) ?? new List<im_data>();

                var employee = _perforImemployeeclinicRepository.GetEntity(t => t.AllotID == resCompute.AllotID && t.AccountingUnit == resCompute.AccountingUnit && t.UnitType == resCompute.UnitType && t.PersonnelNumber == resCompute.JobNumber);

                var response = new UniteDeptDetailResponse
                {
                    Title = $"{resCompute.EmployeeName}({resCompute.JobNumber}) {resCompute.JobTitle ?? ""} {resCompute.AccountingUnit}",
                    UnitType = resCompute?.UnitType,
                    AccountingUnit = resCompute?.AccountingUnit,
                    DetailItems = new List<DeptDetailItem>()
                };
                #region 行政工勤
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.考核前其他绩效.ToString(), Total = resCompute.AssessBeforeOtherFee ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.夜班费.ToString(), Total = resCompute.NightWorkPerfor ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.考核后绩效.ToString(), Total = resCompute.GiveFee ?? 0 });
                #endregion
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.考核前绩效.ToString(), Total = resCompute.PerforTotal ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.核算人数.ToString(), Total = resCompute.Number ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.人均绩效.ToString(), Total = (resCompute.Number ?? 0) == 0 ? 0 : resCompute.PerforTotal / resCompute.Number ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.出勤率.ToString(), Total = resCompute.Attendance ?? 1 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.实际人均.ToString(), Total = resCompute.Avg ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.效率绩效人数.ToString(), Total = resCompute.PermanentStaff ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.效率系数.ToString(), Total = employee?.Efficiency ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.效率绩效小计.ToString(), Total = resCompute.Efficiency ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.规模绩效系数.ToString(), Total = employee?.Scale ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.规模绩效小计.ToString(), Total = resCompute.Scale ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.管理绩效发放系数.ToString(), Total = employee?.Management ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.考核前管理绩效.ToString(), Total = resCompute.ShouldGiveFee ?? 0 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.考核得分率.ToString(), Total = resCompute.ScoreAverageRate ?? 1 });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.考核后管理绩效.ToString(), Total = GetDecimal(resCompute.ShouldGiveFee * resCompute.ScoreAverageRate * resCompute.Attendance + resCompute.Punishment ?? 0), Remark = resCompute.Remark });
                response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.调节系数.ToString(), Total = employee?.Adjust ?? 1 });

                var isShowManage = IsShowManage(resCompute.AllotID.Value);
                var array = new List<string>();
                if (resCompute.AccountType == AccountUnitType.行政中层.ToString())
                {
                    array = new List<string>()
                    {
                            ManagementDept.TitleManage.考核前其他绩效.ToString(),
                            ManagementDept.TitleManage.考核前绩效.ToString(),
                            ManagementDept.TitleManage.出勤率.ToString(),
                            ManagementDept.TitleManage.考核得分率.ToString(),
                            ManagementDept.TitleManage.医院奖罚.ToString(),
                            ManagementDept.TitleManage.夜班费.ToString(),
                            ManagementDept.TitleManage.考核后绩效.ToString(),
                            ManagementDept.TitleManage.调节系数.ToString(),
                            ManagementDept.TitleManage.调节后其他绩效.ToString(),
                            ManagementDept.TitleManage.绩效合计.ToString()
                    };
                    isShowManage = 1;
                }
                else
                {
                    array = new List<string>()
                    {
                                ManagementDept.TitleManage.考核前绩效.ToString(),
                                ManagementDept.TitleManage.核算人数.ToString(),
                                ManagementDept.TitleManage.人均绩效.ToString(),
                                ManagementDept.TitleManage.基础绩效系数.ToString(),
                                ManagementDept.TitleManage.出勤率.ToString(),
                                ManagementDept.TitleManage.实际人均.ToString(),
                                ManagementDept.TitleManage.效率绩效人数.ToString(),
                                ManagementDept.TitleManage.效率系数.ToString(),
                                ManagementDept.TitleManage.效率绩效小计.ToString(),
                                ManagementDept.TitleManage.规模绩效系数.ToString(),
                                ManagementDept.TitleManage.规模绩效小计.ToString(),
                                ManagementDept.TitleManage.管理绩效发放系数.ToString(),
                                ManagementDept.TitleManage.其他管理绩效.ToString(),
                                ManagementDept.TitleManage.考核前管理绩效.ToString(),
                                ManagementDept.TitleManage.考核得分率.ToString(),
                                ManagementDept.TitleManage.考核后管理绩效.ToString(),
                                ManagementDept.TitleManage.医院奖罚.ToString(),
                                ManagementDept.TitleManage.调节系数.ToString(),
                                ManagementDept.TitleManage.调节后其他绩效.ToString(),
                                ManagementDept.TitleManage.调节后其他管理绩效.ToString(),
                                ManagementDept.TitleManage.绩效合计.ToString()
                    };

                }

                // 开启 显示管理绩效
                if (isShowManage != 1)
                {
                    response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.绩效合计.ToString(), Total = GetDecimal(resCompute.ShouldGiveFee * resCompute.ScoreAverageRate * resCompute.Attendance + resCompute.Punishment ?? 0) * (resCompute.Adjust ?? 1) + (resCompute.AdjustLaterOtherFee ?? 0) });
                }
                else
                {
                    response.DetailItems.Add(new DeptDetailItem { Title = ManagementDept.TitleManage.绩效合计.ToString(), Total = resCompute.RealGiveFee ?? 0 });
                }

                var settings = GetEmpDetailSetting(resCompute.AllotID ?? 0);
                // 显示详情 
                #region 补全公共附属表明细
                {
                    var rightSheetTypeMaps = new DeptDetailItem[]
                    {
                        new DeptDetailItem { Title = ManagementDept.TitleManage.基础绩效系数.ToString(),SheetType=SheetType.PersonPostCoefficient, Total = employee?.Basics ?? 0 },
                        new DeptDetailItem { Title = ManagementDept.TitleManage.医院奖罚.ToString(),SheetType=SheetType.PersonExtra, Total = resCompute.Punishment ?? 0 },
                        new DeptDetailItem { Title = ManagementDept.TitleManage.其他管理绩效.ToString(),SheetType=SheetType.PersonOtherManagePerforFee, Total = resCompute.OtherManagePerfor ?? 0 },
                        new DeptDetailItem { Title = ManagementDept.TitleManage.调节后其他绩效.ToString(), SheetType = SheetType.PersonAdjustLaterOtherFee, Total = employee?.AdjustLaterOtherFee ?? 0 },
                        new DeptDetailItem { Title = ManagementDept.TitleManage.调节后其他管理绩效.ToString(),SheetType=SheetType.PersonAdjustLaterOtherManagePerforFee, Total = 0 },
                    };

                    foreach (var detail in rightSheetTypeMaps)
                    {
                        response.DetailItems.Add(detail);
                        foreach (var sheet in persheet.Where(t => t.SheetType == (int)detail.SheetType))
                        {
                            var type = TypeConversion((int)unitType);
                            var items = new List<Dictionary<string, object>>();
                            var postDatas = basicData.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1 && t.UnitType == (int)type);

                            var headersDatas = headers.Where(t => t.SheetID == sheet.ID && t.IsTotal != 1).ToList();
                            postDatas = from temp1 in postDatas
                                        join temp2 in headersDatas on temp1.TypeName equals temp2.CellValue
                                        orderby temp2.PointCell ascending
                                        select temp1;
                            foreach (var post in postDatas.GroupBy(t => new { t.TypeName }))
                            {
                                if (ignore.Contains(post.Key.TypeName)) continue;

                                var settingItem = settings.Find(w => w.Title == detail.Title) ?? new UniteDeptDetailItem() { Children = new List<UniteDeptDetailItem>() };
                                var gs = settingItem.Children.Find(w => w.Title == post.Key.TypeName) ?? new UniteDeptDetailItem { Format = "" };
                                if (string.IsNullOrEmpty(gs.Display))
                                    gs.Display = UniteDept.DisplayRule.始终显示.ToString();

                                if (gs.Display == UniteDept.DisplayRule.始终隐藏.ToString()) continue;

                                var states = new int[] { (int)AllotStates.绩效下发, (int)AllotStates.归档 };
                                if (gs.Display == UniteDept.DisplayRule.下发显示.ToString() && userInfo.IsSecondAdmin && !states.Contains(allot.States)) continue;

                                var fz = post.Sum(group => group.IsFactor == 1 ? (group.CellValue * (group.FactorValue ?? 0)) : group.CellValue);
                                var row = new Dictionary<string, object>
                                {
                                    { "指标", post.Key.TypeName },
                                    { "分值", ValueFormating(fz, gs.Format) },
                                    { "说明", gs.Remark },
                                };
                                items.Add(row);
                            }
                            DeleteBlankColumn(items);
                            //var amount = basicData?.Where(t => t.SheetID == sheet.ID && t.UnitType == (int)type && t.IsTotal == 1)?.FirstOrDefault()?.CellValue ?? 0;
                            //detail.Total = amount;
                            var total = items
                                .Where(w => w.ContainsKey("分值"))
                                .Select(w =>
                                {
                                    if (w["分值"] == null || !decimal.TryParse(w["分值"].ToString(), out var value))
                                    {
                                        return 0m;
                                    }
                                    return value;
                                })
                                .Sum();
                            detail.Items = items;
                            detail.Total = total;
                        }
                    }
                }
                #endregion
                //过滤
                response.DetailItems = response.DetailItems.Where(w => array.Contains(w.Title)).ToList();
                RecursionUseSettings(response.DetailItems, settings, userInfo.IsSecondAdmin, allot);

                response.DetailItems = response.DetailItems.Where(w => w.Display).OrderBy(w => w.Sort).ToList();
                responses.Add(response);
            }
            return responses;
        }


        public string ExcelDownload(List<UniteDeptDetailResponse> uniteDeptDetails, string name)
        {
            var dpath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Files");
            if (!Directory.Exists(dpath)) Directory.CreateDirectory(dpath);

            string filepath = Path.Combine(dpath, $"{name}{DateTime.Now:yyyy年MM月dd日}");
            if (File.Exists(filepath)) File.Delete(filepath);

            using (FileStream fs = new FileStream(filepath, FileMode.OpenOrCreate))
            using (ExcelPackage package = new ExcelPackage(fs))
            {
                foreach (var uniteDeptDetail in uniteDeptDetails)
                {
                    var worksheet = package.Workbook.Worksheets.Add(uniteDeptDetail.Title);
                    worksheet.View.ShowGridLines = false;
                    var maxColIndex = GetMaxColumnIndex(0, uniteDeptDetail.DetailItems);
                    maxColIndex = maxColIndex == 0 ? 1 : maxColIndex;
                    worksheet.SetValue(1, 1, uniteDeptDetail.Title);
                    worksheet.Cells[1, 1, 1, maxColIndex].Merge = true;
                    worksheet.Cells[1, 1, 1, maxColIndex].Style.Font.Bold = true;
                    worksheet.Cells[1, 1, 1, maxColIndex].Style.Font.Size = 18;
                    worksheet.Cells[1, 1, 1, maxColIndex].Style.Border.BorderAround(ExcelBorderStyle.Thin);
                    worksheet.Cells[1, 1, 1, maxColIndex].Style.VerticalAlignment = ExcelVerticalAlignment.Center;
                    worksheet.Cells[1, 1, 1, maxColIndex].Style.HorizontalAlignment = ExcelHorizontalAlignment.Center;
                    worksheet.Cells[1, 1].Style.Fill.PatternType = ExcelFillStyle.Solid;
                    worksheet.Cells[1, 1].Style.Fill.BackgroundColor.SetColor(1, 255, 255, 255);
                    var rowIndex = 1;
                    WriteSheetCells(worksheet, maxColIndex, rowIndex, uniteDeptDetail.DetailItems);
                    worksheet.Cells.AutoFitColumns();
                    int lastColumnIndex = worksheet.Dimension.End.Column;
                    worksheet.Column(lastColumnIndex).Width = 88;
                    for (int row = 1; row <= worksheet.Dimension.End.Row; row++)
                    {
                        worksheet.Row(row).Height = 38;
                    }
                }
                package.Save();
            }
            return filepath;
        }


        #endregion

    }
}
