﻿using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Mail;
using System.Text;
using System.Threading.Tasks;

namespace Performance.Infrastructure
{

    public class EmailOptions
    {
        /// <summary>
        /// smtp
        /// </summary>
        public string SmtpServer { get; set; }
        /// <summary>
        /// 发送邮箱
        /// </summary>
        public string Account { get; set; }
        /// <summary>
        /// 密码
        /// </summary>
        public string Password { get; set; }
        /// <summary>
        /// 是否启用
        /// </summary>
        public bool IsEnable { get; set; }
    }


    public class EmailMessage
    {
        /// <summary>
        /// 邮件显示名称
        /// </summary>
        public string DisplayName { get; set; }
        /// <summary>
        /// 邮件主题
        /// </summary>
        public string Subject { get; set; }
        /// <summary>
        /// 邮件内容
        /// </summary>
        public string Body { get; set; }
        /// <summary>
        /// 发送地址
        /// </summary>
        public List<string> To { get; set; } = new List<string>();
        /// <summary>
        /// 附件
        /// </summary>
        public List<string> Attachments { get; set; } = new List<string>();
    }

    public interface IEmailService
    {
        /// <summary>
        /// 
        /// </summary>
        /// <param name="message"></param>
        /// <returns></returns>
        bool Send(EmailMessage message);
        Task<bool> SendAsync(EmailMessage message);
    }

    public class EmailService : IEmailService
    {
        private EmailOptions options;
        private ILogger<EmailService> logger;
        public EmailService(EmailOptions options, ILogger<EmailService> logger)
        {
            this.options = options;
            this.logger = logger;
        }

        public Task<bool> SendAsync(EmailMessage message)
        {
            return Task.Run<bool>(() => { return Send(message); });
        }

        public bool Send(EmailMessage message)
        {
            if (!options.IsEnable)
                return true;

            MailMessage mail = new MailMessage();
            try
            {
                mail.From = new MailAddress(options.Account, message.DisplayName);
                //多邮箱地址处理 
                message.To.ForEach(t => mail.To.Add(t));

                mail.Subject = message.Subject;//主题
                mail.Body = message.Body;//内容
                mail.BodyEncoding = Encoding.UTF8;//正文编码
                mail.IsBodyHtml = true;//设置为HTML格式
                mail.Priority = MailPriority.Low;//优先级
                //发送附件 指明附件的绝对地址
                message.Attachments.ForEach(t =>
                {
                    if (FileHelper.IsExistFile(t))
                        mail.Attachments.Add(new Attachment(t));
                });
                SmtpClient client = new SmtpClient(options.SmtpServer, 80);
                client.Timeout = 10000;
                client.UseDefaultCredentials = true;
                client.Credentials = new NetworkCredential(options.Account, options.Password);//用户名和密码
                client.Send(mail);

                return true;
            }
            catch (SmtpException ex)
            {
                logger.LogError(ex.ToString());
                throw ex;
            }
            finally
            {
                if (message.Attachments.Any())
                    mail.Attachments.Dispose();
            }
        }

    }
}
