﻿using FluentValidation.AspNetCore;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Options;
using Newtonsoft.Json.Linq;
using Performance.DtoModels;
using Performance.DtoModels.AppSettings;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Performance.Api.Controllers
{
    [Route("api/[controller]")]
    public class AccountController : Controller
    {
        private UserService _userService;
        private RoleService _roleService;
        private IMemoryCache _memoryCache;
        private Application _options;
        private ClaimService _claim;
        private HospitalService _hospitalService;
        public AccountController(UserService userService,
            HospitalService hospitalService,
            RoleService roleService,
            IMemoryCache memoryCache,
            IOptions<Application> options,
            ClaimService claim)
        {
            _userService = userService;
            _roleService = roleService;
            _memoryCache = memoryCache;
            _hospitalService = hospitalService;
            _options = options.Value;
            _claim = claim;
        }

        /// <summary>
        /// 登录
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [NoVerify]
        [HttpPost]
        [Route("login")]
        public ApiResponse<UserIdentity> Login([FromBody]LoginRequest request)
        {
            var user = _userService.Login(request);
            if (user == null)
                return new ApiResponse<UserIdentity>(ResponseType.Fail, "用户不存在");
            int[] roleArray = new int[] { _options.NurseRole, _options.DirectorRole };

            user.Hospital = _hospitalService.GetUserHopital(user.UserID);
            user.Role = _roleService.GetUserRole(user.UserID);
            user.IsAgainAdmin = user.Role != null ? roleArray.Contains(user.Role.First().RoleID) : false;

            if (string.IsNullOrEmpty(user.Token))
                user.Token = Guid.NewGuid().ToString("N");

            var option = new MemoryCacheEntryOptions()
            {
                SlidingExpiration = TimeSpan.FromMinutes(_options.ExpirationMinutes)
            };
            _memoryCache.Set(user.Token, user, option);

            return new ApiResponse<UserIdentity>(ResponseType.OK, user);
        }

        /// <summary>
        /// 修改个人信息
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("updateself")]
        [HttpPost]
        public ApiResponse<UserResponse> UpdateSelf([CustomizeValidator(RuleSet = "Self"), FromBody]UserRequest request)
        {
            request.ID = _claim.At(request.Token).UserID;
            var user = _userService.UpdateSelf(request);
            return new ApiResponse<UserResponse>(ResponseType.OK, user);
        }

        /// <summary>
        /// 用户列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("list")]
        [HttpPost]
        public ApiResponse<List<UserResponse>> List([FromBody]ApiRequest request)
        {
            var userList = _userService.GetUserList(_claim.At(request.Token).UserID);
            return new ApiResponse<List<UserResponse>>(ResponseType.OK, "ok", userList);
        }

        /// <summary>
        /// 新增用户
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("insert")]
        [HttpPost]
        public ApiResponse<UserResponse> Insert([CustomizeValidator(RuleSet = "Insert"), FromBody]UserRequest request)
        {
            var userIdentity = _claim.At(request.Token);
            var user = _userService.Insert(request, userIdentity.UserID, userIdentity.IsAgainAdmin);
            user.Role = request.Role;
            return new ApiResponse<UserResponse>(ResponseType.OK, user);
        }

        /// <summary>
        /// 修改用户
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("update")]
        [HttpPost]
        public ApiResponse<UserResponse> Update([CustomizeValidator(RuleSet = "Update"), FromBody]UserRequest request)
        {
            var userIdentity = _claim.At(request.Token);
            var user = _userService.Update(request, userIdentity.IsAgainAdmin);
            user.Role = request.Role;
            return new ApiResponse<UserResponse>(ResponseType.OK, user);
        }

        /// <summary>
        /// 修改用户密码
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("password")]
        [HttpPost]
        public ApiResponse<UserResponse> Password([FromBody]PasswordRequest request)
        {
            var userid = _claim.At(request.Token).UserID;
            var user = _userService.UpdatePwd(request, userid);
            return new ApiResponse<UserResponse>(ResponseType.OK, user);
        }

        /// <summary>
        /// 角色列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("rolelist")]
        [HttpPost]
        public ApiResponse<List<sys_role>> RoleList([FromBody]ApiRequest request)
        {
            var roleList = _userService.RoleList();
            return new ApiResponse<List<sys_role>>(ResponseType.OK, "ok", roleList);
        }

        /// <summary>
        /// 科室列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("department")]
        [HttpPost]
        public ApiResponse<List<TitleValue>> Department([FromBody]SetDepartmentRequest request)
        {
            var department = _userService.Department(request.HospitalID);
            return new ApiResponse<List<TitleValue>>(ResponseType.OK, "ok", department);
        }
    }
}
