﻿using Performance.DtoModels;
using Performance.EntityModels;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services.AllotCompute
{
    /// <summary>
    /// 绩效标准数据
    /// </summary>
    public class BaiscNormService : IAutoInjection
    {
        /// <summary>
        /// 获取绩效标准基础值
        /// </summary>
        /// <param name="baiscnorms"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        public decimal? GetBaiscNorm(List<res_baiscnorm> baiscnorms, PerformanceType type)
        {
            decimal? result = null;
            if (type == PerformanceType.ReferenceDirector)
            {
                result = baiscnorms.FirstOrDefault(t => t.PositionName == "临床科主任")?.AvgValue;
            }
            else if (type == PerformanceType.ReferenceDirectorAvg)
            {
                List<string> avgObjectList = new List<string> { "临床科主任", "临床科副主任", "医技科主任", "临床科护长" };
                var baisc = baiscnorms.Where(t => avgObjectList.Contains(t.PositionName));

                result = baisc.Sum(s => s.TotelValue) / baisc.Sum(s => s.TotelNumber);
            }
            else if (type == PerformanceType.ReferenceHeadNurse)
            {
                result = baiscnorms.FirstOrDefault(t => t.PositionName == "临床科护长")?.AvgValue;
            }
            else if (type == PerformanceType.ReferenceNurseAvg95)
            {
                result = baiscnorms.FirstOrDefault(t => t.PositionName == "临床护士")?.AvgValue * 0.95m;
            }
            if (result.HasValue)
                result = Math.Round(result.Value, 4);
            return result;
        }

        /// <summary>
        /// 获取临床护士平均绩效
        /// </summary>
        /// <param name="list"></param>
        /// <returns></returns>
        public res_baiscnorm NurseBaiscnorm(List<PerSheet> list)
        {
            var sheet = list.FirstOrDefault(t => t.SheetType == SheetType.ComputeNurseAccount);
            var perdata = sheet.PerData.Select(t => (PerDataAccountNurse)t);
            return new res_baiscnorm
            {
                PositionName = "临床护士",
                TotelNumber = perdata.Sum(t => t.Number),
                TotelValue = perdata.Sum(t => t.PerforTotal),
                AvgValue = perdata.Sum(t => t.PerforTotal) / perdata.Sum(t => t.Number)
            };
        }

        /// <summary>
        /// 绩效标准计算
        /// </summary>
        /// <param name="computes"></param>
        /// <returns></returns>
        public List<res_baiscnorm> ComputeAvg(List<ComputeResult> computes)
        {
            var keyList = new[]
            {
                new { type = "临床科室", reference = "科室主任人均绩效", groupname = "临床科主任" },
                new { type = "临床科室", reference = "科室副主任人均绩效", groupname = "临床科副主任" },
                new { type = "医技科室", reference = "科室主任人均绩效", groupname = "医技科主任" },
                new { type = "临床科室", reference = "科室护士长人均绩效", groupname = "临床科护长" },
            };

            var groupList = from cp in computes
                            join gp in keyList on new { type = cp.AccountType, reference = cp.FitPeople } equals new { gp.type, gp.reference }
                            select new { gp.groupname, cp };

            var result = groupList.GroupBy(t => t.groupname)
                .Select(s => new res_baiscnorm
                {
                    PositionName = s.Key,
                    TotelNumber = s.Count(),
                    TotelValue = s.Sum(t => t.cp.GiveFee),
                    AvgValue = s.Sum(t => t.cp.GiveFee) / s.Count()
                });
            return result.ToList();
        }
    }
}
