﻿using AutoMapper;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services.AllotCompute
{
    /// <summary>
    /// 对excel导入数据进行合并、计算 并保存
    /// </summary>
    public class ProcessComputService : IAutoInjection
    {
        private PerforCofincomeRepository perforCofincomeRepository;
        private PerforCofdrugpropRepository perforCofdrugpropRepository;
        private PerforPersheetRepository perforPerSheetRepository;
        private PerforImdataRepository perforImDataRepository;
        private PerforImheaderRepository perforImHeaderRepository;
        private PerforImemployeeRepository perforImEmployeeRepository;
        private PerforResaccountdoctorRepository perforResAccountdoctorRepository;
        private PerforResaccountnurseRepository perforResAccountnurseRepository;
        public ProcessComputService(PerforCofincomeRepository perforCofincomeRepository,
            PerforCofdrugpropRepository perforCofdrugpropRepository,
            PerforPersheetRepository perforPerSheetRepository,
            PerforImdataRepository perforImDataRepository,
            PerforImheaderRepository perforImHeaderRepository,
            PerforImemployeeRepository perforImEmployeeRepository,
            PerforResaccountdoctorRepository perforResAccountdoctorRepository,
            PerforResaccountnurseRepository perforResAccountnurseRepository)
        {
            this.perforCofincomeRepository = perforCofincomeRepository;
            this.perforCofdrugpropRepository = perforCofdrugpropRepository;
            this.perforPerSheetRepository = perforPerSheetRepository;
            this.perforImDataRepository = perforImDataRepository;
            this.perforImHeaderRepository = perforImHeaderRepository;
            this.perforImEmployeeRepository = perforImEmployeeRepository;
            this.perforResAccountdoctorRepository = perforResAccountdoctorRepository;
            this.perforResAccountnurseRepository = perforResAccountnurseRepository;
        }

        /// <summary>
        /// 合并计算并保存
        /// </summary>
        /// <param name="excel"></param>
        /// <param name="allot"></param>
        /// <returns></returns>
        public List<PerSheet> MergeAndSave(PerExcel excel, per_allot allot)
        {
            List<PerSheet> list = MergeCompute(excel);
            Save(list, allot.ID);
            return list;
        }

        #region save
        /// <summary>
        /// 保存计算后的医生组绩效
        /// </summary>
        /// <param name="sheet"></param>
        /// <param name="allotId"></param>
        /// <returns></returns>
        private void SaveComputeDoctorAccount(PerSheet sheet, int allotId)
        {
            var imsheet = new per_sheet { AllotID = allotId, SheetName = sheet.SheetName, Source = 2, SheetType = (int)sheet.SheetType };
            perforPerSheetRepository.Add(imsheet);

            var dataList = sheet.PerData.Select(t => (PerDataAccountDoctor)t);
            List<res_accountdoctor> addList = new List<res_accountdoctor>();
            foreach (var data in dataList)
            {
                var imdata = Mapper.Map<res_accountdoctor>(data);
                imdata.SheetID = imsheet.ID;
                imdata.AllotID = allotId;
                addList.Add(imdata);
            }
            perforResAccountdoctorRepository.AddRange(addList.ToArray());
        }

        /// <summary>
        /// 保存计算后的护理组绩效
        /// </summary>
        /// <param name="sheet"></param>
        /// <param name="allotId"></param>
        /// <returns></returns>
        private void SaveComputeNurseAccount(PerSheet sheet, int allotId)
        {
            var imsheet = new per_sheet { AllotID = allotId, SheetName = sheet.SheetName, Source = 2, SheetType = (int)sheet.SheetType };
            perforPerSheetRepository.Add(imsheet);

            var dataList = sheet.PerData.Select(t => (PerDataAccountNurse)t);
            List<res_accountnurse> addList = new List<res_accountnurse>();
            foreach (var data in dataList)
            {
                var imdata = Mapper.Map<res_accountnurse>(data);
                imdata.SheetID = imsheet.ID;
                imdata.AllotID = allotId;
                addList.Add(imdata);
            }
            perforResAccountnurseRepository.AddRange(addList.ToArray());
        }


        /// <summary>
        /// 保存通用格式
        /// </summary>
        /// <param name="sheet"></param>
        /// <param name="allotId"></param>
        /// <returns></returns>
        private void SaveCommon(PerSheet sheet, int allotId)
        {
            var imsheet = new per_sheet { AllotID = allotId, SheetName = sheet.SheetName, Source = 2, SheetType = (int)sheet.SheetType };
            perforPerSheetRepository.Add(imsheet);

            List<im_header> addHeadList = new List<im_header>();
            foreach (var header in sheet.PerHeader)
            {
                var imheader = Mapper.Map<im_header>(header);
                imheader.SheetID = imsheet.ID;
                imheader.AllotID = allotId;
                perforImHeaderRepository.Add(imheader);
                if (header.IsHasChildren)
                {
                    foreach (var child in header.Children)
                    {
                        var imheaderChild = Mapper.Map<im_header>(child);
                        imheaderChild.SheetID = imsheet.ID;
                        imheaderChild.ParentID = imheader.ID;
                        imheaderChild.AllotID = allotId;
                        addHeadList.Add(imheaderChild);
                    }
                }
            }
            perforImHeaderRepository.AddRange(addHeadList.ToArray());

            List<im_data> addDataList = new List<im_data>();
            var dataList = sheet.PerData.Select(t => (PerData)t);
            foreach (var data in dataList)
            {
                var imdata = Mapper.Map<im_data>(data);
                imdata.SheetID = imsheet.ID;
                imdata.AllotID = allotId;
                addDataList.Add(imdata);
            }
            perforImDataRepository.AddRange(addDataList.ToArray());
        }

        /// <summary>
        /// 保存计算过程数据
        /// </summary>
        /// <param name="perSheets"></param>
        /// <param name="allotId"></param>
        private void Save(List<PerSheet> perSheets, int allotId)
        {
            foreach (var sheet in perSheets)
            {
                if (sheet.SheetType == SheetType.ComputeDoctorAccount)
                {
                    SaveComputeDoctorAccount(sheet, allotId);
                }
                else if (sheet.SheetType == SheetType.ComputeNurseAccount)
                {
                    SaveComputeNurseAccount(sheet, allotId);
                }
                else
                {
                    SaveCommon(sheet, allotId);
                }
            }
        }
        #endregion

        #region compute
        /// <summary>
        /// 合并计算
        /// </summary>
        private List<PerSheet> MergeCompute(PerExcel excel)
        {
            List<PerSheet> perSheet = new List<PerSheet>();

            //合并科室收入、支出
            var incomeconfs = perforCofincomeRepository.GetEntities();
            var economicCompute = new PerSheetDataComputeEconomic();
            var mergeResult = economicCompute.MergeCompute(excel, incomeconfs);
            //一次计算
            var onceEconomic = economicCompute.OnceCompute(mergeResult);

            //二次计算 
            var twiceEconomicResult = economicCompute.TwiceCompute(onceEconomic);

            twiceEconomicResult.Sheet.SheetType = SheetType.ComputeEconomic;
            perSheet.Add(twiceEconomicResult.Sheet);

            //工作量
            var workloadCompute = new PerSheetDataComputeWorkload();
            var workload1 = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Workload && t.SheetName.Contains("医生组"));
            workload1.SheetName = "医生组工作量绩效测算表";

            var confs = GetDrugConfig(excel);
            //医生组 一次计算
            var onceWorkload1 = workloadCompute.OnceCompute(workload1, confs);
            //医生组 二次计算 
            var twiceWorkloadResult1 = workloadCompute.TwiceCompute(onceWorkload1);

            twiceWorkloadResult1.Sheet.SheetType = SheetType.ComputeDoctorWorkload;
            perSheet.Add(twiceWorkloadResult1.Sheet);

            var workload2 = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Workload && t.SheetName.Contains("护理组"));
            workload2.SheetName = "护理组工作量绩效测算表";
            //护理组 一次计算
            var onceWorkload2 = workloadCompute.OnceCompute(workload2);
            //护理组 二次计算 
            var twiceWorkloadResult2 = workloadCompute.TwiceCompute(onceWorkload2);

            twiceWorkloadResult2.Sheet.SheetType = SheetType.ComputeNurseWorkload;
            perSheet.Add(twiceWorkloadResult2.Sheet);

            var deptAccounting = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.AccountBasic);
            var dataList = deptAccounting.PerData.Select(t => (PerDataAccountBaisc)t);

            PerSheet doctorSheet = new PerSheet("医生组临床科室单元核算表", "医生组临床科室单元核算表", SheetType.ComputeDoctorAccount, new List<PerHeader>(), new List<IPerData>());
            PerSheet nurseSheet = new PerSheet("护理组临床科室单元核算表", "护理组临床科室单元核算表", SheetType.ComputeNurseAccount, new List<PerHeader>(), new List<IPerData>());
            foreach (var dept in dataList)
            {
                var doctor = Mapper.Map<PerDataAccountDoctor>(dept);
                var econDoctor = twiceEconomicResult.PerData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                doctor.Income = econDoctor?.CellValue ?? 0;
                var workDoctor = twiceWorkloadResult1.PerData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                doctor.WorkloadFee = workDoctor?.CellValue ?? 0;
                doctorSheet.PerData.Add(doctor);

                var nurse = Mapper.Map<PerDataAccountNurse>(dept);
                var econNurse = twiceEconomicResult.PerData.FirstOrDefault(t => t.UnitType == "护理组" && t.AccountingUnit == dept.AccountingUnit);
                nurse.Income = econNurse?.CellValue ?? 0;
                var workNurse = twiceWorkloadResult2.PerData.FirstOrDefault(t => t.UnitType == "护理组" && t.AccountingUnit == dept.AccountingUnit);
                nurse.WorkloadFee = workNurse?.CellValue ?? 0;
                nurseSheet.PerData.Add(nurse);
            }
            perSheet.Add(doctorSheet);
            perSheet.Add(nurseSheet);

            return perSheet;
        }


        /// <summary>
        /// 获取药占比分割比例
        /// </summary>
        /// <param name="excel"></param>
        /// <returns></returns>
        private List<CofDrugProp> GetDrugConfig(PerExcel excel)
        {
            //计算药占比
            List<CofDrugProp> cofs = new List<CofDrugProp>();

            var incomeSheet = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Income && t.SheetName.Contains("门诊") && t.SheetName.Contains("就诊"));
            var datalist = incomeSheet.PerData.Select(t => (PerData)t);
            var drugData = datalist.Where(t => t.TypeName == "西药费" || t.TypeName == "中成药费").GroupBy(t => t.AccountingUnit).Select(t => new { AccountingUnit = t.Key, SumValue = t.Sum(s => s.CellValue) });
            var allData = datalist.GroupBy(t => t.AccountingUnit).Select(t => new { AccountingUnit = t.Key, SumValue = t.Sum(s => s.CellValue) });

            var cofList = perforCofdrugpropRepository.GetEntities();

            var unitList = drugData.Select(t => t.AccountingUnit).Union(allData.Select(t => t.AccountingUnit));
            foreach (var unit in unitList)
            {
                var dsv = drugData.FirstOrDefault(t => t.AccountingUnit == unit)?.SumValue;
                var asv = allData.FirstOrDefault(t => t.AccountingUnit == unit)?.SumValue;

                var prop = asv.HasValue && asv.Value > 0 ? Math.Round((dsv ?? 0) / asv.Value, 2) : 0;
                var fvalue = prop == 0
                    ? 0
                    : cofList.FirstOrDefault(t => prop > t.MinRange && prop <= t.MaxRange)?.Value ?? 0;
                cofs.Add(new CofDrugProp { AccoutingUnit = unit, Factor = fvalue, Prop = prop });
            }
            return cofs;
        }
        #endregion
    }
}
