﻿using AutoMapper;
using Microsoft.AspNetCore.Hosting;
using Microsoft.Extensions.Logging;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Repository;
using Performance.Services.AllotCompute;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace Performance.Services
{
    public class AllotService : IAutoInjection
    {
        private BaiscNormService baiscNormService;
        private CheckDataService checkDataService;
        private ImportDataService importDataService;
        private ProcessComputService processComputService;
        private ResultComputeService resultComputeService;
        private PerforLogdbugRepository logdbug;

        private ConfigService configService;
        private IHostingEnvironment _evn;
        private ILogger<AllotService> _logger;
        private PerforPerallotRepository _allotRepository;
        private IEmailService emailService;
        private PerforPeragainallotRepository _againallotRepository;

        public AllotService(PerforPerallotRepository allotRepository,
             BaiscNormService baiscNormService,
             CheckDataService checkDataService,
             ImportDataService importDataService,
             ProcessComputService processComputService,
             ResultComputeService resultComputeService,
             ConfigService configService,
             PerforLogdbugRepository logdbug,
             IHostingEnvironment evn, ILogger<AllotService> logger,
             IEmailService emailService,
             PerforPeragainallotRepository againallotRepository)
        {
            _allotRepository = allotRepository;
            _againallotRepository = againallotRepository;
            _logger = logger;
            _evn = evn;
            this.baiscNormService = baiscNormService;
            this.checkDataService = checkDataService;
            this.importDataService = importDataService;
            this.processComputService = processComputService;
            this.resultComputeService = resultComputeService;
            this.emailService = emailService;
            this.configService = configService;
            this.logdbug = logdbug;
        }

        #region 基础功能
        /// <summary>
        /// 绩效记录
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <returns></returns>
        public List<AllotResponse> GetAllotList(int? hospitalId)
        {
            if (!hospitalId.HasValue || hospitalId.Value <= 0)
                throw new PerformanceException("hospitalId无效");
            var allotList = _allotRepository.GetEntities(t => t.HospitalId == hospitalId);
            allotList = allotList == null ? allotList : allotList.OrderByDescending(t => t.ID).ToList();
            return Mapper.Map<List<AllotResponse>>(allotList);
        }

        /// <summary>
        /// 新增
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public AllotResponse InsertAllot(AllotRequest request, int userID)
        {
            var repAllot = _allotRepository.GetEntities(t => t.HospitalId == request.HospitalId && t.Year == request.Year && t.Month == request.Month);
            if (repAllot != null && repAllot.Count() > 0)
                throw new PerformanceException("当前绩效记录已存在");

            var allot = Mapper.Map<per_allot>(request);
            allot.CreateDate = DateTime.Now;
            allot.CreateUser = userID;
            allot.States = (int)AllotStates.NoData;
            allot.Remark = EnumHelper.GetDescription(AllotStates.NoData);
            if (!_allotRepository.Add(allot))
                throw new PerformanceException("保存失败");

            return Mapper.Map<AllotResponse>(allot);
        }

        /// <summary>
        /// 修改
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public AllotResponse UpdateAllot(AllotRequest request)
        {
            var allot = _allotRepository.GetEntity(t => t.ID == request.ID);
            if (allot == null)
                throw new PerformanceException("当前绩效记录不存在");

            var repAllot = _allotRepository.GetEntities(t => t.ID != request.ID && t.HospitalId == request.HospitalId
                && t.Year == request.Year && t.Month == request.Month);
            if (repAllot != null && repAllot.Count() > 0)
                throw new PerformanceException("当前绩效记录与其他记录冲突");

            allot.HospitalId = request.HospitalId.Value;
            allot.Year = request.Year;
            allot.Month = request.Month;
            if (!_allotRepository.Update(allot))
                throw new PerformanceException("保存失败");

            return Mapper.Map<AllotResponse>(allot);
        }

        /// <summary>
        /// 删除
        /// </summary>
        /// <param name="iD"></param>
        /// <returns></returns>
        public bool DeleteAllot(int iD)
        {
            var allot = _allotRepository.GetEntity(t => t.ID == iD);
            if (allot == null)
                throw new PerformanceException("当前绩效记录不存在");
            var result = _allotRepository.Remove(allot);
            if (!string.IsNullOrEmpty(allot.Path))
            {
                var dpath = Path.Combine(_evn.ContentRootPath, "Files", $"{allot.HospitalId}", $"{allot.Year}{allot.Month.ToString().PadLeft(2, '0')}");
                var name = FileHelper.GetFileName(allot.Path);
                var path = Path.Combine(dpath, name);
                if (allot.Path != path)
                {
                    try
                    {
                        FileHelper.Move(allot.Path, path);
                        allot.Path = path;
                        _allotRepository.Remove(allot);
                    }
                    catch (Exception ex)
                    {
                        _logger.LogError("移动文件失败;{0}", ex);
                    }
                }

                FileHelper.CreateDirectory(dpath);
            }

            return result;
        }

        /// <summary>
        /// 查询
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public per_allot GetAllot(int allotId)
        {
            return _allotRepository.GetEntity(t => t.ID == allotId);
        }

        /// <summary>
        /// 修改
        /// </summary>
        /// <param name="allot"></param>
        /// <returns></returns>
        public bool Update(per_allot allot)
        {
            return _allotRepository.Update(allot);
        }
        #endregion

        /// <summary>
        /// 生成绩效
        /// </summary>
        /// <param name="allot"></param>
        /// <param name="user"></param>
        public void Generate(per_allot allot, UserIdentity user)
        {
            try
            {
                logdbug.Add(allot.ID, "绩效开始执行", JsonHelper.Serialize(allot));
                configService.Clear(allot.ID);
                _allotRepository.UpdateAllotStates(allot.ID, (int)AllotStates.InCheckData, EnumHelper.GetDescription(AllotStates.InCheckData));
                // 导出数据
                var excel = importDataService.ReadDataAndSave(allot);
                if (!checkDataService.Check(excel, allot))
                {
                    _allotRepository.UpdateAllotStates(allot.ID, (int)AllotStates.CheckFail, EnumHelper.GetDescription(AllotStates.CheckFail));
                    SendEmail(allot, user.Mail, "数值健康，绩效数据校验失败", "绩效数据校验失败");
                    logdbug.Add(allot.ID, "绩效数据校验失败", JsonHelper.Serialize(allot));
                    return;
                }
                _allotRepository.UpdateAllotStates(allot.ID, (int)AllotStates.InGenerate, EnumHelper.GetDescription(AllotStates.InGenerate));
                // 计算合并数据 
                logdbug.Add(allot.ID, "计算合并数据", JsonHelper.Serialize(allot));
                List<PerSheet> list = processComputService.MergeAndSave(excel, allot);

                var baiscnorm = baiscNormService.NurseBaiscnorm(list);
                logdbug.Add(allot.ID, "护士平均绩效", JsonHelper.Serialize(baiscnorm));
                // 计算最总数据
                logdbug.Add(allot.ID, "绩效结果计算数据开始", "");
                var baiscnormList = resultComputeService.Compute(allot, excel, baiscnorm);
                resultComputeService.SpecialUnitCompute(excel, allot, baiscnormList);

                _allotRepository.UpdateAllotStates(allot.ID, (int)AllotStates.GenerateSucceed, EnumHelper.GetDescription(AllotStates.GenerateSucceed));
                //发送邮件
                SendEmail(allot, user.Mail, "绩效生成成功", "绩效生成成功");
                logdbug.Add(allot.ID, "绩效开始执行", "绩效生成成功");
            }
            catch (Exception ex)
            {
                _allotRepository.UpdateAllotStates(allot.ID, (int)AllotStates.GenerateFail, EnumHelper.GetDescription(AllotStates.GenerateFail));
                SendEmail(allot, user.Mail, "绩效生成失败", ex.ToString());
                logdbug.Add(allot.ID, "绩效开始执行", ex.ToString());
                throw ex;
            }
        }

        /// <summary>
        /// 发送邮件
        /// </summary>
        /// <param name="allot"></param>
        /// <param name="mail"></param>
        /// <param name="subject"></param>
        /// <param name="body"></param>
        private void SendEmail(per_allot allot, string mail, string subject, string body)
        {
            var message = new EmailMessage
            {
                To = new List<string> { mail },
                DisplayName = "溯直健康",
                Subject = subject,
                Body = body
            };
            emailService.SendAsync(message);
        }

        /// <summary>
        /// 归档绩效记录
        /// </summary>
        /// <param name="allot"></param>
        public void Pigeonhole(per_allot allot)
        {
            allot.States = 8;
            allot.Remark = "归档";
            if (_allotRepository.Update(allot))
            {
                var again = _againallotRepository.GetEntities(t => t.AllotID == allot.ID);
                foreach (var item in again)
                {
                    item.States = 5;
                    item.Remark = "归档";
                    _againallotRepository.Update(item);
                }
            }
        }

        /// <summary>
        /// 检索数据是否合格
        /// </summary>
        /// <param name="allot"></param>
        public List<AgainAllotResponse> CheckData(per_allot allot)
        {
            List<int> states = new List<int>() { 0, 1, 2, 4 };
            var again = _againallotRepository.GetEntities(t => t.AllotID == allot.ID && states.Contains(t.States.Value));
            List<AgainAllotResponse> result = Mapper.Map<List<AgainAllotResponse>>(again);
            if (result != null && result.Count > 0)
            {
                result.ForEach(t => { t.Year = allot.Year; t.Month = allot.Month; });
            }
            return result;
        }
    }
}
