﻿using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Services.AllotCompute;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services
{
    /// <summary>
    /// 临床科室主任
    /// </summary>
    public class ComputeDirector : IAutoInjection
    {
        private BaiscNormService baiscNormService;
        public ComputeDirector(BaiscNormService baiscNormService)
        {
            this.baiscNormService = baiscNormService;
        }

        /// <summary>
        /// 临床科室主任、临床科室副主任、临床科室护士长 计算
        /// </summary>
        /// <param name="empolyeeList"></param>
        /// <param name="nurseList"></param>
        /// <param name="directorList"></param>
        /// <returns></returns>
        public List<ComputeResult> Compute(List<ComputeEmployee> empolyeeList, List<ComputeSource> nurseList, List<cof_director> directorList)
        {
            Dictionary<PerformanceType, string> keyValues = new Dictionary<PerformanceType, string>
            {
                { PerformanceType.Director, "临床科室主任"},
                { PerformanceType.DeputyDirector, "临床科室副主任"},
                { PerformanceType.Nurse, "临床科室护士长"},
            };
            List<ComputeResult> computeList = new List<ComputeResult>();
            foreach (var key in keyValues.Keys)
            {
                var needCompute = empolyeeList.Where(t => t.FitPeople == EnumHelper.GetDescription(key));
                foreach (var item in needCompute)
                {
                    var resAccount = nurseList.FirstOrDefault(t => t.AccountingUnit == item.AccountingUnit);
                    var efficiency = directorList.FirstOrDefault(t => t.TypeName == "效率绩效" && t.JobTitle == keyValues[key]);
                    var scale = directorList.FirstOrDefault(t => t.TypeName == "规模绩效" && t.JobTitle == keyValues[key]);
                    var compute = new ComputeResult
                    {
                        AccountType = item.AccountType,
                        AccountingUnit = item.AccountingUnit,
                        EmployeeName = item.DoctorName,
                        FitPeople = item.FitPeople,
                        JobTitle = item.JobTitle,

                        Number = resAccount?.Number,
                        PerforTotal = resAccount?.PerforTotal,
                        Avg = resAccount?.Avg,
                        Efficiency = efficiency?.Value * resAccount?.Avg,
                        Scale = scale?.Value * resAccount?.PerforTotal,
                        Grant = item.Grant,
                        ScoreAverageRate = item.ScoreAverageRate,
                        Punishment = item.Punishment,
                        OtherPerfor = item.OtherPerfor
                    };
                    //应发管理绩效
                    compute.ShouldGiveFee = (efficiency?.Value * resAccount?.Avg + scale?.Value * resAccount?.PerforTotal) * item.Grant;
                    //绩效合计
                    var perforSumFee = (compute.Avg + compute.Efficiency + compute.Scale) * item.Grant;
                    if (perforSumFee > (compute.Avg ?? 0) * 2.5m)
                        compute.PerforSumFee = (compute.Avg ?? 0) * 2.5m;
                    else
                        compute.PerforSumFee = perforSumFee;
                    //应发绩效
                    compute.GiveFee = compute.Avg * item.Grant + (compute.Efficiency + compute.Scale) * item.Grant * compute.ScoreAverageRate + (item.Punishment ?? 0) + (item.OtherPerfor ?? 0);
                    //实发绩效
                    compute.RealGiveFee = compute.GiveFee * item.Adjust - compute.ShouldGiveFee * item.ScoreAverageRate * item.Grant * 0.1m;
                    computeList.Add(compute);
                }

            }
            return computeList;
        }

        /// <summary>
        /// 绩效标准计算
        /// </summary>
        /// <param name="computes"></param>
        /// <returns></returns>
        public List<res_baiscnorm> ComputeAvg(List<ComputeResult> computes)
        {
            var keyList = new[]
            {
                new { type = "临床科室", reference = "科室主任人均绩效", groupname = "临床科主任" },
                new { type = "临床科室", reference = "科室副主任人均绩效", groupname = "临床科副主任" },
                new { type = "医技科室", reference = "科室主任人均绩效", groupname = "医技科主任" },
                new { type = "临床科室", reference = "科室护士长人均绩效", groupname = "临床科护长" },
            };

            var groupList = from cp in computes
                            join gp in keyList on new { type = cp.AccountType, reference = cp.FitPeople } equals new { gp.type, gp.reference }
                            select new { gp.groupname, cp };

            var result = groupList.GroupBy(t => t.groupname)
                .Select(s => new res_baiscnorm
                {
                    PositionName = s.Key,
                    TotelNumber = s.Count(),
                    TotelValue = s.Sum(t => t.cp.GiveFee),
                    AvgValue = s.Sum(t => t.cp.GiveFee) / s.Count()
                });
            return result.ToList();
        }

        /// <summary>
        /// 院领导、业务中层、工勤人员 计算
        /// </summary>
        /// <param name="empolyeeList"></param>
        /// <param name="baiscnormList"></param>
        /// <returns></returns>
        public List<ComputeResult> Compute(List<ComputeEmployee> empolyeeList, List<res_baiscnorm> baiscnormList, List<cof_workyear> workyearList)
        {
            PerformanceType[] types = new PerformanceType[]
            {
                PerformanceType.ReferenceDirector,
                PerformanceType.ReferenceDirectorAvg,
                PerformanceType.ReferenceNurseAvg95,
                PerformanceType.ReferenceHeadNurse,
                PerformanceType.Null,
            };
            PerformanceType[] yearTypes = new PerformanceType[]
            {
                PerformanceType.ReferenceNurseAvg95,
                PerformanceType.ReferenceHeadNurse,
                PerformanceType.Null,
            };
            List<ComputeResult> computeList = new List<ComputeResult>();

            foreach (var type in types)
            {
                var needCompute = empolyeeList.Where(t => t.FitPeople == EnumHelper.GetDescription(type));
                var baiscnorm = baiscNormService.GetBaiscNorm(baiscnormList, type);

                foreach (var item in needCompute)
                {
                    var compute = new ComputeResult
                    {
                        AccountType = item.AccountType,
                        AccountingUnit = item.AccountingUnit,
                        EmployeeName = item.DoctorName,
                        FitPeople = item.FitPeople,
                        Grant = item.Grant,
                        WorkTime = item.WorkTime,
                        BaiscNormValue = baiscnorm,
                        PostCoefficient = item.PostCoefficient,
                        Attendance = item.Attendance,
                        ScoreAverageRate = item.ScoreAverageRate,
                        Punishment = item.Punishment,
                        OtherPerfor = item.OtherPerfor
                    };
                    //年资系数
                    if (yearTypes.Contains(type) && item.WorkTime.HasValue && item.WorkTime.Value > new DateTime(1970, 1, 1))
                    {
                        var years = ((DateTime.Now.Year - item.WorkTime.Value.Year) * 12 + (DateTime.Now.Month - item.WorkTime.Value.Month)) / 12.0m;
                        var value = workyearList.FirstOrDefault(t => t.MinRange < years && years <= t.MaxRange)?.Value;
                        compute.WorkYear = value;
                    }  
                    //应发绩效
                    compute.GiveFee = baiscnorm * compute.PostCoefficient * compute.Attendance * compute.ScoreAverageRate * (compute.WorkYear ?? 1) + (compute.OtherPerfor ?? 0) + (item.Workload ?? 0) + (compute.Punishment ?? 0);
                    //实发绩效
                    compute.RealGiveFee = compute.GiveFee * item.Adjust * item.Grant;
                    computeList.Add(compute);
                }
            }
            return computeList;
        }
    }


}
