﻿using AutoMapper;
using NPOI.SS.UserModel;
using Performance.DtoModels;
using Performance.Infrastructure;
using Performance.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Performance.Services
{
    public class PerSheetService : IAutoInjection
    {
        private PerHeaderService _perHeader;
        private PerforCofincomeRepository _perforCofincomeRepository;
        private PerforCofdrugpropRepository _perforCofdrugpropRepository;
        public PerSheetService(PerHeaderService perHeader,
            PerforCofincomeRepository perforCofincomeRepository,
            PerforCofdrugpropRepository perforCofdrugpropRepository)
        {
            _perHeader = perHeader;
            _perforCofincomeRepository = perforCofincomeRepository;
            _perforCofdrugpropRepository = perforCofdrugpropRepository;
        }

        /// <summary>
        /// 解析excel数据
        /// </summary>
        /// <param name="sheet"></param>
        /// <returns></returns>
        public PerSheet Sheet(ISheet sheet)
        {
            PerSheet perSheet = new PerSheet();
            perSheet.SheetName = sheet.SheetName;
            perSheet.SheetType = GetSheetType(sheet.SheetName);
            perSheet.ModuleName = EnumHelper.GetDescription(perSheet.SheetType);

            if (perSheet.SheetType == SheetType.Unidentifiable)
                return null;

            var sheetRead = PerSheetDataFactory.GetDataRead(perSheet.SheetType);
            var perHeader = _perHeader.GetPerHeader(sheet, sheetRead.Point);
            var headerReverse = _perHeader.GetPerHeaderReverse(perHeader);

            perSheet.PerHeader = perHeader;
            perSheet.PerData = sheetRead.ReadData(sheet, headerReverse);
            return perSheet;
        }

        /// <summary>
        /// 判断sheet适用模板类型
        /// </summary>
        /// <param name="sheetName"></param>
        /// <returns></returns>
        public SheetType GetSheetType(string sheetName)
        {
            if (sheetName.StartsWith("医院人员名单"))
                return SheetType.Employee;
            else if (sheetName.StartsWith("1.0"))
                return SheetType.OtherIncome;
            else if (sheetName.StartsWith("1."))
                return SheetType.Income;
            else if (sheetName.StartsWith("2."))
                return SheetType.Expend;
            else if (sheetName.StartsWith("3."))
                return SheetType.Workload;
            else if (sheetName.StartsWith("4.1"))
                return SheetType.AccountBasic;
            else if (sheetName.StartsWith("4.2"))
                return SheetType.SpecialUnit;
            return SheetType.Unidentifiable;
        }

        internal List<PerSheet> ProcessCompute(PerExcel excel)
        {
            List<PerSheet> perSheet = new List<PerSheet>();


            //合并科室收入、支出
            var incomeconfs = _perforCofincomeRepository.GetEntities();
            var economicCompute = new PerSheetDataComputeEconomic();
            var mergeResult = economicCompute.MergeCompute(excel, incomeconfs);
            //一次计算
            var onceEconomic = economicCompute.OnceCompute(mergeResult);

            //二次计算 
            var twiceEconomicResult = economicCompute.TwiceCompute(onceEconomic);

            twiceEconomicResult.Sheet.SheetType = SheetType.ComputeEconomic;
            perSheet.Add(twiceEconomicResult.Sheet);

            //工作量
            var workloadCompute = new PerSheetDataComputeWorkload();
            var workload1 = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Workload && t.SheetName.Contains("医生组"));
            workload1.SheetName = "医生组工作量绩效测算表";

            var confs = GetDrugConfig(excel);
            //医生组 一次计算
            var onceWorkload1 = workloadCompute.OnceCompute(workload1, confs);
            //医生组 二次计算 
            var twiceWorkloadResult1 = workloadCompute.TwiceCompute(onceWorkload1);

            twiceWorkloadResult1.Sheet.SheetType = SheetType.ComputeDoctorWorkload;
            perSheet.Add(twiceWorkloadResult1.Sheet);

            var workload2 = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Workload && t.SheetName.Contains("护理组"));
            workload2.SheetName = "护理组工作量绩效测算表";
            //护理组 一次计算
            var onceWorkload2 = workloadCompute.OnceCompute(workload2);
            //护理组 二次计算 
            var twiceWorkloadResult2 = workloadCompute.TwiceCompute(onceWorkload2);

            twiceWorkloadResult2.Sheet.SheetType = SheetType.ComputeNurseWorkload;
            perSheet.Add(twiceWorkloadResult2.Sheet);

            var deptAccounting = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.AccountBasic);
            var dataList = deptAccounting.PerData.Select(t => (PerDataAccountBaisc)t);

            PerSheet doctorSheet = new PerSheet("医生组临床科室单元核算表", "医生组临床科室单元核算表", SheetType.ComputeDoctorAccount, new List<PerHeader>(), new List<IPerData>());
            PerSheet nurseSheet = new PerSheet("护理组临床科室单元核算表", "护理组临床科室单元核算表", SheetType.ComputeNurseAccount, new List<PerHeader>(), new List<IPerData>());
            foreach (var dept in dataList)
            {
                var doctor = Mapper.Map<PerDataAccountDoctor>(dept);
                var econDoctor = twiceEconomicResult.PerData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                doctor.Income = econDoctor?.CellValue ?? 0;
                var workDoctor = twiceWorkloadResult1.PerData.FirstOrDefault(t => t.UnitType == "医生组" && t.AccountingUnit == dept.AccountingUnit);
                doctor.WorkloadFee = workDoctor?.CellValue ?? 0;
                doctorSheet.PerData.Add(doctor);

                var nurse = Mapper.Map<PerDataAccountNurse>(dept);
                var econNurse = twiceEconomicResult.PerData.FirstOrDefault(t => t.UnitType == "护理组" && t.AccountingUnit == dept.AccountingUnit);
                nurse.Income = econNurse?.CellValue ?? 0;
                var workNurse = twiceWorkloadResult2.PerData.FirstOrDefault(t => t.UnitType == "护理组" && t.AccountingUnit == dept.AccountingUnit);
                nurse.WorkloadFee = workNurse?.CellValue ?? 0;
                nurseSheet.PerData.Add(nurse);
            }
            perSheet.Add(doctorSheet);
            perSheet.Add(nurseSheet);

            return perSheet;
        }


        /// <summary>
        /// 获取药占比分割比例
        /// </summary>
        /// <param name="excel"></param>
        /// <returns></returns>
        private List<CofDrugProp> GetDrugConfig(PerExcel excel)
        {
            //计算药占比
            List<CofDrugProp> cofs = new List<CofDrugProp>();

            var incomeSheet = excel.PerSheet.FirstOrDefault(t => t.SheetType == SheetType.Income && t.SheetName.Contains("门诊") && t.SheetName.Contains("就诊"));
            var datalist = incomeSheet.PerData.Select(t => (PerData)t);
            var drugData = datalist.Where(t => t.TypeName == "西药费" || t.TypeName == "中成药费").GroupBy(t => t.AccountingUnit).Select(t => new { AccountingUnit = t.Key, SumValue = t.Sum(s => s.CellValue) });
            var allData = datalist.GroupBy(t => t.AccountingUnit).Select(t => new { AccountingUnit = t.Key, SumValue = t.Sum(s => s.CellValue) });

            var cofList = _perforCofdrugpropRepository.GetEntities();

            var unitList = drugData.Select(t => t.AccountingUnit).Union(allData.Select(t => t.AccountingUnit));
            foreach (var unit in unitList)
            {
                var dsv = drugData.FirstOrDefault(t => t.AccountingUnit == unit)?.SumValue;
                var asv = allData.FirstOrDefault(t => t.AccountingUnit == unit)?.SumValue;

                var prop = asv.HasValue && asv.Value > 0 ? Math.Round((dsv ?? 0) / asv.Value, 2) : 0;
                var fvalue = prop == 0
                    ? 0
                    : cofList.FirstOrDefault(t => prop > t.MinRange && prop <= t.MaxRange)?.Value ?? 0;
                cofs.Add(new CofDrugProp { AccoutingUnit = unit, Factor = fvalue, Prop = prop });
            }
            return cofs;
        }
    }
}
