﻿using Microsoft.AspNetCore.SignalR;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Options;
using Performance.DtoModels.AppSettings;
using Performance.Infrastructure;
using Performance.Repository;
using System;

namespace Performance.Services
{
    public class LogManageService : IAutoInjection
    {
        private readonly ILogger<LogManageService> logger;
        private readonly IHubContext<AllotLogHub> hubContext;
        private readonly WebapiUrl url;
        private PerforLogdbugRepository logdbug;

        public LogManageService(
            ILogger<LogManageService> logger,
            IHubContext<AllotLogHub> hubContext,
            IOptions<WebapiUrl> url,
            PerforLogdbugRepository logdbug)
        {
            this.logger = logger;
            this.hubContext = hubContext;
            this.url = url.Value;
            this.logdbug = logdbug;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="tag">标签</param>
        /// <param name="message">内容</param>
        /// <param name="level">等级1、信息（info）2、警告（warn）3、错误（error）4、异常（exception）5、成功（success）</param>
        /// <param name="allotId">绩效Id</param>
        /// <param name="method">方法名称</param>
        public void WriteMsg(string tag, string message, int level, int allotId, string method, bool isDebug = false)
        {
            logger.LogInformation($"method:{method};tag:{tag}; message:{message}; level:{level}");
            hubContext.Clients.Group(allotId.ToString()).SendAsync(method, tag, message, level);
            if (isDebug)
            {
                logdbug.Add(allotId, tag, message, level, 1);
            }
        }

        /// <summary>
        /// 抽取进度
        /// </summary>
        /// <param name="groupName"></param>
        /// <param name="ratio"></param>
        /// <param name="level">等级1、信息（info）2、警告（warn）3、错误（error）4、异常（exception）5、成功（success）</param>
        /// <param name="success"></param>
        public void Schedule(string groupName, decimal ratio, int level)
        {
            hubContext.Clients.Group(groupName).SendAsync("Schedule", ratio, level);
        }

        /// <summary>
        /// 抽取日志
        /// </summary>
        /// <param name="groupName"></param>
        /// <param name="tag"></param>
        /// <param name="message"></param>
        /// <param name="level">等级1、信息（info）2、警告（warn）3、错误（error）4、异常（exception）5、成功（success）</param>
        public void ExtractLog(string groupName, string tag, string message, int level)
        {
            hubContext.Clients.Group(groupName).SendAsync("ExtractLog", tag, message, level);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="allotId"></param>
        /// <param name="groupName"></param>
        /// <param name="type">1、绩效生成日志 2、绩效提取日志 3、绩效提取进度</param>
        /// <param name="tag"></param>
        /// <param name="message"></param>
        /// <param name="level">1、信息（info）2、警告（warn）3、错误（error）4、异常（exception）5、成功（success）</param>
        /// <param name="isSingle"></param>
        public void ReturnTheLog(int allotId, string groupName, int type, string tag, object message, int level = 1, bool isSingle = false)
        {
            try
            {
                string content = ""; decimal ratio = 0;
                if (type == 2)
                {
                    content = message.ToString();
                    logger.LogInformation(content);
                    if (isSingle) hubContext.Clients.Group(groupName).SendAsync("ExtractLog", tag, content, level);
                }
                else if (type == 3)
                {
                    ratio = Math.Round(Convert.ToDecimal(message), 2, MidpointRounding.AwayFromZero);
                    if (level != 5 && ratio > 100) ratio = 99;
                    if (level == 5 && ratio != 100) ratio = 100;
                    content = ratio.ToString();
                    if (isSingle) hubContext.Clients.Group(groupName).SendAsync("Schedule", ratio, level);
                }

                logdbug.Add(allotId, tag, content, level, type);
                if (!isSingle)
                {
                    var http = url.ImportFile + $"/template/returnlog?type={type}&tag={tag}&message={content}&level={level}&groupName={groupName}";
                    HttpHelper.HttpPost(http);
                }
            }
            catch (Exception ex)
            {
                logger.LogInformation("发送日志失败：" + ex.ToString());
            }
        }

        public bool ClearExtractLog(int allotId)
        {
            return logdbug.ClearExtractLog(allotId);
        }
    }
}
