﻿using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Services;
using Performance.Services.ExtractExcelService;
using System;
using System.IO;
using System.Linq;

namespace Performance.Api.Controllers
{
    [ApiController]
    [Route("api/{hospitalId}/report/global")]
    public class ReportGlobalController : Controller
    {
        private readonly IWebHostEnvironment env;
        private readonly AllotService allotService;
        private readonly ReportGlobalService reportGlobalService;

        public ReportGlobalController(
            IWebHostEnvironment env,
            AllotService allotService,
            ReportGlobalService reportGlobalService
            )
        {
            this.env = env;
            this.allotService = allotService;
            this.reportGlobalService = reportGlobalService;
        }

        /// <summary>
        /// 获取报表配置信息
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <returns></returns>
        [HttpGet]
        public ApiResponse GetAllReportGlobal([FromRoute] int hospitalId)
        {
            //reportGlobalService.CopyPreviousGlobalData(new per_allot
            //{
            //    ID = 101,
            //    HospitalId = 13,
            //    Year = 2020,
            //    Month = 8
            //});

            var result = reportGlobalService.GetReportGlobals(hospitalId);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 添加报表配置
        /// </summary>
        /// <param name="global"></param>
        /// <returns></returns>
        [HttpPost]
        public ApiResponse CreateReportGlobal([FromBody] report_global global)
        {
            var result = reportGlobalService.CreateReportGlobal(global);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 修改报表配置
        /// </summary>
        /// <param name="global"></param>
        /// <returns></returns>
        [HttpPost("update")]
        public ApiResponse UpdateReportGlobal([FromBody] report_global global)
        {
            var result = reportGlobalService.UpdateReportGlobal(global);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 删除报表配置
        /// </summary>
        /// <param name="globalId"></param>
        /// <returns></returns>
        [HttpPost("{globalId}")]
        public ApiResponse DeleteReportGlobal([FromRoute] int globalId)
        {
            var result = reportGlobalService.DeleteReportGlobal(globalId);
            return new ApiResponse(ResponseType.OK, result);
        }

        /// <summary>
        /// 上传人员绩效文件
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="form"></param>
        /// <returns></returns>
        [Route("import")]
        [HttpPost]
        public ApiResponse Import(int hospitalId, [FromForm] IFormCollection form)
        {
            var file = ((FormFileCollection)form.Files).FirstOrDefault();
            if (file == null)
                return new ApiResponse(ResponseType.Fail, "参数错误", "文件无效");

            if (!ExtractHelper.IsXlsxFile(file.FileName))
                return new ApiResponse(ResponseType.Fail, "文件格式错误", "文件暂只支持xlsx文件");

            var name = FileHelper.GetFileNameNoExtension(file.FileName) + DateTime.Now.ToString("yyyyMMddHHmmssfff");
            var ext = FileHelper.GetExtension(file.FileName);
            var dpath = Path.Combine(env.ContentRootPath, "Files", $"{hospitalId}", $"ImportDataFiles");
            FileHelper.CreateDirectory(dpath);

            var path = Path.Combine(dpath, $"{name}{ext}");

            using (var stream = file.OpenReadStream())
            {
                byte[] bytes = new byte[stream.Length];
                stream.Read(bytes, 0, bytes.Length);
                if (!FileHelper.CreateFile(path, bytes))
                    return new ApiResponse(ResponseType.Fail, $"{file.FileName}上传失败");
            }
            reportGlobalService.ImportAllotData(hospitalId, path);
            return new ApiResponse(ResponseType.OK);
        }

        #region 人员、科室标签配置

        /// <summary>
        /// 获取人员标签配置
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <returns></returns>
        [Route("ReportPersonTag")]
        [HttpPost]
        public ApiResponse ReportPersonTag(int hospitalId)
        {
            if (hospitalId<=0)
            {
                return new ApiResponse(ResponseType.Fail,"参数错误", "hospitalId无效");
            }
            var relust = reportGlobalService.GetReportPersonTag(hospitalId);
            return new ApiResponse(ResponseType.OK, relust);
        }

        /// <summary>
        /// 保存科室标签配置
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("saveReportPersonTag")]
        [HttpPost]
        public ApiResponse SaveReportPersonTag(int hospitalId,[FromBody] SaveCollectData request)
        {
            if (hospitalId <= 0)
            {
                return new ApiResponse(ResponseType.Fail, "参数错误", "hospitalId无效");
            }
            reportGlobalService.SaveReportPersonTag(hospitalId,request);
            return new ApiResponse(ResponseType.OK);
        }

        /// <summary>
        /// 获取人员标签配置
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <returns></returns>
        [Route("ReportTag")]
        [HttpPost]
        public ApiResponse ReportTag(int hospitalId)
        {
            if (hospitalId <= 0)
            {
                return new ApiResponse(ResponseType.Fail, "参数错误", "hospitalId无效");
            }
            var relust = reportGlobalService.GetReportTag(hospitalId);
            return new ApiResponse(ResponseType.OK, relust);
        }

        /// <summary>
        /// 保存科室标签配置
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("saveReportTag")]
        [HttpPost]
        public ApiResponse SaveReportTag(int hospitalId, [FromBody] SaveCollectData request)
        {
            if (hospitalId <= 0)
            {
                return new ApiResponse(ResponseType.Fail, "参数错误", "hospitalId无效");
            }
            reportGlobalService.SaveReportTag(hospitalId, request);
            return new ApiResponse(ResponseType.OK);
        }
        #endregion
    }
}
