﻿using Microsoft.AspNetCore.SignalR;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Logging;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Repository;
using System;
using System.Collections.Generic;
using System.Text;
using System.Threading.Tasks;

namespace Performance.Services
{
    public class NotificationsService : IAutoInjection
    {
        private readonly ILogger logger;
        private readonly IMemoryCache cache;
        private readonly IHubContext<AllotLogHub> hubContext;
        private readonly PerforLogdsignalrRepository repository;

        public NotificationsService(
            ILogger<NotificationsService> logger,
            IMemoryCache cache,
            IHubContext<AllotLogHub> hubContext,
            PerforLogdsignalrRepository repository
            )
        {
            this.logger = logger;
            this.cache = cache;
            this.hubContext = hubContext;
            this.repository = repository;
        }

        public const string AllotGenerateKeyPrefix = "Allot-Generate-";

        public void SendGenerateMessage(int allotId, string message, ComponentType type = ComponentType.info)
        {
            var data = GetCache<SingleData>(AllotGenerateKeyPrefix + allotId);
            if (data == null || data.UserId == 0)
                return;

            var info = new MessageInfo(data.Uuid, allotId, "生成绩效", message, Component.Notification, type);
            SendMessage(data.UserId, info);
        }

        public void SendMessage(int userId, MessageInfo data)
        {
            string key = userId.ToString();
            if (cache.TryGetValue(key, out string connectionId) && !string.IsNullOrEmpty(connectionId))
            {
                if (data.Component == Component.Progress)
                {
                    var percentage = ConvertHelper.To<decimal>(data.Message);
                    data.Message = (data.Status == ComponentType.success
                          ? 100
                          : (data.Status != ComponentType.success && percentage >= 100) ? 99 : percentage).ToString();
                }
                SendMessageToClient(connectionId, data);
            }
        }

        private void SaveLogToDataBase()
        {

        }

        private void SendMessageToClient(string connectionId, MessageInfo data, string method = "ReceiveMessage")
        {
            hubContext.Clients.Client(connectionId).SendAsync(method, data.Title, data);
        }

        #region 缓存

        /// <summary>
        /// 添加缓存
        /// </summary>
        /// <param name="key">缓存Key</param>
        /// <param name="value">缓存Value</param>
        public void SetCache(string key, object value)
        {
            if (!string.IsNullOrEmpty(key))
            {
                if (cache.TryGetValue(key, out _))
                {
                    cache.Remove(key);
                }
                cache.Set(key, value);
            }
        }

        /// <summary>
        /// 删除缓存
        /// </summary>
        /// <param name="key">缓存Key</param>
        public void RemoveCache(string key)
        {
            if (!string.IsNullOrEmpty(key))
            {
                cache.Remove(key);
            }
        }

        /// <summary>
        /// 获取缓存
        /// </summary>
        /// <param name="key">缓存Key</param>
        /// <returns></returns>
        public T GetCache<T>(string key) where T : class
        {
            if (key == null)
                throw new ArgumentNullException(nameof(key));

            return cache.Get(key) as T;
        }

        /// <summary>
        /// 获取缓存
        /// </summary>
        /// <param name="key">缓存Key</param>
        /// <returns></returns>
        public object GetCache(string key)
        {
            if (key == null)
                throw new ArgumentNullException(nameof(key));

            return cache.Get(key);
        }

        #endregion
    }
}
