﻿using AutoMapper;
using Microsoft.AspNetCore.Hosting;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Options;
using NPOI.SS.UserModel;
using NPOI.XSSF.UserModel;
using Performance.DtoModels;
using Performance.DtoModels.AppSettings;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Infrastructure.Models;
using Performance.Repository;
using Performance.Repository.Repository;
using Performance.Services.ExtractExcelService;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Text.RegularExpressions;

namespace Performance.Services
{
    public class PersonService : IAutoInjection
    {
        private readonly IMapper _mapper;
        private readonly ILogger<PersonService> logger;
        private readonly PerforPerdeptdicRepository perdeptdicRepository;
        private readonly PerforPeremployeeRepository peremployeeRepository;
        private readonly PerforPeremployeeBackupRepository perforPeremployeeBackupRepository;
        private readonly PerforPerallotRepository perallotRepository;
        private readonly PerforUserRepository perforUserRepository;
        private readonly PerforUserroleRepository perforUserroleRepository;
        private readonly PerforRoleRepository perforRoleRepository;
        private readonly PerforAgsecondallotRepository agsecondallotRepository;
        private readonly PerforHospitalRepository perforHospitalRepository;
        private readonly PerforCofaccountingRepository perforCofaccountingRepository;
        private readonly UserService userService;
        private readonly DapperService dapperService;
        private readonly Application application;
        private readonly IWebHostEnvironment evn;

        public PersonService(
            IMapper mapper,
            ILogger<PersonService> logger,
            PerforPerdeptdicRepository perdeptdicRepository,
            PerforPeremployeeRepository peremployeeRepository,
            PerforPeremployeeBackupRepository perforPeremployeeBackupRepository,
            PerforPerallotRepository perallotRepository,
            PerforUserRepository perforUserRepository,
            PerforUserroleRepository perforUserroleRepository,
            PerforRoleRepository perforRoleRepository,
            PerforAgsecondallotRepository agsecondallotRepository,
            PerforHospitalRepository perforHospitalRepository,
            PerforCofaccountingRepository perforCofaccountingRepository,
            UserService userService,
            DapperService dapperService,
            IOptions<Application> application,
            IWebHostEnvironment evn
            )
        {
            _mapper = mapper;
            this.logger = logger;
            this.perdeptdicRepository = perdeptdicRepository;
            this.peremployeeRepository = peremployeeRepository;
            this.perforPeremployeeBackupRepository = perforPeremployeeBackupRepository;
            this.perallotRepository = perallotRepository;
            this.perforUserRepository = perforUserRepository;
            this.perforUserroleRepository = perforUserroleRepository;
            this.perforRoleRepository = perforRoleRepository;
            this.agsecondallotRepository = agsecondallotRepository;
            this.perforHospitalRepository = perforHospitalRepository;
            this.perforCofaccountingRepository = perforCofaccountingRepository;
            this.userService = userService;
            this.dapperService = dapperService;
            this.application = application.Value;
            this.evn = evn;
        }

        /// <summary>
        /// 创建绩效时，批量创建员工信息
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="allotId"></param>
        /// <param name="prevAllotId"></param>
        public void CreateAllotPersons(int hospitalId, int allotId, int prevAllotId = -1)
        {
            var allot = perallotRepository.GetEntity(t => t.ID == allotId);
            if (allot == null) throw new PerformanceException("绩效信息错误！");

            var isExist = (peremployeeRepository.GetEntities(t => t.HospitalId == hospitalId && t.AllotId == allotId)?.Count ?? 0) > 0;
            //如果为空则先删除在执行下面的代码
            if (isExist) { peremployeeRepository.DeleteFromQuery(u => u.HospitalId == hospitalId && u.AllotId == allotId); }


            List<per_employee> persons = new List<per_employee>();
            if (!new int[] { -1, 0 }.Contains(prevAllotId))
            {
                persons = peremployeeRepository.GetEntities(t => t.HospitalId == hospitalId && t.AllotId == prevAllotId);
                if (persons == null || !persons.Any())
                    persons = peremployeeRepository.GetEntities(t => t.HospitalId == hospitalId && t.AllotId == -1);
            }
            else
                persons = peremployeeRepository.GetEntities(t => t.HospitalId == hospitalId && t.AllotId == -1);

            if (persons == null || !persons.Any()) return;

            int day = allot.Month >= 1 && allot.Month <= 12 ? DateTime.DaysInMonth(allot.Year, allot.Month) : 30;

            var data = persons.Select(t =>
            {
                var entity = new per_employee
                {
                    HospitalId = t.HospitalId,
                    AllotId = allotId,
                    AccountingUnit = t.AccountingUnit,
                    Department = t.Department,
                    DoctorName = t.DoctorName,
                    JobCategory = t.JobCategory,
                    Duty = t.Duty,
                    JobTitle = t.JobTitle,
                    UnitType = t.UnitType,
                    AttendanceDay = day,
                    Attendance = 1,
                    PermanentStaff = t.PermanentStaff,
                    EfficiencyNumber = t.EfficiencyNumber,
                    WorkTime = t.WorkTime,
                    BirthDate = t.BirthDate,
                    Age = t.Age,
                    ReservedRatio = t.ReservedRatio,
                    BankCard = t.BankCard,
                    Remark = t.Remark,
                    CreateTime = DateTime.Now,
                    PersonnelNumber = t.PersonnelNumber,
                    JobNumber = t.JobNumber,
                    Reserve01 = t.Reserve01,
                    Reserve02 = t.Reserve02,
                    Reserve03 = t.Reserve03,
                    Reserve04 = t.Reserve04,
                    Reserve05 = t.Reserve05,
                    Reserve06 = t.Reserve06,
                    Reserve07 = t.Reserve07,
                    Reserve08 = t.Reserve08,
                    Reserve09 = t.Reserve09,
                    Reserve10 = t.Reserve10,
                };
                string number = !string.IsNullOrEmpty(t.PersonnelNumber) ? t.PersonnelNumber : t.JobNumber;
                entity.PersonnelNumber = number;

                return entity;
            }).ToList();

            SaveAllotPersons(data);
        }

        private void SaveAllotPersons(List<per_employee> employees)
        {
            try
            {
                if (employees.Count > 1000)
                {
                    int rows = 500;
                    for (int i = 0; i < Math.Ceiling((double)employees.Count / rows); i++)
                    {
                        peremployeeRepository.AddRange(employees.Skip(rows * i).Take(rows).ToArray());
                    }
                }
                else
                    peremployeeRepository.AddRange(employees.ToArray());
            }
            catch (Exception ex)
            {
                logger.LogError(ex.ToString());
            }
        }

        /// <summary>
        /// 获取所有员工记录
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public List<per_employee> GetPerEmployee(int allotId)
        {
            return peremployeeRepository.GetEntities(t => t.AllotId == allotId)?.OrderBy(t => t.Id).ToList();
        }

        /// <summary>
        /// 获取所有员工记录
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public List<per_employee> GetPersons(int allotId, int userId)
        {
            var userInfo = perforUserRepository.GetUser(userId);
            var dept = userInfo?.User.Department ?? "";
            var unittype = (userInfo != null && userInfo.URole.Type.HasValue && UnitTypeUtil.Maps.ContainsKey(userInfo.URole.Type.Value))
                ? UnitTypeUtil.GetMaps(userInfo.URole.Type.Value)
                : new string[0];

            Expression<Func<per_employee, bool>> exp = t => t.AllotId == allotId;
            if (!string.IsNullOrEmpty(dept) && unittype.Any())
                exp = exp.And(t => t.AccountingUnit == dept && unittype.Contains(t.UnitType));

            var list = peremployeeRepository.GetEntities(exp);
            if (list != null && list.Any())
                return list.OrderBy(w => w.IsVerify).ThenBy(t => t.Id).ToList();

            return new List<per_employee>();
        }

        /// <summary>
        /// 获取所有员工记录分页
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        public PageList<per_employee> GetPersons(int allotId, int userId, PersonParamsRequest request)
        {
            var userInfo = perforUserRepository.GetUser(userId);
            var dept = userInfo?.User.Department ?? "";
            var unittype = (userInfo != null && userInfo.URole.Type.HasValue && UnitTypeUtil.Maps.ContainsKey(userInfo.URole.Type.Value))
                ? UnitTypeUtil.GetMaps(userInfo.URole.Type.Value)
                : new string[0];

            Expression<Func<per_employee, bool>> exp = t => t.AllotId == allotId;
            if (!string.IsNullOrEmpty(dept) && unittype.Any())
                exp = exp.And(t => t.AccountingUnit == dept && unittype.Contains(t.UnitType));

            if (request != null && !string.IsNullOrEmpty(request.SearchQuery))
                exp = exp.And(t => true && (t.AccountingUnit.Contains(request.SearchQuery) || t.PersonnelNumber.Contains(request.SearchQuery) || t.DoctorName.Contains(request.SearchQuery) || t.Department.Contains(request.SearchQuery) || t.UnitType.Contains(request.SearchQuery)));

            var result = new List<per_employee>();

            return peremployeeRepository.GetEntitiesForPaging(request.PageNumber, request.PageSize, exp);
        }

        /// <summary>
        /// 新增员工信息
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public ApiResponse CreatePerson(PerEmployeeResponse request)
        {
            List<Dictionary<string, string>> error = new List<Dictionary<string, string>>();
            var oldEmployees = peremployeeRepository.GetEntities(t => t.AllotId == request.AllotId);

            for (int i = 0; i < oldEmployees.Count; i++)
            {
                if (oldEmployees.Count(w => w.PersonnelNumber == oldEmployees[i].PersonnelNumber) > 1)
                {
                    error.Add(new Dictionary<string, string>
                    {
                        { "行号", $"第{i+1}行" },
                        { "人员工号", oldEmployees[i].PersonnelNumber??"" },
                        { "姓名", oldEmployees[i].DoctorName??"" },
                        { "错误原因", "“人员工号”重复" },
                    });
                }
            }

            if (error.Count > 0)
                return new ApiResponse(ResponseType.WarningTable, "验证不通过,当前操作已拒绝", error);

            if (string.IsNullOrEmpty(request.UnitType?.Trim())
                || string.IsNullOrEmpty(request.Department?.Trim())
                || string.IsNullOrEmpty(request.AccountingUnit?.Trim())
                || string.IsNullOrEmpty(request.DoctorName?.Trim())
                || string.IsNullOrEmpty(request.PersonnelNumber?.Trim()))
            {
                throw new PerformanceException($"“关键信息缺失”请补全！");
            }

            var employee = peremployeeRepository.GetEntity(t => t.AllotId == request.AllotId && t.PersonnelNumber == request.PersonnelNumber);
            if (employee != null)
                throw new PerformanceException($"员工工号为{request.PersonnelNumber}的数据已存在！");

            var unittype = UnitTypeUtil.GetUnitTypeFromEnum().Select(t => t.Description).ToList();
            unittype.AddRange(EnumHelper.GetItems<AccountUnitType>().Select(t => t.Description));

            if (!unittype.Contains(request.UnitType))
                throw new PerformanceException($"核算组别不符合规范！");

            var entity = _mapper.Map<per_employee>(request);
            var allot = perallotRepository.GetEntity(t => t.ID == request.AllotId);
            int day = allot.Month >= 1 && allot.Month <= 12 ? DateTime.DaysInMonth(allot.Year, allot.Month) : 30;
            entity.Attendance = request.AttendanceDay / day;
            entity.CreateTime = DateTime.Now;
            entity.IsVerify = 1;

            //CheckAccountingDept(request.HospitalId.Value, request.AccountingUnit, request.Department);
            if (!peremployeeRepository.Add(entity))
                throw new PerformanceException($"添加失败");

            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);

            perallotRepository.AccoungtingVerify(entity.AllotId ?? 0);


            return new ApiResponse(ResponseType.OK, "添加成功", entity);
        }

        /// <summary>
        /// 修改员工信息
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public ApiResponse UpdatePerson(PerEmployeeResponse request)
        {
            if (string.IsNullOrEmpty(request.UnitType?.Trim())
                || string.IsNullOrEmpty(request.Department?.Trim())
                || string.IsNullOrEmpty(request.AccountingUnit?.Trim())
                || string.IsNullOrEmpty(request.DoctorName?.Trim())
                || string.IsNullOrEmpty(request.PersonnelNumber?.Trim()))
            {
                throw new PerformanceException($"“关键信息缺失”请补全！");
            }

            var employees = peremployeeRepository.GetEntities(t => t.AllotId == request.AllotId && t.Id != request.Id && t.PersonnelNumber == request.PersonnelNumber);
            if (employees != null && employees.Count > 0)
                throw new PerformanceException($"员工工号为“{request.PersonnelNumber}”已存在，请勿重复添加！");

            var unittype = UnitTypeUtil.GetUnitTypeFromEnum().Select(t => t.Description).ToList();
            unittype.AddRange(EnumHelper.GetItems<AccountUnitType>().Select(t => t.Description));

            if (!unittype.Contains(request.UnitType))
                throw new PerformanceException($"核算组别不符合规范！");

            var employee = peremployeeRepository.GetEntity(t => t.AllotId == request.AllotId && t.Id == request.Id);
            if (employee == null)
                throw new PerformanceException($"当前要修改的员工信息不存在，请检查后重试！");

            //_mapper.Map(request, employee, typeof(per_employee), typeof(per_employee));

            employee.AccountingUnit = request.AccountingUnit;
            employee.Department = request.Department;
            employee.DoctorName = request.DoctorName;
            employee.PersonnelNumber = request.PersonnelNumber;
            //employee.JobNumber = request.JobNumber; // his
            employee.JobCategory = request.JobCategory;
            employee.Duty = request.Duty;
            employee.JobTitle = request.JobTitle;
            employee.UnitType = request.UnitType;
            employee.WorkTime = request.WorkTime == null ? null : ConvertHelper.To<DateTime?>(request.WorkTime);
            employee.BirthDate = request.BirthDate == null ? null : ConvertHelper.To<DateTime?>(request.BirthDate);
            employee.Age = request.Age;
            employee.BankCard = request.BankCard;
            employee.Remark = request.Remark;
            employee.EfficiencyNumber = request.EfficiencyNumber;
            employee.PermanentStaff = request.PermanentStaff;
            employee.AttendanceDay = request.AttendanceDay;
            employee.ReservedRatio = request.ReservedRatio;

            // 出勤率
            var allot = perallotRepository.GetEntity(t => t.ID == request.AllotId);
            var day = allot.Month >= 1 && allot.Month <= 12 ? DateTime.DaysInMonth(allot.Year, allot.Month) : 30m;

            employee.Attendance = Math.Round((request.AttendanceDay ?? 0) / day, 4);

            //CheckAccountingDept(request.HospitalId.Value, request.AccountingUnit, request.Department);
            var res = peremployeeRepository.Update(employee);

            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);

            #region 修改成功，但存在其他问题请及时处理
            List<Dictionary<string, string>> error = new List<Dictionary<string, string>>();
            var oldEmployees = peremployeeRepository.GetEntities(t => t.AllotId == request.AllotId);

            for (int i = 0; i < oldEmployees.Count; i++)
            {
                if (oldEmployees.Count(w => w.PersonnelNumber == oldEmployees[i].PersonnelNumber) > 1)
                {
                    error.Add(new Dictionary<string, string>
                    {
                        { "行号", $"第{i+1}行" },
                        { "人员工号", oldEmployees[i].PersonnelNumber??"" },
                        { "姓名", oldEmployees[i].DoctorName??"" },
                        { "错误原因", "“人员工号”重复" },
                    });
                }
            }

            if (error.Count > 0)
                return new ApiResponse(ResponseType.WarningTable, "修改成功，但存在其他问题请及时处理", error);
            #endregion

            return new ApiResponse(ResponseType.OK, "修改成功");
        }

        /// <summary>
        /// 删除员工
        /// </summary>
        /// <param name="employeeId"></param>
        /// <returns></returns>
        public bool DeletePerson(int employeeId)
        {
            var employee = peremployeeRepository.GetEntity(t => t.Id == employeeId);
            if (employee == null)
                throw new PerformanceException($"员工信息错误！");

            var oldbackup = perforPeremployeeBackupRepository.GetEntity(t => t.Id == employee.Id);
            if (oldbackup != null)
                perforPeremployeeBackupRepository.Remove(oldbackup);

            var employeeBackup = _mapper.Map<per_employee_backup>(employee);
            perforPeremployeeBackupRepository.Add(employeeBackup);

            var allot = perallotRepository.GetEntity(t => t.ID == employee.AllotId);
            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);
            return peremployeeRepository.Remove(employee);
        }

        /// <summary>
        /// 删除员工
        /// </summary>
        /// <param name="employeeId"></param>
        /// <returns></returns>
        public bool DeleteAllPerson(int allotId)
        {
            var employees = peremployeeRepository.GetEntities(t => t.AllotId == allotId)
                ?? new List<per_employee>();

            dapperService.PerEmployeeBackup(allotId);
            var allot = perallotRepository.GetEntity(t => t.ID == allotId);
            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);
            return peremployeeRepository.RemoveRange(employees.ToArray());
        }

        #region 弃用

        ///// <summary>
        ///// 检查核算单元是否已存在
        ///// </summary>
        ///// <param name="hosapitalId">医院Id</param>
        ///// <param name="accountingUnit">核算单元</param>
        ///// <param name="department">科室</param>
        //private void CheckAccountingDept(int hosapitalId, string accountingUnit, string department)
        //{
        //    var accountDept = perdeptdicRepository.GetEntity(t => t.HospitalId == hosapitalId && t.AccountingUnit == accountingUnit && t.Department == department);
        //    if (accountDept == null)
        //    {
        //        var entity = new per_dept_dic
        //        {
        //            AccountingUnit = accountingUnit,
        //            Department = department,
        //            HospitalId = hosapitalId,
        //            Source = "人事科增加人员记录时补充"
        //        };
        //        perdeptdicRepository.Add(entity);
        //    }
        //}

        /// <summary>
        /// 获取给医院所有的
        /// </summary>
        /// <param name="allotId"></param>
        /// <returns></returns>
        //public IEnumerable<DeptdicResponse> GetDepartments(int allotId)
        //{
        //    var depts = perdeptdicRepository.GetEntities(t => t.AllotId == allotId);
        //    if (depts == null || !depts.Any()) return null;

        //    var allot = perallotRepository.GetEntity(t => t.ID == allotId);
        //    if (allot == null || !depts.Any()) return null;

        //    Deptdic GetDeptdic(List<per_dept_dic> dics, string department, string hISDeptName, UnitType unitType, string source = "")
        //    {
        //        var dic = dics.FirstOrDefault(group => group.Department == department && group.HISDeptName == hISDeptName && group.UnitType == unitType.ToString() && group.Source == source);
        //        if (dic == null)
        //            dic = dics.FirstOrDefault(group => group.Department == department && group.HISDeptName == hISDeptName && group.UnitType == unitType.ToString());
        //        if (dic == null) return new Deptdic() { IsVerify = 1 };
        //        return new Deptdic { Id = dic.Id, AccountingUnit = dic.AccountingUnit, IsVerify = dic.IsVerify ?? 1, VerifyMessage = dic.VerifyMessage, };
        //    }

        //    var result = depts
        //        .GroupBy(t => new { t.HISDeptName, t.Department })
        //        .Select(t => new DeptdicResponse
        //        {
        //            HospitalId = allot.HospitalId,
        //            HISDeptName = t.Key.HISDeptName,
        //            Department = t.Key.Department,
        //            OutDoctorAccounting = GetDeptdic(t.ToList(), t.Key.Department, t.Key.HISDeptName, UnitType.医生组, "门诊"),
        //            OutNurseAccounting = GetDeptdic(t.ToList(), t.Key.Department, t.Key.HISDeptName, UnitType.护理组, "门诊"),
        //            OutTechnicAccounting = GetDeptdic(t.ToList(), t.Key.Department, t.Key.HISDeptName, UnitType.医技组, "门诊"),
        //            InpatDoctorAccounting = GetDeptdic(t.ToList(), t.Key.Department, t.Key.HISDeptName, UnitType.医生组, "住院"),
        //            InpatNurseAccounting = GetDeptdic(t.ToList(), t.Key.Department, t.Key.HISDeptName, UnitType.护理组, "住院"),
        //            InpatTechnicAccounting = GetDeptdic(t.ToList(), t.Key.Department, t.Key.HISDeptName, UnitType.医技组, "住院"),
        //            LogisticsAccounting = GetDeptdic(t.ToList(), t.Key.Department, t.Key.HISDeptName, UnitType.行政工勤),
        //            SpecialAccounting = GetDeptdic(t.ToList(), t.Key.Department, t.Key.HISDeptName, UnitType.特殊核算组),
        //            CreateTime = t.Max(group => group.CreateTime),
        //            IsVerify = t.Min(w => w.IsVerify) ?? 1,
        //        });

        //    return result.OrderBy(w => w.IsVerify).ThenByDescending(t => t.CreateTime).ThenBy(t => t.Department);
        //}

        //private (string dept, string[] unittype) GetDeptByUser(int userId)
        //{
        //    var user = perforUserRepository.GetEntity(t => t.ID == userId && t.IsDelete == 1);
        //    if (user == null) throw new PerformanceException("用户信息错误");

        //    var userrole = perforUserroleRepository.GetEntity(t => t.UserID == user.ID);
        //    var role = perforRoleRepository.GetEntity(t => t.ID == userrole.RoleID);

        //    if (role == null) return ("", new string[] { });

        //    if (UnitTypeUtil.Maps.ContainsKey(role.Type.Value))
        //    {
        //        return (user.Department, UnitTypeUtil.Maps[role.Type.Value]);
        //    }

        //    return ("", new string[] { });
        //}

        #endregion

        /// <summary>
        /// 创建科室核算信息
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public per_dept_dic CreateDeptDic(per_dept_dic request)
        {
            var deptdic = perdeptdicRepository.GetEntity(t =>
                t.AllotId == request.AllotId
                && t.AccountingUnit == request.AccountingUnit
                && t.Department == request.Department
                && t.HISDeptName == request.HISDeptName
                && t.Source == request.Source
                && t.UnitType == request.UnitType);

            if (deptdic != null)
                throw new PerformanceException($"{request.Department}数据重复！");

            request.CreateTime = DateTime.Now;
            perdeptdicRepository.Add(request);

            var allot = perallotRepository.GetEntity(t => t.ID == request.AllotId);
            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);
            return request;
        }

        ///// <summary>
        ///// 更新科室核算信息
        ///// </summary>
        ///// <param name="request"></param>
        ///// <returns></returns>
        //public bool UpdateDeptDic(per_dept_dic request)
        //{
        //    var deptdic = perdeptdicRepository.GetEntity(t =>
        //        t.HospitalId == request.HospitalId
        //        && t.AllotId == request.AllotId
        //        && t.AccountingUnit == request.AccountingUnit
        //        && t.Department == request.Department
        //        && t.Source == request.Source
        //        && t.UnitType == request.UnitType);

        //    if (deptdic != null && deptdic.Id != request.Id)
        //        throw new PerformanceException($"{request.Department}数据重复！");

        //    _mapper.Map(request, deptdic, typeof(per_dept_dic), typeof(per_dept_dic));
        //    return perdeptdicRepository.Add(deptdic);
        //}

        /// <summary>
        /// 更新科室核算信息
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public bool UpdateDeptDic(DeptdicResponse request)
        {
            try
            {
                Dictionary<string, (string, string)> dict = new Dictionary<string, (string, string)>
                {
                    { nameof(DeptdicResponse.OutDoctorAccounting), (UnitType.医生组.ToString(), "门诊") },
                    { nameof(DeptdicResponse.OutNurseAccounting), (UnitType.护理组.ToString(), "门诊") },
                    { nameof(DeptdicResponse.OutTechnicAccounting), (UnitType.医技组.ToString(), "门诊") },
                    { nameof(DeptdicResponse.InpatDoctorAccounting), (UnitType.医生组.ToString(), "住院") },
                    { nameof(DeptdicResponse.InpatNurseAccounting), (UnitType.护理组.ToString(), "住院") },
                    { nameof(DeptdicResponse.InpatTechnicAccounting), (UnitType.医技组.ToString(), "住院") },
                    { nameof(DeptdicResponse.LogisticsAccounting), (UnitType.行政工勤.ToString(), null) },
                    { nameof(DeptdicResponse.SpecialAccounting), (UnitType.特殊核算组.ToString(), null) },
                };
                foreach (var item in dict)
                {
                    var property = typeof(DeptdicResponse).GetProperty(item.Key);
                    if (property == null) continue;

                    var deptdic = property.GetValue(request);
                    if (deptdic is Deptdic)
                    {
                        var data = (Deptdic)deptdic;
                        if (string.IsNullOrEmpty(data.AccountingUnit) && data.Id == 0)
                            continue;

                        if (data.Id > 0)
                        {
                            var entity = perdeptdicRepository.GetEntity(t => t.Id == data.Id);
                            if (entity == null) continue;
                            entity.HISDeptName = request.HISDeptName;
                            entity.Department = request.Department;
                            entity.AccountingUnit = data.AccountingUnit;
                            perdeptdicRepository.Update(entity);
                        }
                        else
                        {
                            var entity = new per_dept_dic
                            {
                                AllotId = request.AllotId,
                                HospitalId = request.HospitalId,
                                HISDeptName = request.HISDeptName,
                                Department = request.Department,
                                AccountingUnit = data.AccountingUnit,
                                UnitType = item.Value.Item1,
                                Source = item.Value.Item2,
                                CreateTime = DateTime.Now,
                            };
                            perdeptdicRepository.Add(entity);
                        }
                    }
                }
                var allot = perallotRepository.GetEntity(t => t.ID == request.AllotId);
                allot.IsModifyConfig = 1;
                perallotRepository.Update(allot);
            }
            catch (Exception ex)
            {
                logger.LogError(ex.ToString());
                return false;
            }

            return true;
        }

        /// <summary>
        /// 删除科室核算信息
        /// </summary>
        /// <param name="deptdicId"></param>
        /// <returns></returns>
        public bool DeleteDeptDic(DeptdicResponse request)
        {
            if (request == null)
                throw new PerformanceException("科室记录不存在!");

            var deptdics = perdeptdicRepository.GetEntities(t => t.AllotId == request.AllotId && t.HISDeptName == request.HISDeptName && t.Department == request.Department);
            if (deptdics == null || !deptdics.Any())
                throw new PerformanceException("科室记录不存在!");

            var allot = perallotRepository.GetEntity(t => t.ID == request.AllotId);
            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);
            return perdeptdicRepository.RemoveRange(deptdics.ToArray());
        }

        /// <summary>
        /// 系统/标准科室字典
        /// </summary>
        /// <param name="allotId">allotId</param>
        /// <param name="type">1系统科室 2标准科室 3核算单元</param>
        /// <returns></returns>
        public List<TitleValue> DeptDics(int allotId, int type)
        {
            var deptdics = perdeptdicRepository.GetEntities(t => t.AllotId == allotId);
            if (deptdics == null || !deptdics.Any()) return new List<TitleValue>();

            var result = new List<string>();
            if (type == 1)
            {
                result = deptdics.Select(t => t.HISDeptName).Distinct().ToList();
            }
            else if (type == 2)
            {
                result = deptdics.Select(t => t.Department).Distinct().ToList();
            }
            else if (type == 3)
            {
                result = deptdics.Where(t => UnitTypeUtil.Office.Contains(t.UnitType) || UnitType.特殊核算组.ToString() == t.UnitType).Select(t => t.AccountingUnit).Distinct().ToList();
            }
            else if (type == 4)
            {
                result = deptdics.Where(t => UnitTypeUtil.Office.Contains(t.UnitType)).Select(t => t.AccountingUnit).Distinct().ToList();
            }
            else if (type == 5)
            {
                result = deptdics.Where(t => t.UnitType == UnitType.特殊核算组.ToString()).Select(t => t.AccountingUnit).Distinct().ToList();
            }
            return result.Select(t => new TitleValue { Title = t, Value = t }).ToList();
        }

        /// <summary>
        /// 科室工作量数据详情
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public object DeptWorkloadDetail(WorkDetailRequest request, int userId)
        {
            var userInfo = perforUserRepository.GetUser(userId);
            if (userInfo?.User == null) throw new NotImplementedException("当前用户不存在");
            if (userInfo.URole == null) throw new NotImplementedException("当前用户暂未分配角色");
            if (!userInfo.Hospitals.NotNullOrEmpty()) throw new NotImplementedException("当前用户暂未分配医院");


            // 查询当前角色下科室的绩效
            UnitTypeUtil.Maps.TryGetValue(userInfo?.URole.Type ?? 0, out string[] unitTypes);
            if (unitTypes == null || !unitTypes.Any()) return Enumerable.Empty<DeptIncomeResponse>();
            var accountingUnit = request.AccountingUnit;
            var hospitalId = userInfo.HospitalIds.First();
            if (userInfo.IsSecondAdmin)
            {
                var second = agsecondallotRepository.Get(request.AllotId, unitTypes, userInfo.User.Department)?.FirstOrDefault();
                if (second != null)
                {
                    accountingUnit = second.Department ?? request.AccountingUnit;
                    unitTypes = new string[] { second.UnitType };
                }
            }

            var data = perallotRepository.QueryWorkloadData(request.AllotId, accountingUnit, unitTypes, hospitalId);
            if (data != null && data.Any())
            {
                return data.GroupBy(t => new { t.Department, t.DoctorName, t.PersonnelNumber, t.Category })
                    .Select(t => new DeptIncomeResponse
                    {
                        Department = t.Key.Department,
                        DoctorName = t.Key.DoctorName,
                        PersonnelNumber = t.Key.PersonnelNumber,
                        Category = t.Key.Category,
                        Fee = t.Sum(group => group.Fee ?? 0)
                    }).OrderBy(t => t.PersonnelNumber).ThenBy(t => t.Category);
            }

            return new string[] { };
        }

        /// <summary>
        /// 科室工作量数据详情
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public IEnumerable<DeptIncomeResponse> DeptIncomeDetail(WorkDetailRequest request, int userId)
        {
            var userInfo = perforUserRepository.GetUser(userId);
            if (userInfo?.User == null) throw new NotImplementedException("当前用户不存在");
            if (userInfo.URole == null) throw new NotImplementedException("当前用户暂未分配角色");
            if (!userInfo.Hospitals.NotNullOrEmpty()) throw new NotImplementedException("当前用户暂未分配医院");

            // 查询当前角色下科室的绩效
            UnitTypeUtil.Maps.TryGetValue(userInfo?.URole.Type ?? 0, out string[] unitTypes);
            if (unitTypes == null || !unitTypes.Any()) return Enumerable.Empty<DeptIncomeResponse>();

            var accountingUnit = request.AccountingUnit;
            var hospitalId = userInfo.HospitalIds.First();
            if (userInfo.IsSecondAdmin)
            {
                var second = agsecondallotRepository.Get(request.AllotId, unitTypes, userInfo.User.Department)?.FirstOrDefault();
                if (second != null)
                {
                    accountingUnit = second.Department ?? request.AccountingUnit;
                    unitTypes = new string[] { second.UnitType };
                }
            }


            var sources = new[] { "门诊", "住院" };
            if (!sources.Contains(request.Source))
                throw new PerformanceException($"数据来源错误，只支持：{string.Join(";", sources)}");

            var data = perallotRepository.QueryIncomeData(request.AllotId, request.Source, accountingUnit, unitTypes, hospitalId);
            if (data != null && data.Any())
            {
                return data.GroupBy(t => new { t.Department, t.DoctorName, t.PersonnelNumber, t.Category })
                    .Select(t => new DeptIncomeResponse
                    {
                        Department = t.Key.Department,
                        DoctorName = t.Key.DoctorName,
                        PersonnelNumber = t.Key.PersonnelNumber,
                        Category = t.Key.Category,
                        Fee = t.Sum(group => group.Fee ?? 0)
                    }).OrderBy(t => t.PersonnelNumber).ThenBy(t => t.Category);
            }

            return Enumerable.Empty<DeptIncomeResponse>();
        }


        public HandsonTableBase GetBatchPersonStructrue(int allotId)
        {
            HandsonTableBase result = new HandsonTableBase()
            {
                Columns = Person.Select(t => new HandsonColumn(t.Item2)).ToList(),
                ColHeaders = Person.Select(t => t.Item2).ToList(),
            };

            var deptdics = perdeptdicRepository.GetEntities(t => t.AllotId == allotId);
            var ss = deptdics?.Where(t => !new string[] { "专家组" }.Contains(t.UnitType));
            if (result.Columns != null && result.Columns.Any())
            {
                var columns = new[] { "核算单元", "科室名称", "姓名", "员工工号", "银行卡号" };
                foreach (var column in result.Columns)
                {
                    if (columns.Contains(column.Data))
                        column.Type = "text";

                    if (column.Data == "核算组别")
                    {
                        column.Type = "autocomplete";
                        column.Source = UnitTypeUtil.GetUnitTypeFromEnum().Select(w => w.Description).ToArray();
                        column.Strict = true;
                    }
                    else if (new[] { "出勤天数", "预留比例" }.Contains(column.Data))
                    {
                        column.Type = "numeric";
                        column.NumericFormat = new NumericFormat { Pattern = "0,00" };
                    }
                }
            }
            return result;
        }

        public HandsonTableBase GetDepartmentHands(int allotId)
        {
            HandsonTableBase handson = new HandsonTableBase
            {
                Columns = DeptDic.Select(t => new HandsonColumn(t.Value)).ToList(),
                ColHeaders = DeptDic.Select(c => c.Value).ToList(),
            };
            #region 科室下拉
            //if (handson.Columns != null && handson.Columns.Any())
            //{
            //    foreach (var column in handson.Columns)
            //    {
            //        if (column.Data == "标准科室")
            //        {
            //            column.Type = "autocomplete";
            //            column.Source = DeptDics(hospitalId, 2).Select(c => c.Value).ToArray();
            //            column.Strict = true;
            //        }
            //        else if (column.Data == "系统科室")
            //        {
            //            column.Type = "autocomplete";
            //            column.Source = DeptDics(hospitalId, 1).Select(c => c.Value).ToArray();
            //            column.Strict = true;
            //        }
            //        else if (new[] { "住院·核算单元医生组", "住院·核算单元护理组", "住院·核算单元医技组", "门诊·核算单元医生组", "门诊·核算单元护理组", "门诊·核算单元医技组" }.Contains(column.Data))
            //        {
            //            column.Type = "autocomplete";
            //            column.Source = DeptDics(hospitalId, 3).Select(c => c.Value).ToArray();
            //            column.Strict = true;
            //        }
            //        else if (column.Data == "行政后勤")
            //        {
            //            column.Type = "autocomplete";
            //            column.Source = DeptDics(hospitalId, 4).Select(c => c.Value).ToArray();
            //            column.Strict = true;
            //        }
            //        else if (column.Data == "特殊核算组")
            //        {
            //            column.Type = "autocomplete";
            //            column.Source = DeptDics(hospitalId, 5).Select(c => c.Value).ToArray();
            //            column.Strict = true;
            //        }
            //    }
            //}
            #endregion
            return handson;
        }

        public ApiResponse BathSavePerson(int allotId, int HospitalId, SaveCollectData request)
        {
            var dict = new Dictionary<string, string>();
            Person.ForEach(t => dict.Add(t.Item1, t.Item2));

            var dicData = CreateDataRow(request, dict);
            if (dicData == null || dicData.Count == 0)
                return new ApiResponse(ResponseType.Error, "空数据，无效操作");

            var jsons = JsonHelper.Serialize(dicData);
            var newEmployees = JsonHelper.Deserialize<List<per_employee>>(jsons);
            var oldEmployees = peremployeeRepository.GetEntities(t => t.HospitalId == HospitalId && t.AllotId == allotId);
            var accountinglist = perforCofaccountingRepository.GetEntities(t => t.AllotId == allotId);
            List<Dictionary<string, string>> error = new List<Dictionary<string, string>>();

            for (int i = 0; i < newEmployees.Count; i++)
            {
                if (string.IsNullOrEmpty(newEmployees[i].UnitType?.Trim())
                    || string.IsNullOrEmpty(newEmployees[i].Department?.Trim())
                    || string.IsNullOrEmpty(newEmployees[i].AccountingUnit?.Trim())
                    || string.IsNullOrEmpty(newEmployees[i].DoctorName?.Trim())
                    || string.IsNullOrEmpty(newEmployees[i].PersonnelNumber?.Trim()))
                {
                    error.Add(new Dictionary<string, string>
                    {
                        { "行号", $"第{i+1}行" },
                        { "人员工号", newEmployees[i].PersonnelNumber??"" },
                        { "姓名", newEmployees[i].DoctorName??"" },
                        { "来源", "粘贴数据" },
                        { "错误原因", "“关键信息缺失”请补全或删除" },
                    });
                }
                newEmployees[i].UnitType = newEmployees[i].UnitType?.Replace("行政后勤", "行政工勤");
                if (!Enum.IsDefined(typeof(UnitType), newEmployees[i].UnitType?.Trim()))
                {
                    error.Add(new Dictionary<string, string>
                    {
                        { "行号", $"第{i+1}行" },
                        { "人员工号", newEmployees[i].PersonnelNumber??"" },
                        { "姓名", newEmployees[i].DoctorName??"" },
                        { "来源", "粘贴数据" },
                        { "错误原因", "“核算组别”错误，请修改或删除" },
                    });
                }
                if (newEmployees.Count(w => w.PersonnelNumber == newEmployees[i].PersonnelNumber) > 1)
                {
                    error.Add(new Dictionary<string, string>
                    {
                        { "行号", $"第{i+1}行" },
                        { "人员工号", newEmployees[i].PersonnelNumber??"" },
                        { "姓名", newEmployees[i].DoctorName??"" },
                        { "来源", "粘贴数据" },
                        { "错误原因", "“人员工号”重复" },
                    });
                }

                var oldEmp = oldEmployees.FirstOrDefault(w => w.PersonnelNumber == newEmployees[i].PersonnelNumber);
                if (!string.IsNullOrEmpty(newEmployees[i].DoctorName) && oldEmp != null && oldEmp.DoctorName != newEmployees[i].DoctorName)
                {
                    error.Add(new Dictionary<string, string>
                    {
                        { "行号", $"第{i+1}行" },
                        { "人员工号", newEmployees[i].PersonnelNumber??"" },
                        { "姓名", newEmployees[i].DoctorName??"" },
                        { "来源", "“粘贴数据”与“历史数据”比对" },
                        { "错误原因", $"原名“{oldEmp.DoctorName}”,工号相同但姓名不同,请删除“历史数据”中该员工" },
                    });
                }

                var accountingUnit = accountinglist.FirstOrDefault(w => w.UnitType == newEmployees[i].UnitType && w.AccountingUnit == newEmployees[i].AccountingUnit);
                if (accountingUnit == null)
                {
                    error.Add(new Dictionary<string, string>
                    {
                        { "行号", $"第{i+1}行" },
                        { "人员工号", newEmployees[i].PersonnelNumber??"" },
                        { "姓名", newEmployees[i].DoctorName??"" },
                        { "来源", "粘贴数据" },
                        { "错误原因", $"核算单元及组别不存在" },
                    });
                }
            }


            for (int i = 0; i < oldEmployees.Count; i++)
            {
                if (oldEmployees.Count(w => w.PersonnelNumber == oldEmployees[i].PersonnelNumber) > 1)
                {
                    error.Add(new Dictionary<string, string>
                    {
                        { "行号", $"第{i+1}行" },
                        { "人员工号", oldEmployees[i].PersonnelNumber??"" },
                        { "姓名", oldEmployees[i].DoctorName??"" },
                        { "来源", "历史数据" },
                        { "错误原因", "“人员工号”重复" },
                    });
                }
            }

            if (error.Count > 0)
                return new ApiResponse(ResponseType.WarningTable, "验证不通过,当前操作已拒绝", error);


            List<per_employee> employees = new List<per_employee>();
            var delPersonsNum = new List<string>();
            foreach (var data in newEmployees)
            {
                var any = employees.Any(w => w.UnitType?.Trim() == data.UnitType?.Trim() && w.Department?.Trim() == data.Department?.Trim() && w.DoctorName?.Trim() == data.DoctorName?.Trim() && w.PersonnelNumber == data.PersonnelNumber);

                if (!string.IsNullOrEmpty(data.Department?.Trim()) && !string.IsNullOrEmpty(data.AccountingUnit?.Trim()) && !string.IsNullOrEmpty(data.DoctorName?.Trim()) && !any)
                {
                    if (oldEmployees != null && oldEmployees.Any(t => t.PersonnelNumber?.Trim() == data.PersonnelNumber?.Trim()))
                        delPersonsNum.Add(data.PersonnelNumber);

                    data.UnitType = data.UnitType.Replace("行政后勤", "行政工勤");
                    data.HospitalId = HospitalId;
                    data.AllotId = allotId;
                    data.DoctorName = data.DoctorName?.Trim();
                    data.PersonnelNumber = data.PersonnelNumber?.Trim();
                    data.CreateTime = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd hh:mm:dd"));
                    data.IsVerify = 1;
                    employees.Add(data);
                }
            }

            var delPerson = oldEmployees?.Where(t => delPersonsNum.Contains(t.PersonnelNumber));

            if (delPerson != null && delPerson.Any())
            {
                peremployeeRepository.RemoveRange(delPerson.ToArray());
                var backDataup = _mapper.Map<List<per_employee_backup>>(delPerson);
                perforPeremployeeBackupRepository.AddRange(backDataup.ToArray());
            }

            if (employees != null && employees.Any())
            {
                var backupTab = perforPeremployeeBackupRepository.GetEntities(t => t.AllotId == allotId);
                employees.ForEach(e =>
                {
                    var result = backupTab.FirstOrDefault(d => d.PersonnelNumber.Replace(" ", "") == e.PersonnelNumber.Replace(" ", "") && d.AllotId == e.AllotId && d.HospitalId == e.HospitalId);
                    if (result != null)
                    {
                        e.JobNumber = result.JobNumber;
                        e.JobCategory = result.JobCategory;
                        e.Duty = result.Duty;
                        e.JobTitle = result.JobTitle;
                        e.Attendance = result.Attendance;
                        e.AttendanceDay = result.AttendanceDay;
                        e.PermanentStaff = result.PermanentStaff;
                        e.EfficiencyNumber = result.EfficiencyNumber;
                        e.WorkTime = result.WorkTime;
                        e.BirthDate = result.BirthDate;
                        e.Age = result.Age;
                        e.ReservedRatio = result.ReservedRatio;
                        e.BankCard = result.BankCard;
                        e.Remark = result.Remark;
                        e.IsVerify = result.IsVerify;
                        e.VerifyMessage = result.VerifyMessage;
                        e.IsHrpEmployee = result.IsHrpEmployee;
                        e.CreateUser = result.CreateUser;
                        e.Reserve01 = result.Reserve01;
                        e.Reserve02 = result.Reserve02;
                        e.Reserve03 = result.Reserve03;
                        e.Reserve04 = result.Reserve04;
                        e.Reserve05 = result.Reserve05;
                        e.Reserve06 = result.Reserve06;
                        e.Reserve07 = result.Reserve07;
                        e.Reserve08 = result.Reserve08;
                        e.Reserve09 = result.Reserve09;
                        e.Reserve10 = result.Reserve10;
                    }
                });

                peremployeeRepository.AddRange(employees.ToArray());
            }

            var allot = perallotRepository.GetEntity(t => t.ID == allotId);
            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);

            perallotRepository.AccoungtingVerify(allotId);
            return new ApiResponse(ResponseType.OK, "");
        }

        public void SaveDeptDicHands(int allotId, SaveCollectData request)
        {
            var allot = perallotRepository.GetEntity(t => t.ID == allotId);
            if (allot == null) throw new PerformanceException("绩效记录不存在");

            var dicData = CreateDataRow(request, DeptDic);
            var deptDic = perdeptdicRepository.GetEntities(t => t.AllotId == allotId);
            var depts = deptDic?.Select(w => new { w.Department, w.HISDeptName }).Distinct();
            List<per_dept_dic> deptDics = new List<per_dept_dic>();
            var delDepartment = new List<string>();
            foreach (var dic in dicData)
            {
                var json = JsonHelper.Serialize(dic);
                var data = JsonHelper.Deserialize<DeptdicHands>(json);

                if (depts != null)
                    if (depts.Any(t => t.HISDeptName?.Trim() == data.HISDeptName?.Trim())) delDepartment.Add(data.HISDeptName);
                var any = deptDics.Any(w => w.HISDeptName?.Trim() == data.HISDeptName?.Trim());

                if (!string.IsNullOrEmpty(data.HISDeptName?.Trim()) && !any)
                {
                    DeptDicList(allot.HospitalId, allotId, deptDics, data);
                }

            }
            var delDept = deptDic?.Where(t => delDepartment.Contains(t.HISDeptName));
            if (delDept != null && delDept.Any())
                perdeptdicRepository.RemoveRange(delDept.ToArray());
            if (deptDics != null && deptDics.Any())
                perdeptdicRepository.AddRange(deptDics.ToArray());

            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);
        }

        private void DeptDicList(int hospitalId, int allotId, List<per_dept_dic> deptDics, DeptdicHands data)
        {
            var nowTime = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd hh:mm:ss"));

            Func<string, string, string, per_dept_dic> getDeptDic = (source, unittype, accountingUnit) =>
            {
                return new per_dept_dic()
                {
                    Source = source,
                    HospitalId = hospitalId,
                    AllotId = allotId,
                    Department = data.Department?.Trim(),
                    HISDeptName = data.HISDeptName?.Trim(),
                    CreateTime = nowTime,
                    AccountingUnit = accountingUnit,
                    UnitType = unittype,
                };
            };

            if (!string.IsNullOrEmpty(data.InpatDoctorAccounting))
                deptDics.Add(getDeptDic("住院", "医生组", data.InpatDoctorAccounting?.Trim()));

            if (!string.IsNullOrEmpty(data.InpatNurseAccounting))
                deptDics.Add(getDeptDic("住院", "护理组", data.InpatNurseAccounting?.Trim()));

            if (!string.IsNullOrEmpty(data.InpatTechnicAccounting))
                deptDics.Add(getDeptDic("住院", "医技组", data.InpatTechnicAccounting?.Trim()));

            if (!string.IsNullOrEmpty(data.OutDoctorAccounting))
                deptDics.Add(getDeptDic("门诊", "医生组", data.OutDoctorAccounting?.Trim()));

            if (!string.IsNullOrEmpty(data.OutNurseAccounting))
                deptDics.Add(getDeptDic("门诊", "护理组", data.OutNurseAccounting?.Trim()));

            if (!string.IsNullOrEmpty(data.OutTechnicAccounting))
                deptDics.Add(getDeptDic("门诊", "医技组", data.OutTechnicAccounting?.Trim()));

            if (!string.IsNullOrEmpty(data.LogisticsAccounting))
                deptDics.Add(getDeptDic("", "行政后勤", data.LogisticsAccounting?.Trim()));

            if (!string.IsNullOrEmpty(data.SpecialAccounting))
                deptDics.Add(getDeptDic("", "特殊核算组", data.SpecialAccounting?.Trim()));
        }

        private List<Dictionary<string, string>> CreateDataRow(SaveCollectData request, Dictionary<string, string> config)
        {
            List<Dictionary<string, string>> allData = new List<Dictionary<string, string>>();

            for (int r = 0; r < request.Data.Length; r++)
            {
                // 创建固定数据列
                Dictionary<string, string> baseData = CreateBaseData(request, config, r);
                allData.Add(baseData);

            }
            return allData;
        }

        private Dictionary<string, string> CreateBaseData(SaveCollectData request, Dictionary<string, string> config, int rownumber)
        {
            Dictionary<string, string> result = new Dictionary<string, string>();
            for (int c = 0; c < request.ColHeaders.Length; c++)
            {
                var header = request.ColHeaders[c];
                var first = config.FirstOrDefault(w => w.Value == header);
                if (!default(KeyValuePair<string, string>).Equals(first)
                    && !result.ContainsKey(header)
                    && request.Data[rownumber].Length > c)
                {
                    result.Add(first.Key, request.Data[rownumber][c]);
                }
            }

            return result;
        }

        public static List<(string, string, Func<per_employee, object>)> Person { get; } = new List<(string, string, Func<per_employee, object>)>
        {
            (nameof(per_employee.UnitType), "核算组别", t => t.UnitType),
            (nameof(per_employee.AccountingUnit), "核算单元", t => t.AccountingUnit),
            (nameof(per_employee.Department), "科室名称", t => t.Department),
            (nameof(per_employee.DoctorName), "姓名" ,t => t.DoctorName),
            (nameof(per_employee.PersonnelNumber), "员工工号", t => t.PersonnelNumber),
            //(nameof(per_employee.JobCategory), "正式/临聘", t => t.JobCategory),
            //(nameof(per_employee.Duty), "职务", t => t.Duty),
            //(nameof(per_employee.JobTitle), "职称", t => t.JobTitle),
            //(nameof(per_employee.AttendanceDay), "出勤天数", t => t.AttendanceDay),
            //(nameof(per_employee.ReservedRatio), "预留比例", t => t.ReservedRatio),
            //(nameof(per_employee.BankCard), "银行卡号", t => t.BankCard),
            //(nameof(per_employee.Remark), "备注", t => t.Remark),
        };

        private static Dictionary<string, string> DeptDic { get; } = new Dictionary<string, string>
        {
            { nameof(DeptdicResponse.Department), "标准科室" },
            { nameof(DeptdicResponse.HISDeptName), "系统科室" },
            { nameof(DeptdicResponse.OutDoctorAccounting), "门诊·核算单元医生组" },
            { nameof(DeptdicResponse.OutNurseAccounting), "门诊·核算单元护理组" },
            { nameof(DeptdicResponse.OutTechnicAccounting), "门诊·核算单元医技组" },
            { nameof(DeptdicResponse.InpatDoctorAccounting), "住院·核算单元医生组" },
            { nameof(DeptdicResponse.InpatNurseAccounting), "住院·核算单元护理组" },
            { nameof(DeptdicResponse.InpatTechnicAccounting), "住院·核算单元医技组" },
            { nameof(DeptdicResponse.LogisticsAccounting), "行政工勤" },
            { nameof(DeptdicResponse.SpecialAccounting), "特殊核算组" }
        };

        public string GetPersonDictFile(int allotId, int userId)
        {
            var allot = perallotRepository.GetEntity(t => t.ID == allotId);
            if (allot == null)
                throw new PerformanceException("绩效记录不存在");

            var data = GetPersons(allotId, userId) ?? new List<per_employee>();

            var dpath = Path.Combine(evn.ContentRootPath, "Files", "Dictionary", $"{allot.HospitalId}");
            FileHelper.CreateDirectory(dpath);

            string filename = $"{allot.Year}{allot.Month.ToString().PadLeft(2, '0')}人员字典-{DateTime.Now.ToString("yyyyMMddhhmmss")}.xlsx";
            string filepath = Path.Combine(dpath, filename);
            FileStream stream = new FileStream(filepath, FileMode.Create);
            try
            {
                XSSFWorkbook workbook = new XSSFWorkbook();
                ExcelStyle excelStyle = new ExcelStyle(workbook);
                var style = excelStyle.SetBgkColorAndFormat(excelStyle.GetCellStyle());

                ISheet sheet = workbook.CreateSheet("人员字典");

                var header = sheet.CreateRow(0);
                int cellIndex = 0;
                foreach (var column in Person.Select(t => t.Item2))
                {
                    var cell = header.CreateCell(cellIndex);
                    cell.SetCellValue(column);
                    cell.CellStyle = style;
                    cellIndex++;
                }

                int startIndex = 1;
                foreach (var item in data)
                {
                    var row = sheet.CreateRow(startIndex);
                    cellIndex = 0;
                    foreach (var field in Person.Select(t => t.Item3))
                    {
                        var cell = row.CreateCell(cellIndex);
                        cell.SetCellOValue(field?.Invoke(item));
                        cell.CellStyle = style;
                        cellIndex++;
                    }
                    startIndex++;
                }

                workbook.Write(stream);
            }
            catch (Exception ex)
            {
                Console.WriteLine("写入异常" + ex);
            }
            finally
            {
                stream.Close();
            }
            return filepath;
        }

        public bool ReloadPersonnel(int hospitalId, int allotId)
        {
            var allotList = perallotRepository.GetEntities(w => w.HospitalId == hospitalId)?.OrderBy(s => s.Year).ThenBy(s => s.Month).ToList();
            if (allotList == null || !allotList.Any())
                throw new PerformanceException("加载失败，绩效记录不存在！");

            var allot = allotList.FirstOrDefault(w => w.ID == allotId);
            if (allot == null)
                throw new PerformanceException("加载失败，绩效记录不存在！");

            var index = allotList.IndexOf(allot);
            if (index == 0)
                throw new PerformanceException("加载失败，绩效记录不存在！");

            var prevAllot = allotList[index - 1];
            if (prevAllot == null) prevAllot.ID = -1;

            var persons = peremployeeRepository.GetEntities(t => t.HospitalId == hospitalId && t.AllotId == prevAllot.ID);
            if (persons == null || !persons.Any())
                throw new PerformanceException("加载失败，未查询到上月人员信息！");

            var i = peremployeeRepository.Execute($@"DELETE FROM per_employee WHERE HospitalId=@HospitalId and AllotId=@AllotId;", new { hospitalId, allotId });
            if (i <= 0)
                throw new PerformanceException("加载失败，人员信息重置失败！");

            int day = allot.Month >= 1 && allot.Month <= 12 ? DateTime.DaysInMonth(allot.Year, allot.Month) : 30;
            var data = persons.Select(t =>
            {
                var entity = new per_employee
                {
                    HospitalId = t.HospitalId,
                    AllotId = allotId,
                    AccountingUnit = t.AccountingUnit,
                    Department = t.Department,
                    DoctorName = t.DoctorName,
                    JobCategory = t.JobCategory,
                    Duty = t.Duty,
                    JobTitle = t.JobTitle,
                    UnitType = t.UnitType,
                    AttendanceDay = day,
                    Attendance = 1,
                    PermanentStaff = t.PermanentStaff,
                    EfficiencyNumber = t.EfficiencyNumber,
                    WorkTime = t.WorkTime,
                    BirthDate = t.BirthDate,
                    Age = t.Age,
                    ReservedRatio = t.ReservedRatio,
                    BankCard = t.BankCard,
                    Remark = t.Remark,
                    CreateTime = DateTime.Now,
                };
                if (!string.IsNullOrEmpty(t.PersonnelNumber) && !string.IsNullOrEmpty(t.JobNumber))
                {
                    entity.PersonnelNumber = t.PersonnelNumber;
                    entity.JobNumber = t.JobNumber;
                }
                else
                {
                    string number = !string.IsNullOrEmpty(t.PersonnelNumber) ? t.PersonnelNumber : t.JobNumber;
                    entity.PersonnelNumber = number;
                    entity.JobNumber = number;
                }
                return entity;

            }).ToList();
            SaveAllotPersons(data);

            allot.IsModifyConfig = 1;
            perallotRepository.Update(allot);
            return true;
        }

        /// <summary>
        /// 保存绩效查询角色的用户细信息信息
        /// </summary>
        public void SaveQueryRole(int hospitalId, int userId, List<Dictionary<string, string>> dicData)
        {
            var dict = new Dictionary<string, string>();
            var hospital = perforHospitalRepository.GetEntity(t => t.ID == hospitalId);
            if (hospital.IsOwnerQuery != 1) return;

            var usercollor = new UserCollectData
            {
                HospitalId = hospital.ID,
                CreateUser = userId,
                ColHeaders = UserService.Users.Values.ToArray()
            };

            var role = perforRoleRepository.GetEntity(t => t.RoleName == "绩效查询");
            if (role == null) return;
            var userRole = perforUserroleRepository.GetEntities(t => t.RoleID == role.ID)?.Select(t => t.UserID);
            var users = perforUserRepository.GetEntities(t => userRole.Contains(t.ID));

            var newUsers = new List<string[]>();
            var updateUsers = new List<sys_user>();
            foreach (var item in dicData)
            {
                var json = JsonHelper.Serialize(item);
                var data = JsonHelper.Deserialize<per_employee>(json);

                string pwd = item.ContainsKey("Password") ? item["Password"] : "";

                if (hospital.IsOwnerQuery == 1)
                {
                    var isNewRole = !string.IsNullOrEmpty(data.PersonnelNumber?.Trim()) && !string.IsNullOrEmpty(pwd);
                    var isRepeat = users.FirstOrDefault(t => t.Login == data.PersonnelNumber?.Trim());
                    if (isRepeat != null)
                    {
                        isRepeat.Password = PwdHelper.MD5AndSalt2(pwd);
                        updateUsers.Add(isRepeat);
                    }
                    if (isNewRole && isRepeat == null)
                    {
                        newUsers.Add(new[] { $"{data.DoctorName?.Trim()}", $"{data.PersonnelNumber?.Trim()}", $"{item["Password"]}", "", "", "绩效查询", $"{hospital.HosName}", $"{data.AccountingUnit?.Trim()}" });
                    }
                }
            }
            if (updateUsers.Count > 0)
            {
                perforUserRepository.UpdateRange(updateUsers.ToArray());
            }
            if (newUsers.Count > 0)
            {
                usercollor.Data = newUsers.ToArray();
                userService.SaveUserHandsFlat(usercollor);
            }
        }
    }
}
