﻿using Dapper;
using GraphQL.Types;
using Performance.Infrastructure;
using Performance.Repository;
using Performance.Services;
using System.Collections.Generic;
using System.Linq;

namespace Performance.Api
{
    public class PerformanceQuery : ObjectGraphType
    {
        public PerformanceQuery(GraphQLService service, ClaimService claimService)
        {
            Field<ListGraphType<ReportDataType>>("info",
                resolve: context =>
                {
                    return service.GetReportsInfo();
                }
            );

            Field<ReportDataType>("report",
                arguments: new QueryArguments
                (
                    new QueryArgument<IntGraphType>() { Name = QueryParams.reportId }
                ),
                resolve: context =>
                {
                    int reportId = context.Arguments.ContainsKey(QueryParams.reportId)
                    ? ConvertHelper.To<int>(context.Arguments[QueryParams.reportId])
                    : 0;
                    return service.GetReport(reportId);
                }
            );

            Field<ListGraphType<ChartDataType>>("chartdata",
                arguments: Arguments(
                    new QueryArgument<StringGraphType>() { Name = QueryParams.accountingUnit }
                ),
                resolve: context =>
                {
                    int reportId = context.Arguments.ContainsKey(QueryParams.reportId)
                    ? ConvertHelper.To<int>(context.Arguments[QueryParams.reportId])
                    : 0;
                    var parameters = GetDynamicParameters(context.Arguments, QueryParams.hospitalId, QueryParams.year, QueryParams.month, QueryParams.accountingUnit);
                    return service.GetChartData(reportId, parameters);
                }
            );

            Field<ListGraphType<ReportPerformanceType>>("performances",
                arguments: Arguments(
                    new QueryArgument<StringGraphType>() { Name = QueryParams.accountingUnit },
                    new QueryArgument<StringGraphType>() { Name = QueryParams.category },
                    new QueryArgument<StringGraphType>() { Name = QueryParams.itemName }
                ),
                resolve: context =>
                {
                    int reportId = context.Arguments.ContainsKey(QueryParams.reportId)
                        ? ConvertHelper.To<int>(context.Arguments[QueryParams.reportId])
                        : 0;

                    var parameters = GetDynamicParameters(context.Arguments, QueryParams.hospitalId, QueryParams.year, QueryParams.month, QueryParams.accountingUnit, QueryParams.category, QueryParams.itemName);
                    var userId = claimService.GetUserId();
                    return service.GetReportPerformance(reportId, parameters, userId);
                }
            );
        }

        public static QueryArguments Arguments(params QueryArgument[] args)
        {
            var basic = new QueryArguments
            {
                new QueryArgument<IntGraphType>() { Name = QueryParams.reportId },
                new QueryArgument<IntGraphType>() { Name = QueryParams.hospitalId },
                new QueryArgument<IntGraphType>() { Name = QueryParams.year },
                new QueryArgument<IntGraphType>() { Name = QueryParams.month }
            };

            if (args != null && args.Any())
            {
                foreach (var item in args)
                {
                    basic.Add(item);
                }
            }

            return basic;
        }

        public DynamicParameters GetDynamicParameters(Dictionary<string, object> arguments, params string[] fields)
        {
            DynamicParameters parameters = new DynamicParameters();
            if (arguments == null || !arguments.Any()) return parameters;
            if (fields != null && fields.Any())
            {
                foreach (var item in fields)
                {
                    if (arguments.ContainsKey(item))
                    {
                        parameters.Add(item.ToLower(), arguments[item]);
                    }
                }
            }
            return parameters;
        }
    }
}
