﻿using Dapper;
using MySql.Data.MySqlClient;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Linq.Expressions;
using System.Text;
using System.Threading.Tasks;

namespace Performance.Infrastructure.Extensions
{
    public static class DapperExtensions
    {
        /// <summary>
        /// 批量新增
        /// </summary>
        /// <typeparam name="TEntity"></typeparam>
        /// <param name="connectionString"></param>
        /// <param name="entities"></param>
        /// <param name="tableName"></param>
        /// <param name="transaction"></param>
        /// <param name="commandTimeout"></param>
        /// <param name="relevantColumns"></param>
        /// <param name="ignoreColumns"></param>
        /// <param name="batchCount"></param>
        /// <returns></returns>
        public static int BulkInsert<TEntity>(string connectionString,
                                              List<TEntity> entities,
                                              string? tableName = null,
                                              IDbTransaction? transaction = null,
                                              int? commandTimeout = null,
                                              List<string>? relevantColumns = null,
                                              List<string>? ignoreColumns = null,
                                              int batchCount = 1000)
        {
            var type = typeof(TEntity);
            var properties = type.GetProperties().AsEnumerable();

            if (relevantColumns?.Any() == true)
                properties = properties.Where(w => relevantColumns.Contains(w.Name));

            if (ignoreColumns?.Any() == true)
                properties = properties.Where(w => !ignoreColumns.Contains(w.Name));

            var columnNames = properties.Select(w => w.Name).ToList();
            for (int i = 0; i < Math.Ceiling((double)entities.Count / batchCount); i++)
            {
                var startIndex = batchCount * i;
                var currEntities = entities.Skip(startIndex).Take(batchCount);

                var strsql = new StringBuilder();
                strsql.Append($"INSERT INTO {tableName}({string.Join(',', columnNames)}) VALUES");

                DynamicParameters parameters = new DynamicParameters();

                foreach (var item in currEntities.Select((row, index) => (row, index)))
                {
                    var vNames = new List<string>();

                    foreach (var name in columnNames)
                    {
                        var pName = $"@{name}_{item.index}";
                        if (!vNames.Contains(pName)) vNames.Add(pName);

                        var value = GetValue<TEntity, object>(item.row, name);
                        if (value != null && value.GetType().IsClass)
                            value = value.ToString();

                        parameters.Add(pName, value);
                    }
                    tableName ??= typeof(TEntity).Name;
                    strsql.Append($"({string.Join(',', vNames)}),");
                }
                strsql.Remove(strsql.Length - 1, 1);
                strsql.Append(';');
                var inserSql = strsql.ToString();
                using (var conn = new MySqlConnection(connectionString))
                {
                    if (conn.State != ConnectionState.Open) conn.Open();
                    conn.Execute(inserSql, parameters, transaction, commandTimeout);
                    conn.Close();
                }
            }
            return entities.Count;
        }

        /// <summary>
        /// 对象取值
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <typeparam name="TValue"></typeparam>
        /// <param name="item"></param>
        /// <param name="propertyName"></param>
        /// <returns></returns>
        public static TValue GetValue<T, TValue>(T item, string propertyName)
        {
            ParameterExpression parameter = Expression.Parameter(typeof(T), "w");
            MemberExpression member = Expression.Property(parameter, propertyName);
            UnaryExpression conversion = Expression.Convert(member, typeof(TValue));
            var lambda = Expression.Lambda<Func<T, TValue>>(conversion, parameter);
            Func<T, TValue> getter = lambda.Compile();

            return getter(item);
        }
    }
}
