﻿using AutoMapper;
using Dapper;
using Masuit.Tools.Systems;
using Microsoft.EntityFrameworkCore.Metadata.Internal;
using Microsoft.Extensions.Configuration;
using MySql.Data.MySqlClient;
using NPOI.Util;
using OfficeOpenXml;
using Performance.DtoModels;
using Performance.EntityModels;
using Performance.Infrastructure;
using Performance.Infrastructure.Extensions;
using System;
using System.Collections.Generic;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Threading.Tasks;

namespace Performance.Services.AllotCompute
{
    public class EPImportDataService : IAutoInjection
    {
        private readonly IConfiguration _configuration;
        private readonly IMapper _mapper;
        private readonly PerSheetService _perSheetService;
        private readonly LogManageService _logManageService;

        public EPImportDataService(IConfiguration configuration, IMapper mapper, PerSheetService perSheetService, LogManageService logManageService)
        {
            _configuration = configuration;
            _mapper = mapper;
            _perSheetService = perSheetService;
            _logManageService = logManageService;
        }
        public void DeleteSheetData(per_allot allot)
        {
            var connectionString = _configuration.GetValue("AppConnection:PerformanceConnectionString", "");
            using var conn = new MySqlConnection(connectionString);
            if (conn.State != ConnectionState.Open) conn.Open();
            conn.Execute("DELETE FROM im_orig_data WHERE AllotId = @AllotId", new { AllotId = allot.ID });
        }

        public List<ImportData> ReadSheetData(per_allot allot)
        {
            var maximumColIndex = _configuration.GetValue("AllotFileReadOption:MaximumColIndex", 256);
            var maxEmptyRowIndex = _configuration.GetValue("AllotFileReadOption:MaxEmptyRowIndex", 30);
            var maxEmptyColIndex = _configuration.GetValue("AllotFileReadOption:MaxEmptyColIndex", 10);
            var skipUnidentifiable = _configuration.GetValue("AllotFileReadOption:SkipUnidentifiable", true);

            var version = SnowFlake.GetInstance().GetLongId();
            List<ImportData> importDatas = new List<ImportData>();
            _logManageService.WriteMsg("读取文件", $"文件读取中...当前操作需要一点时间，请耐心等待", 1, allot.ID, "ReceiveMessage", true);
            try
            {
                using ExcelPackage package = new ExcelPackage(new FileInfo(allot.Path));
                foreach (var sheet in package.Workbook.Worksheets.Where(sheet => sheet.Dimension != null))
                {
                    if (sheet.AutoFilterAddress != null)
                        sheet.Cells[sheet.AutoFilterAddress.Address].AutoFilter = false;
                    var sheetType = (int)_perSheetService.GetSheetType(sheet.Name);

                    if (skipUnidentifiable && sheetType == (int)SheetType.Unidentifiable) continue;

                    _logManageService.WriteMsg("准备读取文件", $"正在准备读取“{sheet.Name}”", 1, allot.ID, "ReceiveMessage", true);

                    for (int rowIndex = sheet.Dimension.Start.Row; rowIndex <= sheet.Dimension.End.Row; rowIndex++)
                    {
                        // 判断当前行与最后一条有效数据的行号 连续空行 中断读取
                        if (rowIndex > maxEmptyRowIndex && rowIndex - importDatas.Max(w => w.RowIndex) > maxEmptyRowIndex)
                        {
                            break;
                        }
                        var maxColIndex = rowIndex > maxEmptyRowIndex ? importDatas.Max(w => w.ColIndex) + maxEmptyColIndex : maximumColIndex;
                        var colCount = Math.Min(sheet.Dimension.End.Column, maxColIndex);
                        for (int colIndex = sheet.Dimension.Start.Column; colIndex <= colCount; colIndex++)
                        {
                            var cell = sheet.Cells[rowIndex, colIndex];

                            var endRowIndex = rowIndex;
                            var endColIndex = colIndex;

                            if (cell.Merge)
                            {
                                var mergedCellRang = sheet.MergedCells[rowIndex, colIndex];
                                var endCellName = mergedCellRang.Split(":").Last();
                                var endCell = sheet.Cells[endCellName];

                                endRowIndex = endCell.End.Row;
                                endColIndex = endCell.End.Column;
                            }
                            if (!string.IsNullOrEmpty(cell?.Text))
                            {
                                var cellData = new ImportData
                                {
                                    AllotId = allot.ID,
                                    SheetName = sheet.Name,
                                    SheetIndex = sheet.Index,
                                    SheetType = sheetType,
                                    RowIndex = rowIndex,
                                    ColIndex = colIndex,
                                    Address = cell.Address,
                                    Text = cell.Text?.Clean() ?? "",
                                    Value = cell.Value?.ToString()?.Clean() ?? "",
                                    //Comment = cell.Comment?.Text ?? "",
                                    EndRowIndex = endRowIndex,
                                    EndColIndex = endColIndex,
                                    Sign = SnowFlake.GetInstance().GetUniqueShortId(),
                                    ColName = ExcelCellBase.GetAddressCol(colIndex),
                                    Version = version,
                                    //Fulladdress = cell.FullAddress,
                                    //Formula = cell.Formula,
                                    //Format = cell.Style.Numberformat.Format,
                                    //NumFmtId = cell.Style.Numberformat.NumFmtID,
                                };
                                importDatas.Add(cellData);
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _logManageService.WriteMsg("读取文件失败", ex.Message.ToString(), 4, allot.ID, "ReceiveMessage", true);
                throw;
            }
            _logManageService.WriteMsg("读取文件", $"EXCEL文件基础数据读取完成!", 1, allot.ID, "ReceiveMessage", true);

            _logManageService.WriteMsg("保存基础数据", $"基础数据保存中...当前操作需要一点时间，请耐心等待", 1, allot.ID, "ReceiveMessage", true);

            try
            {
                var connectionString = _configuration.GetValue("AppConnection:PerformanceConnectionString", "");
                var tasks = new List<Task>();
                foreach (var sheetName in importDatas.Select(w => w.SheetName).Distinct())
                {
                    var task = Task.Factory.StartNew(() =>
                    {
                        var items = importDatas.Where(w => w.SheetName == sheetName).ToList();
                        DapperExtensions.BulkInsert(connectionString, items, tableName: "im_orig_data", commandTimeout: 60 * 60 * 5);
                    });
                    tasks.Add(task);
                }
                Task.WaitAll(tasks.ToArray());
                _logManageService.WriteMsg("保存基础数据", $"基础数据保存完成!", 1, allot.ID, "ReceiveMessage", true);
            }
            catch (Exception ex)
            {
                _logManageService.WriteMsg("保存数据失败", ex.Message.ToString(), 4, allot.ID, "ReceiveMessage", true);
                throw;
            }

            return importDatas;
        }
        public PerExcel GetImportExcel(per_allot allot)
        {
            PerExcel excel = new PerExcel
            {
                Path = allot.Path,
                FileName = Path.GetFileName(allot.Path),
                PerSheet = new List<PerSheet>()
            };
            var (accountbasic, specialunit, data, employee, employee_clinic, employee_logistics, header, sheets) = GetImportData(allot);

            foreach (var sheet in sheets)
            {
                if (sheet.SheetType == (int)SheetType.Employee)
                {
                    var items = header.Where(w => w.SheetID == sheet.ID);
                    var rows = _mapper.Map<List<PerDataEmployee>>(employee);
                    PerSheet perSheet = new PerSheet
                    {
                        SheetName = sheet.SheetName,
                        SheetType = (SheetType)sheet.SheetType,
                        ModuleName = EnumHelper.GetDescription((SheetType)sheet.SheetType),
                        PerHeader = _mapper.Map<List<PerHeader>>(items),
                        PerData = rows.Select(w => (IPerData)w).ToList(),
                    };
                    excel.PerSheet.Add(perSheet);
                }
                else if (sheet.SheetType == (int)SheetType.LogisticsEmployee)
                {
                    var items = header.Where(w => w.SheetID == sheet.ID);
                    var rows = _mapper.Map<List<PerDataLogisticsEmployee>>(employee_logistics);
                    PerSheet perSheet = new PerSheet
                    {
                        SheetName = sheet.SheetName,
                        SheetType = (SheetType)sheet.SheetType,
                        ModuleName = EnumHelper.GetDescription((SheetType)sheet.SheetType),
                        PerHeader = _mapper.Map<List<PerHeader>>(items),
                        PerData = rows.Select(w => (IPerData)w).ToList(),
                    };
                    excel.PerSheet.Add(perSheet);
                }
                else if (sheet.SheetType == (int)SheetType.ClinicEmployee)
                {
                    var items = header.Where(w => w.SheetID == sheet.ID);
                    var rows = _mapper.Map<List<PerDataClinicEmployee>>(employee_clinic);
                    PerSheet perSheet = new PerSheet
                    {
                        SheetName = sheet.SheetName,
                        SheetType = (SheetType)sheet.SheetType,
                        ModuleName = EnumHelper.GetDescription((SheetType)sheet.SheetType),
                        PerHeader = _mapper.Map<List<PerHeader>>(items),
                        PerData = rows.Select(w => (IPerData)w).ToList(),
                    };
                    excel.PerSheet.Add(perSheet);
                }
                else if (sheet.SheetType == (int)SheetType.AccountBasic)
                {
                    var items = header.Where(w => w.SheetID == sheet.ID);
                    var rows = accountbasic.Select(item =>
                        new PerDataAccountBaisc
                        {
                            UnitType = Enum.IsDefined(typeof(UnitType), item.UnitType) ? ((UnitType)item.UnitType).ToString() : "",
                            AccountingUnit = item.DoctorAccountingUnit,
                            Drg = item.Drg,
                            PermanentStaff = item.PermanentStaff,
                            Number = item.DoctorNumber ?? 0,
                            BasicFactor = item.DoctorBasicFactor ?? 0,
                            Extra = item.DoctorExtra,
                            AssessBeforeOtherFee = item.AssessBeforeOtherFee,
                            MedicineExtra = item.MedicineExtra,
                            MaterialsExtra = item.MaterialsExtra,
                            ScoringAverage = item.DoctorScoringAverage,
                            AssessLaterOtherFee = item.AssessLaterOtherFee,
                            AdjustFactor = item.DoctorAdjustFactor,
                            AdjustLaterOtherFee = item.AdjustLaterOtherFee,
                            RealGiveFee = item.RealGiveFee,
                        });


                    PerSheet perSheet = new PerSheet
                    {
                        SheetName = sheet.SheetName,
                        SheetType = (SheetType)sheet.SheetType,
                        ModuleName = EnumHelper.GetDescription((SheetType)sheet.SheetType),
                        PerHeader = _mapper.Map<List<PerHeader>>(items),
                        PerData = rows.Select(w => (IPerData)w).ToList(),
                    };
                    excel.PerSheet.Add(perSheet);
                }
                else if (sheet.SheetType == (int)SheetType.SpecialUnit)
                {
                    var items = header.Where(w => w.SheetID == sheet.ID);
                    var rows = _mapper.Map<List<PerDataSpecialUnit>>(specialunit);
                    PerSheet perSheet = new PerSheet
                    {
                        SheetName = sheet.SheetName,
                        SheetType = (SheetType)sheet.SheetType,
                        ModuleName = EnumHelper.GetDescription((SheetType)sheet.SheetType),
                        PerHeader = _mapper.Map<List<PerHeader>>(items),
                        PerData = rows.Select(w => (IPerData)w).ToList(),
                    };
                    excel.PerSheet.Add(perSheet);
                }
                else if (sheet.SheetType != (int)SheetType.Unidentifiable)
                {
                    var items = header.Where(w => w.SheetID == sheet.ID);
                    var list = data.Where(w => w.SheetID == sheet.ID);
                    var rows = _mapper.Map<List<PerData>>(list);
                    PerSheet perSheet = new PerSheet
                    {
                        SheetName = sheet.SheetName,
                        SheetType = (SheetType)sheet.SheetType,
                        ModuleName = EnumHelper.GetDescription((SheetType)sheet.SheetType),
                        PerHeader = _mapper.Map<List<PerHeader>>(items),
                        PerData = rows.Select(w => (IPerData)w).ToList(),
                    };
                    excel.PerSheet.Add(perSheet);
                }
            }
            return excel;
        }


        private (IEnumerable<im_accountbasic> accountbasic, IEnumerable<im_specialunit> specialunit, IEnumerable<im_data> data, IEnumerable<im_employee> employee, IEnumerable<im_employee_clinic> employee_clinic, IEnumerable<im_employee_logistics> employee_logistics, IEnumerable<im_header> header, IEnumerable<per_sheet> sheet) GetImportData(per_allot allot)
        {
            var connectionString = _configuration.GetValue("AppConnection:PerformanceConnectionString", "");
            using var conn = new MySqlConnection(connectionString);
            if (conn.State != ConnectionState.Open) conn.Open();

            var sql = @"
SELECT * FROM im_accountbasic WHERE AllotId = @AllotId;
SELECT * FROM im_specialunit WHERE AllotId = @AllotId;
SELECT * FROM im_data WHERE AllotId = @AllotId;
SELECT * FROM im_employee WHERE AllotId = @AllotId;
SELECT * FROM im_employee_clinic WHERE AllotId = @AllotId;
SELECT * FROM im_employee_logistics WHERE AllotId = @AllotId;
SELECT * FROM im_header WHERE AllotId = @AllotId;
SELECT * FROM per_sheet WHERE AllotId = @AllotId;
";
            var multipleResults = conn.QueryMultiple(sql, new { AllotId = allot.ID });


            var accountbasic = multipleResults.Read<im_accountbasic>();
            var specialunit = multipleResults.Read<im_specialunit>();
            var data = multipleResults.Read<im_data>();
            var employee = multipleResults.Read<im_employee>();
            var employee_clinic = multipleResults.Read<im_employee_clinic>();
            var employee_logistics = multipleResults.Read<im_employee_logistics>();
            var header = multipleResults.Read<im_header>();
            var sheet = multipleResults.Read<per_sheet>();

            return (accountbasic, specialunit, data, employee, employee_clinic, employee_logistics, header, sheet);
        }


        public void SheetDataImport(per_allot allot)
        {
            var connectionString = _configuration.GetValue("AppConnection:PerformanceConnectionString", "");
            using var conn = new MySqlConnection(connectionString);
            if (conn.State != ConnectionState.Open) conn.Open();
            conn.Execute("call proc_allot_import(@AllotId);", new { AllotId = allot.ID }, commandTimeout: 60 * 60 * 5);
        }
    }
    public class ImportData
    {
        public long Id { get; set; }

        /// <summary>
        /// 绩效分配月ID
        /// </summary>
        public long AllotId { get; set; }

        /// <summary>
        /// 工作表ID
        /// </summary>
        public long SheetId { get; set; }

        /// <summary>
        /// 工作表名称
        /// </summary>
        public string SheetName { get; set; }

        /// <summary>
        /// 工作表索引
        /// </summary>
        public int SheetIndex { get; set; }

        /// <summary>
        /// 行号
        /// </summary>
        public int RowIndex { get; set; }

        /// <summary>
        /// 列号
        /// </summary>
        public int ColIndex { get; set; }

        /// <summary>
        /// 合并结束行号
        /// </summary>
        public int EndRowIndex { get; set; }

        /// <summary>
        /// 合并结束列号
        /// </summary>
        public int EndColIndex { get; set; }

        /// <summary>
        /// 文本值
        /// </summary>
        public string Value { get; set; }

        /// <summary>
        /// 单元格名称
        /// </summary>
        public string Address { get; set; }

        /// <summary>
        /// 文本值
        /// </summary>
        public string Text { get; set; }

        /// <summary>
        /// 唯一签名
        /// </summary>
        public string Sign { get; set; }

        /// <summary>
        /// 列名
        /// </summary>
        public string ColName { get; set; }

        ///// <summary>
        ///// 单元格备注
        ///// </summary>
        //public string Comment { get; set; }
        public int SheetType { get; set; }
        public long Version { get; set; }

        /// <summary>
        /// 数据格式
        /// </summary>
        //public string Format { get; set; }

        ///// <summary>
        ///// 公式
        ///// </summary>
        //public string Formula { get; set; }

        ///// <summary>
        ///// 格式ID(第三方)
        ///// </summary>
        //public int? NumFmtId { get; set; }

        ///// <summary>
        ///// 单元格地址
        ///// </summary>
        //public string Fulladdress { get; set; }

        ///// <summary>
        ///// 创建人
        ///// </summary>
        //public long CreateBy { get; set; }

        ///// <summary>
        ///// 创建时间
        ///// </summary>
        //public DateTime CreateTime { get; set; }

        ///// <summary>
        ///// 最后修改人
        ///// </summary>
        //public long UpdateBy { get; set; }

        ///// <summary>
        ///// 修改时间
        ///// </summary>
        //public DateTime UpdateTime { get; set; }
    }
}
